% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rsample-time_series_cv.R
\name{time_series_cv}
\alias{time_series_cv}
\title{Time Series Cross Validation}
\usage{
time_series_cv(
  data,
  date_var = NULL,
  initial = 5,
  assess = 1,
  skip = 1,
  lag = 0,
  cumulative = FALSE,
  slice_limit = n(),
  point_forecast = FALSE,
  ...
)
}
\arguments{
\item{data}{A data frame.}

\item{date_var}{A date or date-time variable.}

\item{initial}{The number of samples used for analysis/modeling in the
initial resample.}

\item{assess}{The number of samples used for each assessment resample.}

\item{skip}{A integer indicating how many (if any) \emph{additional} resamples
to skip to thin the total amount of data points in the analysis resample.
See the example below.}

\item{lag}{A value to include an lag between the assessment
and analysis set. This is useful if lagged predictors will be used
during training and testing.}

\item{cumulative}{A logical. Should the analysis resample grow beyond the
size specified by \code{initial} at each resample?.}

\item{slice_limit}{The number of slices to return. Set to \code{dplyr::n()},
which returns the maximum number of slices.}

\item{point_forecast}{Whether or not to have the testing set be a single point forecast or to be a forecast horizon.
The default is to be a forecast horizon. Default: \code{FALSE}}

\item{...}{These dots are for future extensions and must be empty.}
}
\value{
An tibble with classes \code{time_series_cv}, \code{rset}, \code{tbl_df}, \code{tbl},
and \code{data.frame}. The results include a column for the data split objects
and a column called \code{id} that has a character string with the resample
identifier.
}
\description{
Create \code{rsample} cross validation sets for time series.
This function produces a sampling plan starting with the most recent
time series observations, rolling backwards. The sampling procedure
is similar to \code{rsample::rolling_origin()}, but places the focus
of the cross validation on the most recent time series data.
}
\details{
\strong{Time-Based Specification}

The \code{initial}, \code{assess}, \code{skip}, and \code{lag} variables can be specified as:
\itemize{
\item Numeric: \code{initial = 24}
\item Time-Based Phrases: \code{initial = "2 years"}, if the \code{data} contains
a \code{date_var} (date or datetime)
}

\strong{Initial (Training Set) and Assess (Testing Set)}

The main options, \code{initial} and \code{assess}, control the number of
data points from the original data that are in the analysis (training set)
and the assessment (testing set), respectively.

\strong{Skip}

\code{skip} enables the function to not use every data point in the resamples.
When \code{skip = 1}, the resampling data sets will increment by one position.

Example: Suppose that the rows of a data set are consecutive days. Using \code{skip = 7}
will make the analysis data set operate on \emph{weeks} instead of days. The
assessment set size is not affected by this option.

\strong{Lag}

The Lag parameter creates an overlap between the Testing set. This is needed
when lagged predictors are used.

\strong{Cumulative vs Sliding Window}

When \code{cumulative = TRUE}, the \code{initial} parameter is ignored and the
analysis (training) set will grow as
resampling continues while the assessment (testing) set size will always remain
static.

When \code{cumulative = FALSE}, the \code{initial} parameter fixes the analysis (training)
set and resampling occurs over a fixed window.

\strong{Slice Limit}

This controls the number of slices. If \code{slice_limit = 5}, only the most recent
5 slices will be returned.

\strong{Point Forecast}

A point forecast is sometimes desired such as if you want to forecast a value
"4-weeks" into the future. You can do this by setting the following parameters:
\itemize{
\item assess = "4 weeks"
\item point_forecast = TRUE
}

\strong{Panel Data / Time Series Groups / Overlapping Timestamps}

Overlapping timestamps occur when your data has more than one
time series group. This is sometimes called \emph{Panel Data} or \emph{Time Series Groups}.

When timestamps are duplicated (as in the case of "Panel Data" or "Time Series Groups"),
the resample calculation applies a sliding window of
fixed length to the dataset. See the example using \code{walmart_sales_weekly}
dataset below.
}
\examples{
library(dplyr)

# DATA ----
m750 <- m4_monthly \%>\% dplyr::filter(id == "M750")


# RESAMPLE SPEC ----
resample_spec <- time_series_cv(data = m750,
                                initial     = "6 years",
                                assess      = "24 months",
                                skip        = "24 months",
                                cumulative  = FALSE,
                                slice_limit = 3)

resample_spec

# VISUALIZE CV PLAN ----

# Select date and value columns from the tscv diagnostic tool
resample_spec \%>\% tk_time_series_cv_plan()

# Plot the date and value columns to see the CV Plan
resample_spec \%>\%
    plot_time_series_cv_plan(date, value, .interactive = FALSE)


# PANEL DATA / TIME SERIES GROUPS ----
# - Time Series Groups are processed using an *ungrouped* data set
# - The data has sliding windows applied starting with the beginning of the series
# - The seven groups of weekly time series are
#   processed together for <split [358/78]> dimensions

walmart_tscv <- walmart_sales_weekly \%>\%
    time_series_cv(
        date_var    = Date,
        initial     = "12 months",
        assess      = "3 months",
        skip        = "3 months",
        slice_limit = 4
    )

walmart_tscv

walmart_tscv \%>\%
    plot_time_series_cv_plan(Date, Weekly_Sales, .interactive = FALSE)

}
\seealso{
\itemize{
\item \code{\link[=time_series_cv]{time_series_cv()}} and \code{\link[rsample:rolling_origin]{rsample::rolling_origin()}} - Functions used to create
time series resample specifications.
\item \code{\link[=plot_time_series_cv_plan]{plot_time_series_cv_plan()}} - The plotting function used for visualizing the
time series resample plan.
\item \code{\link[=time_series_split]{time_series_split()}} - A convenience function to return a single time series
split containing a training/testing sample.
}
}
