% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/BAS.R
\name{BAS}
\alias{BAS}
\title{Balanced Acceptance Sampling (BAS).}
\usage{
BAS(
  shapefile = NULL,
  n = 100,
  boundingbox = NULL,
  minRadius = NULL,
  panels = NULL,
  panel_overlap = NULL,
  stratum = NULL,
  seeds = NULL,
  verbose = FALSE
)
}
\arguments{
\item{shapefile}{Shape file as a polygon (sp or sf) to select sites for.}

\item{n}{Number of sites to select. If using stratification it is a named vector containing
sample sizes of each group.}

\item{boundingbox}{Bounding box around the study area. If a bounding box is not supplied
then spbal will generate a bounding box for the shapefile.}

\item{minRadius}{If specified, the minimum distance, in meters, allowed between sample
points. This is applied to the $sample points. Points that meet the minRadius criteria
are retuned in the minRadius output variable.}

\item{panels}{A list of integers that define the size of each panel in a
non-overlapping panels design. The length of the list determines the number of
panels required. The sum of the integers in the panels parameter will determine
the total number of samples selected, n. The default value for panels is NULL,
this indicates that a non-overlapping panel design is not wanted.}

\item{panel_overlap}{A list of integers that define the overlap into the previous
panel. Is only used when the panels parameter is not NULL. The default value for
panel_overlap is NULL. The length of panel_overlap must be equal to the length
of panels. The first value is always forced to zero as the first panel never
overlaps any region.}

\item{stratum}{The name of a column in the data.frame attached to shapefile that defines
the strata of interest.}

\item{seeds}{A vector of 2 seeds, u1 and u2. If not specified, the default is NULL and will
be defined randomly using function \code{generateUVector}.}

\item{verbose}{Boolean if you want to see any output printed to screen. Helpful if taking a
long time. Default is FALSE i.e. no informational messages are displayed.}
}
\value{
A list containing three variables, \code{$sample} containing locations in the BAS sample,
in BAS order, \code{$seeds}, the u1 and u2 seeds used to generate the sample and \code{$minRadius}
containing points from $sample that meet the minRadius criteria. If the minRadius
parameter is NULL then the $minRadius returned will also be NULL.

The sample points are returned in the form of a simple feature collection of POINT objects.
They have the following attributes:
\itemize{
\item \code{SiteID} A unique identifier for every sample point. This
encodes the BAS order.
\item \code{spbalSeqID} A unique identifier for every sample point. This
encodes the BAS sample order.
\item \code{geometry} The XY co-ordinates of the sample point in the CRS of the original
shapefile.
}
}
\description{
BAS draws spatially balanced samples from areal resources. To draw BAS samples,
spbal requires a study region shapefile and the region’s bounding box. An initial sample size
is also needed, which can be easily increased or decreased within spbal for master sampling
applications
}
\examples{
# Equal probability BAS sample ----------------------------------------------

# Use the North Carolina shapefile supplied in the sf R package.
shp_file <- sf::st_read(system.file("shape/nc.shp", package="sf"))
shp_gates <- shp_file[shp_file$NAME == "Gates",]

# Vertically aligned master sample bounding box.
bb <- spbal::BoundingBox(shapefile = shp_gates)

set.seed(511)
n_samples <- 20
# Equal probability BAS sample.
result <- spbal::BAS(shapefile = shp_gates,
                     n = n_samples,
                     boundingbox = bb)
BAS20 <- result$sample
# display first three sample points.
BAS20[1:3,]

# Increase the BAS sample size ----------------------------------------------
n_samples <- 50
result2 <- spbal::BAS(shapefile = shp_gates,
                      n = n_samples,
                      boundingbox = bb,
                      seeds = result$seed)
BAS50 <- result2$sample
BAS50[1:3,]

# Check, first n_samples points in both samples must be the same.
all.equal(BAS20$geometry, BAS50$geometry[1:20])

}
\author{
This function was first written by Paul van Dam-Bates for the
package BASMasterSample and later simplified by Phil Davies.
}
