% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sfcr_baseline.R
\name{sfcr_baseline}
\alias{sfcr_baseline}
\title{Simulate the baseline scenario of a stock-flow consistent model}
\usage{
sfcr_baseline(
  equations,
  external,
  periods,
  initial = NULL,
  hidden = NULL,
  max_iter = 350,
  .hidden_tol = 0.1,
  tol = 1e-08,
  method = "Broyden",
  rhtol = FALSE,
  ...
)
}
\arguments{
\item{equations}{A \code{sfcr_set} containing all the equations of the model to be simulated. The equations
must be written with the R formula syntax, with the left-hand side separated from the right-hand side
by a twiddle \code{~}.}

\item{external, initial}{A \code{sfcr_set} of external variables (exogenous and parameters) or of initial
values. They should be written as equations using the R syntax.}

\item{periods}{A number specifying the total number of periods of the model to be simulated.}

\item{hidden}{Named object that identify the two variables that make the hidden equality
in the SFC model, e.g., \code{c("H_h" = "H_s")}. Defaults to NULL.
If \code{hidden} is supplied, the model will evaluate if the hidden equation is satisfied.}

\item{max_iter}{Maximum iterations allowed per period.}

\item{.hidden_tol}{Error tolerance to accept the equality of the hidden equation. Defaults to 1.
In growth models, computational errors might buildup in the hidden equation, which renders any absolute
comparison inadequate. For such models, please turn \code{rhtol} to \code{TRUE}, and set the value
of \code{.hidden_tol} accordingly. See details for further information.}

\item{tol}{Tolerance accepted to determine convergence.}

\item{method}{The method to use to find a solution. Defaults to "Broyden".}

\item{rhtol}{A logical argument that defines whether the a relative measure is used to evaluate
the hidden equation or not. Defaults to \code{FALSE}, i.e., a absolute measure is used.}

\item{...}{Extra arguments to pass to \code{rootSolve::multiroot()} function if "Newton" method
is selected.}
}
\value{
A \code{sfcr_tbl}.
}
\description{
The \code{sfcr_baseline()} function is used to simulate a SFC model.
}
\details{
The output of a \code{sfcr_baseline()} is a \code{sfcr_tbl}. The only difference between
a \code{sfcr_tbl} and a standard \code{tbl_df} is that the former has two extra attributes:
\code{matrix} and \code{call}. The \code{matrix} attribute, for example, can be accessed by
calling \code{attributes(sfcr_sim_object)$matrix}.
It is possible to see, in the matrix, the number of iterations required to calculate each
block of equations in the model.
The \code{call} attribute shows the blocks of equations and preserve the call that are used
internally.

The \code{equations}, \code{exogenous}, and \code{parameters} arguments must be written
with the R formula syntax, i.e., the left-hand side of each item is separated to the
right-hand side by a twiddle. Variables that represent lags of endogenous or exogenous
variables must be followed by \code{[-1]}. See examples for details on the syntax.

Before solving the system of equations, two consecutive depth-first searches identify
and order the blocks of independent equations in the system. The system is then solved
sequentially, i.e., the variables that depend only on lagged or exogenous values are evaluated
first, and then the variables that depends on these variables, etc. The solving algorithms
are only applied to the blocks of mutually dependent equations. The great \code{igraph}
package is used to implement the two consecutive depth-first searches.
\itemize{
\item Methods:
}

The \code{sfcr} package provides three algorithms to solve the blocks of cyclical equations:
the Gauss-Seidel algorithm, the Broyden algorithm, and the Newton-Raphson algorithm. The
default method is "Broyden" as it tends to be fastest one.

See \insertCite{kinsella2011gauss}{sfcr} for details on the Gauss-Seidel algorithm and
\insertCite{peressini1988nonlinear}{sfcr} for details on the Broyden and Newton-Raphson
algorithms.

The "Broyden" algorithm uses the \code{rootSolve::jacobian.full()} function to get the
initial Jacobian matrix, and compiled code from \code{RcppArmadillo} to invert the
jacobians. See also https://www.math.usm.edu/lambers/mat419/lecture11.pdf.

The Gauss Seidel algorithm is implemented as described by \insertCite{kinsella2011gauss}{sfcr}.
Finally, the "Newton" method uses the \code{rootSolve::multiroot()} function to solve the system.
\itemize{
\item Hidden equation:
}

One of the defining aspects of a SFC model is its water tight accounting. One way
to check whether the model was correctly defined is to see if the hidden (redundant)
equation is satisfied after the model is simulated. In stationary models, an absolute
comparison should suffice as the model converges to a stationary state. However,
growth models converge to a stable growth rate where stocks are perpetually increasing.
It is inadequate to use a absolute comparison in such models. In these cases, the
\code{rhtol} argument ("relative hidden tolerance") must be set to \code{TRUE} in order
to perform a relative comparison. The relative comparison evaluates the numerical
discrepancy in the hidden equation as a ratio of one of its elements. For example,
if \code{hidden = c("Bbs" = "Bbd")}, the hidden equation will be evaluated according to
the following steps:
\enumerate{
\item \code{d = (Bbs - Bbd)}
\item \code{isTRUE(d/Bbs < .hidden_tol)}
}

In general, the \code{.hidden_tol} argument should be set to a small number (e.g. 1e-6).
The function will check that this proportion remains the same for all simulated periods.
}
\examples{
eqs <- sfcr_set(
  TXs ~ TXd,
  YD ~ W * Ns - TXs,
  Cd ~ alpha1 * YD + alpha2 * Hh[-1],
  Hh ~ YD - Cd + Hh[-1],
  Ns ~ Nd,
  Nd ~ Y / W,
  Cs ~ Cd,
  Gs ~ Gd,
  Y ~ Cs + Gs,
  TXd ~ theta * W * Ns,
  Hs ~ Gd - TXd + Hs[-1]
)

external <- sfcr_set(Gd ~ 20, W ~ 1, alpha1 ~ 0.6, alpha2 ~ 0.4, theta ~ 0.2)

# Periods is set to 10 to run faster. A usual model should run at
# least 50 periods to find a steady state
sfcr_baseline(equations = eqs, external = external, periods = 10)

}
\references{
\insertRef{kinsella2011gauss}{sfcr}
\insertRef{peressini1988nonlinear}{sfcr}
}
\author{
João Macalós, \email{joaomacalos@gmail.com}
}
