% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cf_lgcm.R
\name{semPower.powerLGCM}
\alias{semPower.powerLGCM}
\title{semPower.powerLGCM}
\usage{
semPower.powerLGCM(
  type,
  comparison = "restricted",
  nWaves = NULL,
  means = NULL,
  variances = NULL,
  covariances = NULL,
  quadratic = FALSE,
  timeCodes = NULL,
  ticExogSlopes = NULL,
  ticEndogSlopes = NULL,
  groupEqual = NULL,
  nullEffect = NULL,
  nullWhichGroups = NULL,
  autocorResiduals = TRUE,
  ...
)
}
\arguments{
\item{type}{type of power analysis, one of \code{'a-priori'}, \code{'post-hoc'}, \code{'compromise'}.}

\item{comparison}{comparison model, one of \code{'saturated'} or \code{'restricted'} (the default). This determines the df for power analyses. \code{'saturated'} provides power to reject the model when compared to the saturated model, so the df equal the one of the hypothesized model. \code{'restricted'} provides power to reject the hypothesized model when compared to an otherwise identical model that just omits the restrictions defined in \code{nullEffect}, so the df equal the number of restrictions.}

\item{nWaves}{number of waves, must be >= 3 for linear and > 3 for quadratic trends.}

\item{means}{vector providing the means of the intercept and the linear slope factor (and the quadratic slope factor, if \code{quadratic = TRUE}). A list for multiple group models.}

\item{variances}{vector providing the variances of the intercept and the linear slope factor (and the quadratic slope factor, if \code{quadratic = TRUE}). Can be omitted, if a matrix is provided to \code{covariances}. Takes precedence over the diagonal in \code{covariances} when both are defined. A list for multiple group models.}

\item{covariances}{either the variance-covariance matrix between the intercept and the slope (and the quadratic slope factor, if \code{quadratic = TRUE}), or a single number giving the covariance between intercept and slope factor, or \code{NULL} for orthogonal factors. A list for multiple group models.}

\item{quadratic}{whether to include a quadratic slope factor in addition to a linear slope factor. Defaults to \code{FALSE} for no quadratic slope factor.}

\item{timeCodes}{vector of length \code{nWaves} defining the loadings on the slope factor. If omitted, the time codes default to (0, 1, ..., (nWaves - 1)).}

\item{ticExogSlopes}{vector defining the slopes for an exogenous time-invariant covariate in the prediction of the intercept and slope factors (and the quadratic slope factor, if \code{quadratic = TRUE}). Can be omitted for no covariate.}

\item{ticEndogSlopes}{vector defining the slopes for the intercept and slope factors (and the quadratic slope factor, if \code{quadratic = TRUE}) in the prediction of an endogenous time-invariant covariate. Can be omitted for no covariate.}

\item{groupEqual}{parameters that are restricted across groups in both the H0 and the H1 model, when \code{nullEffect} implies a multiple group model. Valid are \code{'imean'}, \code{'smean'}, \code{'s2mean'} to restrict the means of the intercept, linear slope, and quadratic slope factors, and \code{'ivar'}, \code{'svar'}, \code{'s2var'} for their variances, and \code{'iscov'}, \code{'is2cov'}, \code{'ss2cov'} for the covariances between intercept and slope, intercept and quadratic slope, and linear and quadratic slope.}

\item{nullEffect}{defines the hypothesis of interest. See details for valid arguments.}

\item{nullWhichGroups}{for hypothesis involving cross-groups comparisons, vector indicating the groups for which equality constrains should be applied, e.g. \code{c(1, 3)} to constrain the relevant parameters of the first and the third group. If \code{NULL}, all groups are constrained to equality.}

\item{autocorResiduals}{whether the residuals of the indicators of latent variables are autocorrelated over waves (\code{TRUE}, the default) or not (\code{FALSE}). This is only applied to second order LGCMs. This affects the df when the comparison model is the saturated model and generally affects power (also for comparisons to the restricted model).}

\item{...}{mandatory further parameters related to the specific type of power analysis requested, see \code{\link[=semPower.aPriori]{semPower.aPriori()}}, \code{\link[=semPower.postHoc]{semPower.postHoc()}}, and \code{\link[=semPower.compromise]{semPower.compromise()}}, and parameters specifying the factor model. The order of factors is (X1, X2, ..., X_nWaves, ticExogSlopes, ticEndogSlopes). See details.}
}
\value{
a list. Use the \code{summary} method to obtain formatted results. Beyond the results of the power analysis and a number of effect size measures, the list contains the following components:
\item{\code{Sigma}}{the population covariance matrix. A list for multiple group models.}
\item{\code{mu}}{the population mean vector or \code{NULL} when no meanstructure is involved. A list for multiple group models.}
\item{\code{SigmaHat}}{the H0 model implied covariance matrix. A list for multiple group models.}
\item{\code{muHat}}{the H0 model implied mean vector or \code{NULL} when no meanstructure is involved. A list for multiple group models.}
\item{\code{modelH0}}{\code{lavaan} H0 model string.}
\item{\code{modelH1}}{\code{lavaan} H1 model string or \code{NULL} when the comparison refers to the saturated model.}
\item{\code{simRes}}{detailed simulation results when a simulated power analysis (\code{simulatedPower = TRUE}) was performed.}
}
\description{
Convenience function for performing power analysis on effects in a latent growth curve model (LGCM).
This requires the lavaan package.
}
\details{
This function performs a power analysis to reject various hypotheses arising
in latent growth curve models (LGCM), where one variable \code{X} is repeatedly
assessed at different time points (\code{nWaves}), and a latent intercept and a
linear (and optionally a quadratic) latent slope factor is assumed.

Relevant hypotheses in arising in a LCGM are:
\itemize{
\item \code{iMean = 0}, \code{sMean = 0}, \code{s2Mean = 0}: Tests the hypothesis that the mean of the intercept, linear slope, and quadratic slope factors, respectively, is zero.
\item \code{iVar = 0}, \code{sVar = 0}, \code{s2Var = 0}: Tests the hypothesis that the variance of the intercept, linear slope, and quadratic slope factors, respectively, is zero.
\item \code{isCov = 0}: Tests the hypothesis that covariance between the intercept and linear slope factor is zero.
\item \code{is2Cov = 0}: Tests the hypothesis that covariance between the intercept and quadratic slope factor is zero.
\item \code{ss2Cov = 0}: Tests the hypothesis that covariance between the linear and the quadratic slope factor is zero.
\item \code{betaIT = 0}, \code{betaST = 0}, \code{betaS2T = 0}: Tests the hypothesis that the slope for an exogenous time-invariant covariate in the prediction of the intercept, the linear slope, and the quadratic slope factor, respectively, is zero (\verb{TIC -> I, S, S2}).
\item \code{betaTI = 0}, \code{betaTS = 0}, \code{betaTS2 = 0}: Tests the hypothesis that the slope the intercept, the linear slope, and the quadratic slope factor, respectively, in the prediction of an endogenous time-invariant covariate is zero (\verb{I, S, S2 -> TIC}).
\item \code{iMeanA = iMeanB}, \code{sMeanA = sMeanB}, \code{s2MeanA = s2MeanB}: Tests the hypothesis that the means of the intercept, linear slope, and quadratic slope factors, respectively, are equal across groups.
\item \code{iVarA = iVarB}, \code{sVarA = sVarB}, \code{s2VarA = s2VarB}: Tests the hypothesis that the variances of the intercept, linear slope, and quadratic slope factors, respectively, are equal across groups.
\item \code{isCovA = isCovA}: Tests the hypothesis that covariance between the intercept and linear slope factor is equal across groups.
\item \code{is2CovA = is2CovA}: Tests the hypothesis that the covariance between the intercept and quadratic slope factor is equal across groups.
\item \code{ss2CovA = ss2CovA}: Tests the hypothesis that the covariance between the linear and quadratic slope factor is equal across groups.
\item \code{betaITA = betaITB}, \code{betaSTA = betaSTB}, \code{betaS2TA = betaS2TB}: Tests the hypothesis that the slopes for the time-invariant covariate in the prediction of the intercept, the linear slope, and the quadratic slope factor, respectively, are equal across groups (\verb{TIC -> I, S, S2}).
\item \code{betaTIA = betaTIB}, \code{betaTSA = betaTSB}, \code{betaTS2A = betaTS2B}: Tests the hypothesis that the slope the intercept, the linear slope, and the quadratic slope factor, respectively, in the prediction of the time-invariant covariate are equal across groups (\verb{I, S, S2 -> TIC}).
}

For hypotheses regarding longitudinal invariance, see \code{\link[=semPower.powerLI]{semPower.powerLI()}}. For hypotheses regarding a simple autoregression, see \code{\link[=semPower.powerAutoreg]{semPower.powerAutoreg()}}. For hypotheses in an ARMA model, see \code{\link[=semPower.powerARMA]{semPower.powerARMA()}}.

Note that power analyses concerning the hypotheses \code{iVar = 0}, \code{sVar = 0}, and \code{s2Var = 0} are only approximate,
because the H0 model involves a parameter constraint on the boundary of the parameter space (a variance of zero),
so that the correct limiting distribution is a mixture of non-central \eqn{\chi^2} distributions
(see Stoel et al., 2006). In effect, power is (slightly) underestimated.

Stoel, R. D., Garre, F. G., Dolan, C., & Van Den Wittenboer, G. (2006). On the likelihood ratio test in structural equation modeling when parameters are subject to boundary constraints. \emph{Psychological Methods, 11}, 439-455.

Beyond the arguments explicitly contained in the function call, additional arguments
are required specifying the factor model and the requested type of power analysis.

Additional arguments related to the \strong{definition of the factor model}:
\itemize{
\item \code{Lambda}: The factor loading matrix (with the number of columns equaling the number of factors).
\item \code{loadings}: Can be used instead of \code{Lambda}: Defines the primary loadings for each factor in a list structure, e. g. \code{loadings = list(c(.5, .4, .6), c(.8, .6, .6, .4))} defines a two factor model with three indicators loading on the first factor by .5, , 4., and .6, and four indicators loading on the second factor by .8, .6, .6, and .4.
\item \code{nIndicator}: Can be used instead of \code{Lambda}: Used in conjunction with \code{loadM}. Defines the number of indicators by factor, e. g., \code{nIndicator = c(3, 4)} defines a two factor model with three and four indicators for the first and second factor, respectively. \code{nIndicator} can also be a single number to define the same number of indicators for each factor.
\item \code{loadM}: Can be used instead of \code{Lambda}: Used in conjunction with \code{nIndicator}. Defines the loading either for all indicators (if a single number is provided) or separately for each factor (if a vector is provided), e. g. \code{loadM = c(.5, .6)} defines the loadings of the first factor to equal .5 and those of the second factor do equal .6.
}

So either \code{Lambda}, or \code{loadings}, or \code{nIndicator} and \code{loadM} need to be defined. Neither may contain entries referring to the intercept and slope factors.
If the model contains observed variables only, use \code{Lambda = diag(x)} where \code{x} is the number of variables.

The order of the factors is (X1, X2, ..., X_nWaves, ticExogenous, ticEndogenous). If ticExogenous is undefined, ticEndogenous takes its place.

Additional arguments related to the requested type of \strong{power analysis}:
\itemize{
\item \code{alpha}: The alpha error probability. Required for \code{type = 'a-priori'} and \code{type = 'post-hoc'}.
\item Either \code{beta} or \code{power}: The beta error probability and the statistical power (1 - beta), respectively. Only for \code{type = 'a-priori'}.
\item \code{N}: The sample size. Always required for \code{type = 'post-hoc'} and \code{type = 'compromise'}. For \code{type = 'a-priori'} and multiple group analysis, \code{N} is a list of group weights.
\item \code{abratio}: The ratio of alpha to beta. Only for \code{type = 'compromise'}.
}

If a \strong{simulated power analysis} (\code{simulatedPower = TRUE}) is requested, optional arguments can be provided as a list to \code{simOptions}:
\itemize{
\item \code{nReplications}: The targeted number of simulation runs. Defaults to 250, but larger numbers greatly improve accuracy at the expense of increased computation time.
\item \code{minConvergenceRate}:  The minimum convergence rate required, defaults to .5. The maximum actual simulation runs are increased by a factor of 1/minConvergenceRate.
\item \code{type}: specifies whether the data should be generated from a population assuming multivariate normality (\code{'normal'}; the default), or based on an approach generating non-normal data (\code{'IG'}, \code{'mnonr'}, \code{'RC'}, or \code{'VM'}).
The approaches generating non-normal data require additional arguments detailed below.
\item \code{missingVars}: vector specifying the variables containing missing data (defaults to NULL).
\item \code{missingVarProp}: can be used instead of \code{missingVars}: The proportion of variables containing missing data (defaults to zero).
\item \code{missingProp}: The proportion of missingness for variables containing missing data (defaults to zero), either a single value or a vector giving the probabilities for each variable.
\item \code{missingMechanism}: The missing data mechanism, one of \code{MCAR} (the default), \code{MAR}, or \code{NMAR}.
\item \code{nCores}: The number of cores to use for parallel processing. Defaults to 1 (= no parallel processing). This requires the \code{doSNOW} package.
}

\code{type = 'IG'} implements the independent generator approach (IG, Foldnes & Olsson, 2016) approach
specifying third and fourth moments of the marginals, and thus requires that skewness (\code{skewness}) and excess kurtosis (\code{kurtosis}) for each variable are provided as vectors. This requires the \code{covsim} package.

\code{type = 'mnonr'} implements the approach suggested by Qu, Liu, & Zhang (2020) and requires provision of  Mardia's multivariate skewness (\code{skewness})  and kurtosis  (\code{kurtosis}), where
skewness must be non-negative and kurtosis must be at least 1.641 skewness + p (p + 0.774), where p is the number of variables. This requires the \code{mnonr} package.

\code{type = 'RK'} implements the approach suggested by Ruscio & Kaczetow (2008) and requires provision of the population distributions
of each variable (\code{distributions}). \code{distributions} must be a list (if all variables shall be based on the same population distribution) or a list of lists.
Each component must specify the population distribution (e.g. \code{rchisq}) and additional arguments (\code{list(df = 2)}).

\code{type = 'VM'} implements the third-order polynomial method (Vale & Maurelli, 1983)
specifying third and fourth moments of the marginals, and thus requires that skewness (\code{skewness}) and excess kurtosis (\code{kurtosis}) for each variable are provided as vectors.
}
\examples{
\dontrun{
# Determine required N in a 3-wave LGCM model
# to detect that the mean of the slope factor differs from zero
# with a power of 80\% on alpha = 5\%, where
# X is measured by 3 indicators loading by .5 each (at each wave), and
# the mean of the intercept factor is .5 and
# the mean of the slope factor is .2 and
# the variance of the intercept factor is 1 and
# the variance of the slope factor is .5 and
# the intercept-slope covariance is .25 and
# autocorrelated residuals are assumed
powerLGCM <- semPower.powerLGCM(
  'a-priori', alpha = .05, power = .80,
  nWaves = 3,
  means = c(.5, .2),     # i, s
  variances = c(1, .5),  # i, s
  covariances = .25,
  nullEffect = 'sMean = 0',
  nIndicator = rep(3, 3), loadM = .5,
  autocorResiduals = TRUE
)

# show summary
summary(powerLGCM)
# optionally use lavaan to verify the model was set-up as intended
lavaan::sem(powerLGCM$modelH1, sample.cov = powerLGCM$Sigma,
            sample.mean = powerLGCM$mu,
            sample.nobs = powerLGCM$requiredN,
            sample.cov.rescale = FALSE)
lavaan::sem(powerLGCM$modelH0, sample.cov = powerLGCM$Sigma,
            sample.mean = powerLGCM$mu,
            sample.nobs = powerLGCM$requiredN,
            sample.cov.rescale = FALSE)


# same as above, but determine power with N = 250 on alpha = .05
powerLGCM <- semPower.powerLGCM(
  'post-hoc', alpha = .05, N = 250,
  nWaves = 3,
  means = c(.5, .2),
  variances = c(1, .5),
  covariances = .25,
  nullEffect = 'sMean = 0',
  nIndicator = rep(3, 3), loadM = .5,
  autocorResiduals = TRUE
)

# same as above, but determine the critical chi-square with N = 250 so that alpha = beta
powerLGCM <- semPower.powerLGCM(
  'compromise', abratio = 1, N = 250,
  nWaves = 3,
  means = c(.5, .2),
  variances = c(1, .5),
  covariances = .25,
  nullEffect = 'sMean = 0',
  nIndicator = rep(3, 3), loadM = .5,
  autocorResiduals = TRUE
)

# same as above, but compare to the saturated model
# (rather than to the less restricted model)
powerLGCM <- semPower.powerLGCM(
  'a-priori', alpha = .05, power = .80, comparison = 'saturated',
  nWaves = 3,
  means = c(.5, .2),
  variances = c(1, .5),
  covariances = .25,
  nullEffect = 'sMean = 0',
  nIndicator = rep(3, 3), loadM = .5,
  autocorResiduals = TRUE
)

# same as above, but assume only observed variables
powerLGCM <- semPower.powerLGCM(
  'a-priori', alpha = .05, power = .80, comparison = 'saturated',
  nWaves = 3,
  means = c(.5, .2),
  variances = c(1, .5),
  covariances = .25,
  nullEffect = 'sMean = 0',
  Lambda = diag(3),
  autocorResiduals = TRUE
)

# same as above, but provide reduced loadings matrix to define that
# X is measured by 3 indicators each loading by .5, .6, .4 (at each wave)
powerLGCM <- semPower.powerLGCM(
  'a-priori', alpha = .05, power = .80, comparison = 'saturated',
  nWaves = 3,
  means = c(.5, .2),
  variances = c(1, .5),
  covariances = .25,
  nullEffect = 'sMean = 0',
  loadings = list(
    c(.5, .6, .4),  # X1
    c(.5, .6, .4),  # X2 
    c(.5, .6, .4)   # X3
  ),
  autocorResiduals = TRUE
)

# same as above, but get required N to detect that
# the variance of the intercept factor differs from zero
powerLGCM <- semPower.powerLGCM(
  'a-priori', alpha = .05, power = .80,
  nWaves = 3,
  means = c(.5, .2),
  variances = c(1, .5),
  covariances = .25,
  nullEffect = 'iVar = 0',
  nIndicator = rep(3, 3), loadM = .5,
  autocorResiduals = TRUE
)

# same as above, but get required N to detect that
# the intercept-slope covariance differs from zero
powerLGCM <- semPower.powerLGCM(
  'a-priori', alpha = .05, power = .80,
  nWaves = 3,
  means = c(.5, .2),
  variances = c(1, .5),
  covariances = .25,
  nullEffect = 'iscov = 0',
  nIndicator = rep(3, 3), loadM = .5,
  autocorResiduals = TRUE
)


# include a quadratic slope factor
# and get required N to detect that
# its variance differs from zero.
# provide the variance-covariance matrix
# between intercept, slope, and quadratic slope factors
powerLGCM <- semPower.powerLGCM(
  'a-priori', alpha = .05, power = .80,
  nWaves = 4,
  quadratic = TRUE,
  means = c(.5, .2, .1),
  covariances = matrix(c(
    # i  s   s2
    c(1, .3, .2),
    c(.3, .5, .01),
    c(.2, .01, .1)
  ), ncol = 3, byrow = TRUE),
  nullEffect = 's2var = 0',
  nIndicator = rep(3, 4), loadM = .5,
  autocorResiduals = TRUE
)

# Determine required N in a 3-wave LGCM model
# to detect that the slope of an time-invariant covariate (TIC)
# on the slope factor differs from zero. 
# The TIC is measured by 4 indicators loading
# by .7, .7, .5, and .8. The slope of the TIC in the prediction of
# the intercept factor is .5, and in the prediction of the slope factor is .4.
powerLGCM <- semPower.powerLGCM(
  'a-priori', alpha = .05, power = .80,
  nWaves = 3,
  means = c(.5, .2),
  variances = c(1, .5),
  covariances = .25,
  ticExogSlopes = c(.5, .4),  # i s
  nullEffect = 'betaST = 0',
  loadings = list(
    c(.5, .6, .4),      # X1
    c(.5, .6, .4),      # X2 
    c(.5, .6, .4),      # X3
    c(.7, .7, .5, .8)   # TIC
  ),
  autocorResiduals = TRUE
)

# Determine required N in a 3-wave LGCM model
# to detect that the slope of the slope factor in 
# the prediction of a time-invariant covariate (TIC) differs from zero. 
# The TIC is measured by 4 indicators loading
# by .7, .7, .5, and .8. The slopes of the intercept and the slope factors in
# the prediction of the TIC are .1 and .3, respectively.
powerLGCM <- semPower.powerLGCM(
  'a-priori', alpha = .05, power = .80,
  nWaves = 3,
  means = c(.5, .2),
  variances = c(1, .5),
  covariances = .25,
  ticEndogSlopes = c(.1, .3),  # i s
  nullEffect = 'betaTS = 0',
  loadings = list(
    c(.5, .6, .4),      # X1
    c(.5, .6, .4),      # X2 
    c(.5, .6, .4),      # X3
    c(.7, .7, .5, .8)   # TIC
  ),
  autocorResiduals = TRUE
)

# Determine required N in a 3-wave LGCM model
# to detect that the mean of the slope factor in group 1
# differs from the mean of the slope factor in group 2
# with a power of 80\% on alpha = 5\%, where
# X is measured by 3 indicators loading by .5 each (at each wave and in each group), and
# the means of the intercept factor in group 1 and 2 are .5 and .25
# the means of the slope factor in group 1 and 2 are .25 and .4
# the variance of the intercept factor is 1 in both groups and
# the variance of the slope factor is .5in both groups and
# the intercept-slope covariance is .25 in both groups and
# autocorrelated residuals are assumed and
# the groups are equal-sized
powerLGCM <- semPower.powerLGCM(
  'a-priori', alpha = .05, power = .80, N = list(1, 1),
  nWaves = 3,
  means = list(
    # i, s
    c(.5, .2),     # group 1 
    c(.25, .4)),   # group 2
  variances = c(1, .5),
  covariances = .25,
  nullEffect = 'sMeanA = sMeanB',
  nIndicator = rep(3, 3), loadM = .5,
  autocorResiduals = TRUE
)

# similar as above, but get required N to detect that
# the intercept-slope covariance differs across groups, 
# assuming that intercept and slope variances are equal across groups.
powerLGCM <- semPower.powerLGCM(
  'a-priori', alpha = .05, power = .80, N = list(1, 1),
  nWaves = 3,
  means = c(.5, .2),  
  variances = c(1, .5),
  covariances = list(
    c(.25), # group 1 
    c(.1)), # group 2
  nullEffect = 'isCovA = isCovB',
  groupEqual = c('ivar', 'svar'),
  nIndicator = rep(3, 3), loadM = .5,
  autocorResiduals = TRUE
)

# perform a simulated post-hoc power analysis
# with 250 replications
set.seed(300121)
powerLGCM <- semPower.powerLGCM(
  'post-hoc', alpha = .05, N = 500,
  nWaves = 3,
  means = c(.5, .2),     # i, s
  variances = c(1, .5),  # i, s
  covariances = .25,
  nullEffect = 'sMean = 0',
  nIndicator = rep(3, 3), loadM = .5,
  autocorResiduals = TRUE,
  simulatedPower = TRUE,
  simOptions = list(nReplications = 250)
)
}
}
\seealso{
\code{\link[=semPower.genSigma]{semPower.genSigma()}} \code{\link[=semPower.aPriori]{semPower.aPriori()}} \code{\link[=semPower.postHoc]{semPower.postHoc()}} \code{\link[=semPower.compromise]{semPower.compromise()}}
}
