################################################################################
# --------------------------- nulldistr_estimator ---------------------------- #
################################################################################

test_that("nulldistr_estimator rejects invalid N and cores", {
  X1 <- rnorm(10)
  n1 <- length(X1)
  n2 <- 8
  expect_error(
    nulldistr_estimator(
      X1 = X1, n2 = n2, Te = 2,
      m1 = 5, m2 = 4,
      q1 = 1, method1 = "ML", f1_true = NULL,
      q2 = 1, method2 = "ML", f2_true = NULL,
      D1 = diag(n1), D2 = diag(n2),
      evals1 = rep(1, 2), evals2 = rep(1, 2),
      N = 0, cores = 1
    ),
    "natural number"
  )
  
  expect_error(
    nulldistr_estimator(
      X1 = X1, n2 = n2, Te = 2,
      m1 = 5, m2 = 4,
      q1 = 1, method1 = "ML", f1_true = NULL,
      q2 = 1, method2 = "ML", f2_true = NULL,
      D1 = diag(n1), D2 = diag(n2),
      evals1 = rep(1, 2), evals2 = rep(1, 2),
      N = 5, cores = 0
    ),
    "natural number"
  )
})

test_that("nulldistr_estimator computes the statistic correctly", {
  
  fake_logsdf <- function(...) rep(1, 4)
  
  fake_sdf2acf <- function(x) c(1, rep(0, length(x) - 1))
  
  local_mocked_bindings(
    logsdf_estimator = fake_logsdf,
    sdf2acf = fake_sdf2acf
  )
  
  X1 <- rnorm(10)
  n1 <- length(X1)
  n2 <- 8
  
  S <- nulldistr_estimator(
    X1 = X1, n2 = n2, Te = 2,
    m1 = 5, m2 = 4,
    q1 = 1, method1 = "ML", f1_true = NULL,
    q2 = 1, method2 = "ML", f2_true = NULL,
    D1 = diag(n1), D2 = diag(n2),
    evals1 = rep(1, 2), evals2 = rep(1, 2),
    N = 5, cores = 1
  )
  
  expect_true(all(is.finite(S)))
  expect_type(S, "double")
  expect_length(S, 5)
  expect_equal(S, numeric(5))
})

################################################################################
# --------------------------------- sdf.test --------------------------------- #
################################################################################

test_that("sdf.test returns correct structure and uses mocks", {
  
  X1 <- rnorm(20)
  X2 <- rnorm(18)
  
  fake_logsdf <- function(...) rep(1, 4)
  
  fake_null <- function(...) rep(0.5, 5)
  
  fake_DCT <- function(n) diag(n)
  
  fake_Q <- function(...) rep(1, 2)
  
  fake_sinc <- function(x) rep(1, length(x))
  
  local_mocked_bindings(
    logsdf_estimator = fake_logsdf,
    nulldistr_estimator = fake_null,
    DCT.matrix = fake_DCT,
    Q_pminus1 = fake_Q,
    sinc = fake_sinc
  )
  
  out <- sdf.test(
    X1 = X1,
    X2 = X2,
    Te = 2,
    alpha = 0.05,
    q1 = 1,
    method1 = "ML",
    N = 5,
    cores = 1
  )
  
  expect_s3_class(out, "test")
  expect_named(out, c("result", "S", "quantile", "pval", "nulldistr"))
  
  expect_equal(out$S, 0)
  expect_true(out$result)
})

test_that("custom nulldistr prevents null estimator from being called", {
  
  X1 <- rnorm(20)
  X2 <- rnorm(18)
  
  local_mocked_bindings(
    nulldistr_estimator = function(...) stop("Should not be called"),
    logsdf_estimator = function(...) rep(1, 4),
    DCT.matrix = function(n) diag(n),
    Q_pminus1 = function(...) rep(1, 2),
    sinc = function(x) rep(1, length(x))
  )
  
  out <- sdf.test(
    X1 = X1,
    X2 = X2,
    Te = 2,
    alpha = 0.05,
    nulldistr = rep(0.1, 10)
  )
  
  expect_equal(out$nulldistr, rep(0.1, 10))
})

################################################################################
# -------------------------------- print.test -------------------------------- #
################################################################################

test_that("print.test snapshot", {
  
  x <- structure(
    list(
      result = FALSE,
      S = 0.1,
      quantile = 0.2,
      pval = 0.3,
      nulldistr = 1:3
    ),
    class = "test"
  )
  
  expect_snapshot(print.test(x))
})