#' @title Run SaTScan in the OS
#' 
#' @description Calls out into the OS to run SaTScan, with the parameter file specified
#' 
#' @param prmlocation A string containing the 
#' directory location where the paramter file is located. 
#' @param prmfilename A string containg the name of the parameter file, without the extension, i.e., no ".prm".
#' @param sslocation A string containing the directory location where satscanbatch.exe (Windows) 
#' is located.  The default value is a common location in Windows 7.
#' @param ssbatchfilename Name of the file containing the SaTScan executable.  This is likely to be
#' either SaTScanBatch or SaTScanBatch64.  Omit the file extension.
#' @param cleanup If true, deletes any SaTScan output files from the OS.
#' @param verbose If true, will display the results in the R console as if running SaTScan in batch.  This may be especially
#' useful if you expect SaTScan to take a long time to run.
#' 
#' @details The parameter file may have been made by the \code{\link{ss.options}} function or not.  
#' 
#' @return A satscan-class object, which is a list of 8 items, not all of which are always made, depending on SaTScan options and whether the program call was successful or not:
#' \describe{
#'   \item{main}{A character vector containing the main text output from SaTScan.  This is 
#'        probably identical to the material displayed when verbose=True}
#'   \item{col}{A data frame with the basic cluster information dataset SaTScan makes.}
#'   \item{rr}{A data frame with the risk ratio dataset SaTScan makes.}
#'   \item{gis}{A data frame with the geographic information dataset SaTScan makes.}
#'   \item{llr}{A data frame with the log likelihood ratios dataset SaTScan makes.}
#'   \item{sci}{A data frame with the other cluster information dataset SaTScan makes.}
#'   \item{shapeclust}{A list object, of class sf, defined by the \code{sf} package.
#'   It contains the ESRI shapefile(s) SaTScan makes.  This is made only if the \code{sf}
#'   package is available.}
#'   \item{prm}{A character vector containing the contents of the parameter file you told SaTScan 
#'   to use.}
#' }
#' If an item is not made by SaTScan, it will be NA.
#' 
#' @examples 
#' 
#' \dontrun{
#' ## Please see vignette("rsatscan"); example() code doesn't make sense since
#' ## all examples rely on calls to SaTScan in the OS.
#' }
#' 
#' 
#' @seealso \code{\link{ss.options}}, \code{\link{write.ss.prm}}
#' 
#' @export

satscan = function(
  prmlocation,  prmfilename,  sslocation = "c:/progra~2/satscan",
  ssbatchfilename = "SaTScanBatch", cleanup = TRUE, verbose=FALSE) {

    ssfile = paste0(stripslash(sslocation), "/", ssbatchfilename)
    if (Sys.which(ssfile) =="")
          stop("SaTScan is not there or is not runnable")
    prmloc = paste0(stripslash(prmlocation),"/")
    infile = paste0(prmloc, prmfilename,".prm")
    if (!file.exists(infile))  stop("I can't find that parameter file")
    prm = suppressWarnings(readLines(infile))
    
    # spec.version.line = prm[startsWith(prm, "Version")]
    # spec.version.string = strsplit(spec.version.line, "=", fixed=TRUE)[[1]][2]
    # spec.version.components = strsplit(spec.version.string, ".", fixed=TRUE)[[1]]
    # spec.major.version = as.numeric(spec.version.components[1])
    # spec.minor.version = as.numeric(spec.version.components[2])
    # 
    # satscan.version.command = paste(shQuote(ssfile), "--version")
    # satscan.version.line = system(satscan.version.command, intern=TRUE)
    # satscan.version.string = strsplit(satscan.version.line, " ", fixed=TRUE)[[1]][2]
    # satscan.version.components = strsplit(satscan.version.string, ".", fixed=TRUE)[[1]]
    # satscan.major.version = as.numeric(satscan.version.components[1])
    # satscan.minor.version = as.numeric(satscan.version.components[2])
    # 
    # if((spec.major.version > satscan.major.version) || ((spec.major.version == satscan.major.version) && (spec.minor.version > satscan.minor.version))) {
    #   warning.message <- paste0("Warning: You are running SaTScan version ", satscan.version.string, " with parameters defined for version ", spec.version.string)
    #   warning(warning.message)
    # }
 
    status <- system(paste(shQuote(ssfile), shQuote(infile)), show.output.on.console=verbose)
    
    if(status != 0 && !verbose) {
      
      warning("Error running SaTScan executable. Please use verbose = TRUE for more information.")
    }
    
    resultsfile.line <- prm[startsWith(prm, "ResultsFile")]
    resultsfilepath <- strsplit(resultsfile.line, "=", fixed=TRUE)[[1]][2]
    
    resultsfile <- basename(resultsfilepath)
    if(resultsfile == resultsfilepath) {
      
      resultsfileloc <- prmloc
    } else {
      
      resultsfileloc <- paste0(dirname(resultsfilepath), "/")
    }

    resultsfilename <- sub("\\.[^\\.]*$", "", resultsfile)
    
    mainfile = if  (file.exists(paste0(resultsfileloc,resultsfile)))
      read.satscanmain(resultsfileloc,resultsfile) else NA
    colfile = if  (file.exists(paste0(resultsfileloc,resultsfilename,".col.dbf")))
      read.col(resultsfileloc,resultsfilename) else NA
    rrfile = if  (file.exists(paste0(resultsfileloc,resultsfilename,".rr.dbf")))
      read.rr(resultsfileloc,resultsfilename) else NA
    gisfile = if  (file.exists(paste0(resultsfileloc,resultsfilename,".gis.dbf")))
      read.gis(resultsfileloc,resultsfilename) else NA
    llrfile = if  (file.exists(paste0(resultsfileloc,resultsfilename,".llr.dbf")))
      read.llr(resultsfileloc,resultsfilename) else NA
    scifile = if  (file.exists(paste0(resultsfileloc,resultsfilename,".sci.dbf")))
      read.sci(resultsfileloc,resultsfilename) else NA  
    if (verbose) cat("\n \n Any following message is from st_read() in the sf package \n")
      shpfile = if (file.exists(paste0(resultsfileloc,resultsfilename,".col.shp"))) {
        if (requireNamespace("sf", quietly = TRUE)) {
          sf::st_read(dsn=stripslash(resultsfileloc), layer=paste0(resultsfilename,".col"),
                    quiet=!verbose) 
                }  else cat("\n \n sf package not installed, so shapefile can't be imported")
          } else NA
    if (cleanup) {
        suppressWarnings(file.remove(paste0(resultsfileloc,resultsfile)))
        suppressWarnings(file.remove(paste0(resultsfileloc,resultsfilename,".col.dbf")))
        suppressWarnings(file.remove(paste0(resultsfileloc,resultsfilename,".rr.dbf")))
        suppressWarnings(file.remove(paste0(resultsfileloc,resultsfilename,".gis.dbf")))
        suppressWarnings(file.remove(paste0(resultsfileloc,resultsfilename,".llr.dbf")))
        suppressWarnings(file.remove(paste0(resultsfileloc,resultsfilename,".sci.dbf")))
        suppressWarnings(file.remove(paste0(resultsfileloc,resultsfilename,".col.shp")))
        suppressWarnings(file.remove(paste0(resultsfileloc,resultsfilename,".col.shx")))
      }
  return(structure(list(main=mainfile, col=colfile, rr = rrfile, gis=gisfile, llr=llrfile, 
                        sci=scifile, shapeclust=shpfile, prm=prm), class="satscan"))
}




