
#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
#' Generate bytes from a password using Argon2 password-based key derivation
#' 
#' Argon2 is a resource intensive password-based key derivation scheme. A typical
#' application is generating an encryption key from a text password.
#' 
#' @section Note:
#' Using the same password with the same salt will always generate the same
#' key.  It is recommended that a random salt be used.
#' 
#' @section Technical Note:
#' The 'C' version of the ARgon2 algorithm is configured with:
#' 
#' \itemize{
#'   \item{Use the \code{Argon2id} variant of the algorithm}
#'   \item{single-threaded}
#'   \item{3 iterations}
#'   \item{100 megabytes of memory}
#' }
#' 
#' See \url{https://en.wikipedia.org/wiki/Argon2} and 
#' \url{https://monocypher.org/manual/argon2} for more information.
#' 
#' 
#' @param password A character string used to derive the random bytes
#' @param length Number of bytes to output. Default: 32
#' @param salt 16-byte raw vector or 32-character hexadecimal string.
#'        A salt is data used as additional input to key derivation
#'        which helps defend against attacks that use pre-computed (i.e. rainbow) tables.
#'        Note: A salt does not need to be a secret.
#'        See \url{https://en.wikipedia.org/wiki/Salt_(cryptography)} for more details.
#'        The 'salt' may also be a non-hexadecimal string, in which case a real
#'        salt will be created by using Argon2 with a default internal salt.
#' @param type Should the data be returned as raw bytes? Default: "chr". 
#'        Possible values "chr" or 'raw'
#'
#' @return raw vector of the requested length
#' @export
#' 
#' @examples
#' # For the sake of convenience for novice users, a salt will be 
#' # derived internally from the password.
#' argon2("my secret")
#'
#' # Calling 'argon2()' without a seed is equivalent to using the password
#' # as the seed.  This is not the best security practice
#' argon2("my secret", salt = "my secret")
#'
#' # Best practice is to use random bytes for the salt
#' # This particular key can then only be recovered if the password and
#' # the salt are known.
#' salt <- rbyte(16) # You'll want to save this value somewhere
#' argon2("my secret", salt = salt)
#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
argon2 <- function(password, salt = password, length = 32, type = "chr") {
  .Call(argon2_, password, salt, length, type);
}
