#' Results Tables for Epidemiology
#'
#' @description This function displays descriptive
#' and inferential results for binary, continuous, and survival data
#' in the format of a table stratified by exposure and, if requested, by
#' effect modifiers.
#'
#' This function is intended only for tabulations of final results.
#' Model diagnostics for regression models need to be conducted separately.
#'
#' @param design Design matrix (data frame) that sets up the table.
#'   See Details. Must be provided.
#' @param data Dataset to be used for all analyses. Must be provided unless
#'   the \code{design} was generated by \code{\link[rifttable]{table1_design}}.
#' @param id Optional. Name of an \code{id} variable in the \code{data} that
#'   identifies clustered observations, for example if the data are in a
#'   long format with rows encoding time-varying covariates.
#'   See documentation for which estimators use this information.
#'   Defaults to \code{""}, i.e., each row is a unique individual.
#' @param layout Optional. \code{"rows"} uses the \code{design} as rows and
#'   exposure categories as columns. \code{"cols"} is the
#'   opposite: \code{design} as columns and exposure categories as rows.
#'   Defaults to \code{"rows"}.
#' @param factor Optional. Used for \code{type = "rates"}: Factor to multiply
#'   events per person-time by. Defaults to \code{1000}.
#' @param risk_percent Optional. Show risk and risk difference estimates in
#'   percentage points instead of proportions. Defaults to \code{FALSE} unless
#'   the \code{design} was generated by \code{\link[rifttable]{table1_design}}.
#'   In this latter case, if \code{risk_percent} is not provided, it will
#'   default to \code{TRUE}.
#' @param risk_digits Optional. Number of decimal digits to show for risks/
#'   cumulative incidence. Defaults to \code{2} for \code{risk_percent = FALSE}
#'   and to \code{0} for \code{risk_percent = TRUE}. Alternatively,
#'   \code{digits} can be specified directly for each row of the \code{design}.
#' @param diff_digits Optional. Number of decimal digits to show for
#'   rounding of means and mean difference estimates. Defaults to \code{2}.
#'   Alternatively, \code{digits} can be specified directly for each row of the
#'   \code{design}.
#' @param ratio_digits Optional. Number of decimal digits to show for ratio
#'   estimates. Defaults to \code{2}. Alternatively, \code{digits} can be
#'   specified directly for each row of the \code{design}.
#' @param ratio_digits_decrease Optional. Lower limits of ratios above which
#'   fewer digits should be shown. Provide a named vector of the format,
#'   \code{c(`3` = -1, `10` = -2)} to reduce the number of rounding digits by
#'   1 digit for ratios greater than 3 and by 2 digits for ratios greater than
#'   10 (the default). To disable, set to \code{NULL}.
#' @param rate_digits Optional. Number of decimal digits to show for rates.
#'   Defaults to \code{1}. Alternatively,  \code{digits} can be specified
#'   directly for each row of the \code{design}.
#' @param to Optional. Separator between the lower and the upper bound
#'   of the 95% confidence interval (and interquartile range for medians).
#'   Defaults to \code{", "}.
#' @param type2_layout Optional. If a second estimate is requested via
#'   \code{type2} in the \code{design} matrix, display it as rows below
#'   (\code{"rows"}) or as columns (\code{"columns"}) to the right. Defaults to
#'   \code{"rows"}.
#' @param overall Optional. Defaults to \code{FALSE}. Add a first column with
#'   unstratified estimates to an exposure-stratified table? Elements will be
#'   shown only for absolute estimates (e.g., \code{type = "mean"}) and blank
#'   for comparative estimates (e.g., mean difference via
#'   \code{type = "diff"}).
#' @param reference Optional. Defaults to \code{"(reference)"}. Alternative
#'   label for the reference category.
#' @param exposure_levels Optional. Defaults to \code{"noempty"}. Show only
#'   exposure levels that exist in the data or are \code{NA}
#'   (\code{"noempty"}); show only exposure levels that are neither \code{NA}
#'   nor empty (\code{"nona"}); or show all exposure levels even if they are
#'   \code{NA} or a factor level that does not occur in the data
#'   (\code{"all"}).
#'
#' @details The main input parameter is the dataset \code{design}.
#'   Always required are the column \code{type} (the type of requested
#'   statistic, see below), as well as \code{outcome} for binary outcomes or
#'   \code{time} and \code{event} for survival outcomes:
#'
#'   *  \code{label} A label for each row (or column). If missing, \code{type}
#'        will be used as the label.
#'   *  \code{exposure} Optional. The exposure variable. Must be categorical
#'        (factor or logical). If missing (\code{NA}), then an unstratified
#'        table with absolute estimates only will be returned.
#'   *  \code{outcome} The outcome variable for non-survival data
#'        (i.e., whenever \code{event} and \code{time} are not used).
#'        For risk/prevalence data, this variable must be \code{0}/\code{1}
#'        or \code{FALSE}/\code{TRUE}.
#'   *  \code{time} The time variable for survival data. Needed for,
#'        e.g., \code{type = "hr"} and \code{type = "rate"}
#'        (i.e., whenever \code{outcome} is not used).
#'   *  \code{time2} The second time variable for late entry models.
#'        Only used in conjunction with \code{time}. If provided,
#'        \code{time} will become the entry time and \code{time2}
#'        the exit time, following conventions of
#'        \code{\link[survival]{Surv}}.
#'   *  \code{event} The event variable for survival data.
#'        Events are typically \code{1}, censored observations \code{0}.
#'        If competing events are present, censoring should be the
#'        first-ordered level, e.g., of a factor, and the level corresponding
#'        to the event of interest should be supplied as
#'        \code{event = "event_variable@Recurrence"} if \code{"Recurrence"} is
#'        the event of interest. The \code{event} variable is needed for, e.g.,
#'        \code{type = "hr"} and \code{type = "rate"}, i.e., whenever
#'        \code{outcome} is not used.
#'   *  \code{trend} Optional. For regression models, a continuous
#'        representation of the exposure, for which a slope per one unit
#'        increase ("trend") will be estimated. Must be a numeric variable.
#'        If joint models for \code{exposure} and \code{effect_modifier} are
#'        requested, trends are still reported within each stratum of the
#'        \code{effect_modifier}. Use \code{NA} to leave blank.
#'   *  \code{effect_modifier} Optional. A categorical effect modifier variable.
#'        Use \code{NA} to leave blank.
#'   *  \code{stratum} Optional. A stratum of the effect modifier.
#'        Use \code{NULL} to leave blank. \code{NA} will evaluate
#'        observations with missing data for the \code{effect_modifier}.
#'   *  \code{confounders} Optional. A string in the format
#'        \code{"+ var1 + var2"} that will be substituted into
#'        into \code{formula = exposure + confounders}.
#'        Use \code{NA} or \code{""} (empty string) to leave blank; the default.
#'        For Cox models, can add \code{"+ strata(site)"}
#'        to obtain models with stratification by, e.g., \code{site}.
#'        For Poisson models, can add \code{"+ offset(log(persontime))"}
#'        to define, e.g., \code{persontime} as the offset.
#'   * \code{weights} Optional. Variable with weights, for example inverse-
#'        probability weights. Used by comparative survival estimators (e.g.,
#'        \code{type = "hr"} and \code{type = "cumincdiff"}) as well as
#'        \code{type = "cuminc"} and \code{type = "surv"}. They are ignored
#'        by other estimators. The spelling \code{weight} is also accepted as a
#'        fallback.
#'   *  \code{type} The statistic requested (case-insensitive):
#'
#'      Comparative estimates with 95% confidence intervals:
#'
#'      * \code{"hr"} Hazard ratio from Cox proportional
#'        hazards regression.
#'      * \code{"irr"} Incidence rate ratio for count outcomes
#'        from Poisson regression model.
#'      * \code{"irrrob"} Ratio for other outcomes
#'        from Poisson regression model with robust (sandwich) standard errors.
#'      * \code{"rr"} Risk ratio (or prevalence ratio)
#'        from \code{\link[risks]{riskratio}}. Can request specific model
#'        fitting  approach and, for marginal
#'        standardization only, the number of bootstrap repeats.
#'        Examples: \code{"rrglm_start"} or \code{"rrmargstd 2000"}.
#'      * \code{"rd"} Risk difference (or prevalence difference)
#'        from \code{\link[risks]{riskdiff}}. Can request model fitting
#'        approach and bootstrap repeats as for \code{"rr"}.
#'      * \code{"diff"} Mean difference from linear model.
#'      * \code{"quantreg"} Quantile difference from quantile regression using
#'        \code{\link[quantreg]{rq}} with \code{method = "fn"}.
#'        By default, this is the difference in medians. For a different
#'        quantile, e.g., the 75th percentile, use \code{"quantreg 0.75"}.
#'      * \code{"fold"} Fold change from generalized linear
#'        model with log link (i.e., ratio of arithmetic means).
#'      * \code{"foldlog"} Fold change from linear
#'        model after log transformation of the outcome
#'        (i.e., ratio of geometric means).
#'      * \code{"or"} Odds ratio from logistic regression.
#'      * \code{"survdiff"} Difference in survival from Kaplan-Meier estimator.
#'        Provide time horizon, e.g., \code{"survdiff 2.5"} to evaluate
#'        differences in survival at 2.5 years.
#'        Uses \code{\link[rifttable]{survdiff_ci}}.
#'      * \code{"cumincdiff"} Difference in cumulative incidence from the
#'        Kaplan-Meier estimator or, if competing risks are present, its
#'        generalized form, the Aalen-Johansen estimator. Provide time horizon,
#'        e.g., \code{"cumincdiff 2.5"} to evaluate differences in cumulative
#'        incidence at 2.5 years. Uses \code{\link[rifttable]{survdiff_ci}}.
#'      * \code{"survratio"} Ratio in survival from Kaplan-Meier estimator.
#'        Provide time horizon, e.g., \code{"survdiff 2.5"} to evaluate
#'        2.5-year relative risk. Uses \code{\link[rifttable]{survdiff_ci}}.
#'      * \code{"cumincratio"} Ratio in cumulative incidence from the
#'        Kaplan-Meier estimator or, if competing risks are present, its
#'        generalized form, the Aalen-Johansen estimator. Provide time horizon,
#'        e.g., \code{"cumincdiff 2.5"} to evaluate the 2.5-year risk
#'        difference. Uses \code{\link[rifttable]{survdiff_ci}}.
#'
#'      Absolute estimates per exposure category:
#'
#'      * \code{"events"} Event count.
#'      * \code{"time"} Person-time.
#'      * \code{"outcomes"} Outcome count.
#'      * \code{"total"} Number of observations.
#'      * \code{"events/time"} Events slash person-time.
#'      * \code{"events/total"} Events slash number of observations.
#'      * \code{"cases/controls"} Cases and non-cases (events and non-events);
#'        useful for case-control studies.
#'      * \code{"risk"} Risk (or prevalence), calculated as a proportion,
#'        i.e., outcomes divided by number of observations. Change between
#'        display as proportion or percent using the parameter
#'        \code{risk_percent}.
#'      * \code{"risk (ci)"} Risk with 95% confidence interval
#'        (Wilson score interval for binomial proportions, see
#'        \code{\link[rifttable]{scoreci}}).
#'      * \code{"cuminc"} Cumulative incidence ("risk") from the Kaplan-Meier
#'        estimator or, if competing risks are present, its generalized form,
#'        the Aalen-Johansen estimator. Provide time point (e.g., 1.5-year
#'        cumulative incidence) using \code{"cuminc 1.5"}. If no time point is
#'        provided, the cumulative incidence at end of follow-up is returned.
#'        Change between display as proportion or percent using the parameter
#'        \code{risk_percent}.
#'      * \code{"cuminc (ci)"} Cumulative incidence ("risk"), as above,
#'        with 95% confidence intervals (Greenwood standard errors with log
#'        transformation, the default of the survival package/
#'        \code{\link[survival]{survfit}}).
#'        Provide time point as in \code{"cuminc"}.
#'      * \code{"surv"} Survival from the Kaplan-Meier
#'        estimator. Provide time point (e.g., 1.5-year survival)
#'        using \code{"surv 1.5"}. If no time point is provided, returns
#'        survival at end of follow-up. Change between display as
#'        proportion or percent using the parameter \code{risk_percent}.
#'      * \code{"surv (ci)"} Survival from the
#'        Kaplan-Meier estimator with 95% confidence interval (Greenwood
#'        standard errors with log transformation, the
#'        default of the survival package/\code{\link[survival]{survfit}}).
#'        Provide time point as in \code{"surv"}.
#'      * \code{"rate"} Event rate: event count divided by person-time,
#'        multiplied by \code{factor}.
#'      * \code{"rate (ci)"} Event rate with 95% confidence interval
#'        (Poisson-type large-sample interval).
#'      * \code{"outcomes (risk)"} A combination: Outcomes
#'        followed by risk in parentheses.
#'      * \code{"outcomes/total (risk)"} A combination: Outcomes slash total
#'        followed by risk in parentheses.
#'      * \code{"events/time (rate)"} A combination: Events slash time
#'        followed by rate in parentheses.
#'      * \code{"medsurv"} Median survival.
#'      * \code{"medsurv (ci)"} Median survival with 95% confidence interval.
#'      * \code{"medfu"} Median follow-up (reverse Kaplan-Meier), equals median
#'        survival for censoring.
#'      * \code{"medfu (iqr)"} Median and interquartile range for follow-up.
#'      * \code{"maxfu"} Maximum follow-up time.
#'      * \code{"mean"} Mean (arithmetic mean).
#'      * \code{"mean (ci)"} Mean and 95% CI.
#'      * \code{"mean (sd)"} Mean and standard deviation.
#'      * \code{"geomean"} Geometric mean.
#'      * \code{"median"} Median.
#'      * \code{"median (iqr)"} Median and interquartile range.
#'      * \code{"range"} Range: Minimum to maximum value.
#'      * \code{"sum"} Sum.
#'      * \code{"blank"} or \code{""} An empty line.
#'      * Custom: A custom function that must be available under the name
#'        \code{estimate_my_function} in order to be callable as
#'        \code{type = "my_function"}.
#'
#'      By default, regression models will be fit separately for each
#'      stratum of the \code{effect_modifier}. Append \code{"_joint"}
#'      to \code{"hr"}, \code{"rr"}, \code{"rd"}, \code{"irr"},
#'      \code{"irrrob"}, \code{"diff"}, \code{"fold"}, \code{"foldlog"},
#'      \code{"quantreg"}, or \code{"or"} to
#'      obtain "joint" models for exposure and effect modifier that have a
#'      single reference category.
#'      Example: \code{type = "hr_joint"}. The reference categories
#'      for exposure and effect modifier are their first factor levels, which
#'      can be changed using \code{fct_relevel} from the forcats package.
#'      Note that
#'      the joint model will be fit across all non-missing (\code{NA}) strata
#'      of the effect modifier, even if the \code{design} table does not
#'      request all strata be shown.
#'
#'   * \code{type2} Optional. A second statistic that is added in an adjacent
#'     row or column (global option \code{type2_layout} defaults to \code{"row"}
#'     and can alternatively be set to \code{"column"}). For example, use
#'     \code{type = "events/times", type2 = "hr"} to get both event
#'     counts/person-time and hazard ratios for the same data, exposure,
#'     stratum, confounders, and outcome.
#'   * \code{digits} Optional. The number of digits for rounding an individual
#'     line. Defaults to \code{NA}, where the number of
#'     digits will be determined based on \code{rifttable}'s arguments
#'     \code{risk_percent}, \code{risk_digits}, \code{diff_digits},
#'     \code{ratio_digits}, or \code{rate_digits}, as applicable.
#'   * \code{digits2} Optional. As \code{digits}, for the second
#'     estimate (\code{type2}).
#'   * \code{nmin}. Optional. Suppress estimates with \code{"--"} if a cell
#'     defined by exposure, and possibly the effect modifier, contains fewer
#'     observations or, for survival analyses, fewer events than \code{nmin}.
#'     Defaults to \code{NA}, i.e., to print all estimates.
#'   * \code{na_rm}. Optional. Exclude observations with missing outcome.
#'     Defaults to \code{FALSE}. Use with caution.
#'   * \code{ci}. Optional. Confidence level. Defaults to \code{0.95}.
#'
#' Use \code{\link[tibble]{tibble}}, \code{\link[tibble]{tribble}}, and
#' \code{\link[dplyr]{mutate}} to construct the \code{design} dataset,
#' especially variables that are used repeatedly (e.g., \code{exposure, time,
#' event}, or \code{outcome}). See examples.
#'
#' If regression models cannot provide estimates in a stratum, e.g.,
#' because there are no events, then \code{"--"} will be printed. Accompanying
#' warnings need to be suppressed manually, if appropriate, using
#' \code{suppressWarnings(rifttable(...))}.
#'
#' @return Tibble. Get formatted output as a gt table by passing on to
#'   \code{\link[rifttable]{rt_gt}}.
#' @export
#' @importFrom rlang .data `:=`
#'
#' @references
#' Greenland S, Rothman KJ (2008). Introduction to Categorical Statistics. In:
#' Rothman KJ, Greenland S, Lash TL. Modern Epidemiology, 3rd edition.
#' Philadelpha, PA: Lippincott Williams & Wilkins. Page 242.
#' (Poisson/large-sample approximation for variance of incidence rates)
#'
#' @examples
#' # Load 'cancer' dataset from survival package (Used in all examples)
#' data(cancer, package = "survival")
#'
#' # The exposure (here, 'sex') must be categorical
#' cancer <- cancer |>
#'   tibble::as_tibble() |>
#'   dplyr::mutate(
#'     sex = factor(
#'       sex,
#'       levels = 1:2,
#'       labels = c("Male", "Female")
#'     ),
#'     time = time / 365.25,
#'     status = status - 1
#'   )
#'
#'
#' # Example 1: Binary outcomes (use 'outcome' variable)
#' # Set table design
#' design1 <- tibble::tibble(
#'   label = c(
#'     "Outcomes",
#'     "Total",
#'     "Outcomes/Total",
#'     "Risk",
#'     "Risk (CI)",
#'     "Outcomes (Risk)",
#'     "Outcomes/Total (Risk)",
#'     "RR",
#'     "RD"
#'   )
#' ) |>
#'   dplyr::mutate(
#'     type = label,
#'     exposure = "sex",
#'     outcome = "status"
#'   )
#'
#' # Generate rifttable
#' rifttable(
#'   design = design1,
#'   data = cancer
#' )
#'
#' # Use 'design' as columns (selecting RR and RD only)
#' rifttable(
#'   design = design1 |>
#'     dplyr::filter(label %in% c("RR", "RD")),
#'   data = cancer,
#'   layout = "cols"
#' )
#'
#'
#' # Example 2: Survival outcomes (use 'time' and 'event'),
#' #   with an effect modifier and a confounder
#' # Set table design
#' design2 <- tibble::tribble(
#'   # Elements that vary by row:
#'   ~label,                       ~stratum, ~confounders, ~type,
#'   "**Overall**",                NULL,     "",           "blank",
#'   "  Events",                   NULL,     "",           "events",
#'   "  Person-years",             NULL,     "",           "time",
#'   "  Rate/1000 py (95% CI)",    NULL,     "",           "rate (ci)",
#'   "  Unadjusted HR (95% CI)",   NULL,     "",           "hr",
#'   "  Age-adjusted HR (95% CI)", NULL,     "+ age",      "hr",
#'   "",                           NULL,     "",           "blank",
#'   "**Stratified models**",      NULL,     "",           "",
#'   "*ECOG PS1* (events/N)",      1,        "",           "events/total",
#'   "  Unadjusted",               1,        "",           "hr",
#'   "  Age-adjusted",             1,        "+ age",      "hr",
#'   "*ECOG PS2* (events/N)",      2,        "",           "events/total",
#'   "  Unadjusted",               2,        "",           "hr",
#'   "  Age-adjusted",             2,        "+ age",      "hr",
#'   "",                           NULL,     "",           "",
#'   "**Joint model**, age-adj.",  NULL,     "",           "",
#'   "  ECOG PS1",                 1,        "+ age",      "hr_joint",
#'   "  ECOG PS2",                 2,        "+ age",      "hr_joint"
#' ) |>
#'   # Elements that are the same for all rows:
#'   dplyr::mutate(
#'     exposure = "sex",
#'     event = "status",
#'     time = "time",
#'     effect_modifier = "ph.ecog"
#'   )
#'
#' # Generate rifttable
#' rifttable(
#'   design = design2,
#'   data = cancer |>
#'     dplyr::filter(ph.ecog %in% 1:2)
#' )
#'
#'
#' # Example 3: Get two estimates using 'type' and 'type2'
#' design3 <- tibble::tribble(
#'   ~label,     ~stratum, ~type,          ~type2,
#'   "ECOG PS1", 1,        "events/total", "hr",
#'   "ECOG PS2", 2,        "events/total", "hr"
#' ) |>
#'   dplyr::mutate(
#'     exposure = "sex",
#'     event = "status",
#'     time = "time",
#'     confounders = "+ age",
#'     effect_modifier = "ph.ecog"
#'   )
#'
#' rifttable(
#'   design = design3,
#'   data = cancer |>
#'     dplyr::filter(ph.ecog %in% 1:2)
#' )
#'
#' rifttable(
#'   design = design3,
#'   data = cancer |>
#'     dplyr::filter(ph.ecog %in% 1:2),
#'   layout = "cols",
#'   type2_layout = "cols"
#' )
#'
#'
#' # Example 4: Continuous outcomes (use 'outcome' variable);
#' # request rounding to 1 decimal digit in some cases;
#' # add continuous trend (slope per one unit of the 'trend' variable)
#' tibble::tribble(
#'   ~label,                   ~stratum, ~type,        ~digits,
#'   "Marginal mean (95% CI)", NULL,     "mean (ci)",  1,
#'   "  Male",                 "Male",   "mean",       NA,
#'   "  Female",               "Female", "mean",       NA,
#'   "",                       NULL,     "",           NA,
#'   "Stratified model",       NULL,     "",           NA,
#'   "  Male",                 "Male",   "diff",       1,
#'   "  Female",               "Female", "diff",       1,
#'   "",                       NULL,     "",           NA,
#'   "Joint model",            NULL,     "",           NA,
#'   "  Male",                 "Male",   "diff_joint", NA,
#'   "  Female",               "Female", "diff_joint", NA
#' ) |>
#'   dplyr::mutate(
#'     exposure = "ph.ecog_factor",
#'     trend = "ph.ecog",
#'     outcome = "age",
#'     effect_modifier = "sex"
#'   ) |>
#'   rifttable(
#'     data = cancer |>
#'       dplyr::filter(ph.ecog < 3) |>
#'       dplyr::mutate(ph.ecog_factor = factor(ph.ecog))
#'   )
#'
#'
#' # Example 5: Get formatted output for Example 2
#' rifttable(
#'   design = design2,
#'   data = cancer |>
#'     dplyr::filter(ph.ecog %in% 1:2)
#' ) |>
#'   rt_gt()
#'
rifttable <- function(
    design,
    data,
    id = "",
    layout = "rows",
    factor = 1000,
    risk_percent = FALSE,
    risk_digits = dplyr::if_else(
      risk_percent == TRUE,
      true = 0,
      false = 2
    ),
    diff_digits = 2,
    ratio_digits = 2,
    ratio_digits_decrease = c(`2.995` = -1, `9.95` = -2),
    rate_digits = 1,
    to = ", ",
    reference = "(reference)",
    type2_layout = "rows",
    overall = FALSE,
    exposure_levels = c("noempty", "nona", "all")
) {
  if (!is.data.frame(design)) {
    stop("No 'design' data frame/tibble was provided.")
  }
  if (missing(data)) {
    if (!is.null(attr(x = design, which = "rt_data"))) {
      data <- attr(x = design, which = "rt_data")
    }
  }
  if (!is.data.frame(data)) {
    stop("No 'data' data frame/tibble was provided.")
  }
  if (nrow(data) < 1) {
    stop("The data set is empty.")
  }
  if (missing(risk_percent) &
    !is.null(attr(
      x = design,
      which = "rt_data"
    ))) {
    risk_percent <- TRUE
  }
  exposure_levels <- match.arg(exposure_levels)[1]
  if (!("type" %in% names(design))) {
    stop(paste(
      "The 'design' data frame must contain a 'type' column",
      "specifying the requested statistics."
    ))
  }
  data$.id <- find_id(
    data = data,
    id_variable = id
  )
  if(!("label"       %in% names(design))) design$label       <- design$type
  if(!("exposure"    %in% names(design))) design$exposure    <- NA
  if(!("event"       %in% names(design))) design$event       <- NA
  if(!("time"        %in% names(design))) design$time        <- NA
  if(!("time2"       %in% names(design))) design$time2       <- NA
  if(!("outcome"     %in% names(design))) design$outcome     <- NA
  if(!("trend"       %in% names(design))) design$trend       <- NA
  if(!("confounders" %in% names(design))) design$confounders <- ""
  if(!("type2"       %in% names(design))) design$type2       <- ""
  if(!("digits"      %in% names(design))) design$digits      <- NA
  if(!("digits2"     %in% names(design))) design$digits2     <- NA
  if(!("to"          %in% names(design))) design$to          <- NA
  if(!("nmin"        %in% names(design))) design$nmin        <- NA
  if(!("na_rm"       %in% names(design))) design$na_rm       <- NA
  if(!("ci"          %in% names(design))) design$ci          <- NA
  if(!("arguments"   %in% names(design))) design$arguments   <- NA
  if(!("weights"     %in% names(design))) {
    if("weight" %in% names(design)) {
      design$weights <- design$weight
    } else {
      design$weights     <- NA
    }
  }
  if (
    !("effect_modifier" %in% names(design) &
      "stratum" %in% names(design))
  ) {
    design <- design |>
      dplyr::mutate(effect_modifier = NA, stratum = NA)
  }
  to_use <- to
  design <- design |>
    dplyr::mutate(
      type2 = dplyr::if_else(
        is.na(.data$type2),
        true = "",
        false = .data$type2
      ),
      to = dplyr::if_else(
        is.na(.data$to),
        true = as.character(to_use),
        false = as.character(.data$to)
      )
    )

  exposure_nona <- stats::na.omit(design$exposure)
  exposure_nona <- exposure_nona[exposure_nona != ""]
  if (length(exposure_nona > 0)) {
    name <- attr(
      x = data |>
        dplyr::pull(exposure_nona[1]),
      which = "label"
    )
    if (is.null(name)) {
      name <- exposure_nona[1]
    }

    if (overall == TRUE) {
      if (layout == "rows") {
        return(
          dplyr::bind_cols(
            rifttable(
              design = design |>
                dplyr::select(-"exposure", -"trend"),
              data = data,
              layout = layout,
              factor = factor,
              risk_percent = risk_percent,
              risk_digits = risk_digits,
              diff_digits = diff_digits,
              ratio_digits = ratio_digits,
              ratio_digits_decrease = ratio_digits_decrease,
              rate_digits = rate_digits,
              type2_layout = type2_layout,
              to = to,
              reference = reference,
              overall = FALSE,
              exposure_levels = exposure_levels
            ),
            rifttable(
              design = design,
              data = data,
              layout = layout,
              factor = factor,
              risk_percent = risk_percent,
              risk_digits = risk_digits,
              diff_digits = diff_digits,
              ratio_digits = ratio_digits,
              ratio_digits_decrease = ratio_digits_decrease,
              rate_digits = rate_digits,
              type2_layout = type2_layout,
              to = to,
              reference = reference,
              overall = FALSE,
              exposure_levels = exposure_levels
            ) |>
              dplyr::select(-1, -dplyr::any_of("Overall"))
          )
        )
      } else {
        res_strat <- rifttable(
          design = design,
          data = data,
          layout = layout,
          factor = factor,
          risk_percent = risk_percent,
          risk_digits = risk_digits,
          diff_digits = diff_digits,
          ratio_digits = ratio_digits,
          ratio_digits_decrease = ratio_digits_decrease,
          rate_digits = rate_digits,
          type2_layout = type2_layout,
          to = to,
          reference = reference,
          overall = FALSE,
          exposure_levels = exposure_levels
        )
        return(
          dplyr::bind_rows(
            rifttable(
              design = design |>
                dplyr::select(-"exposure"),
              data = data,
              layout = layout,
              factor = factor,
              risk_percent = risk_percent,
              risk_digits = risk_digits,
              diff_digits = diff_digits,
              ratio_digits = ratio_digits,
              ratio_digits_decrease = ratio_digits_decrease,
              rate_digits = rate_digits,
              type2_layout = type2_layout,
              to = to,
              reference = reference,
              overall = FALSE,
              exposure_levels = exposure_levels
            ) |>
              dplyr::rename(!!names(res_strat)[1] := 1),
            res_strat
          )
        )
      }
    }
  } else {
    name <- "Summary"
  }

  # For Table 1s from table1_design()
  if (any(
    stringr::str_detect(string = design$outcome, "@"),
    na.rm = TRUE
  )) {
    to_code <- tibble::tibble(
      combo = stringr::str_subset(design$outcome, "@")
    ) |>
      tidyr::separate(
        col = "combo",
        into = c("outcome", "var_level"),
        sep = "@",
        extra = "merge"
      ) |>
      dplyr::distinct()
    data <- purrr::map2_dfc(
      .x = to_code$outcome,
      .y = to_code$var_level,
      .f = \(x, y) {
        varname <- paste0(x, "@", y)
        if (y == "_NA_") {
          data |>
            dplyr::rename(variable = {{ x }}) |>
            dplyr::mutate(result = is.na(.data$variable)) |>
            dplyr::select(!!varname := "result")
        } else {
          data |>
            dplyr::rename(variable = {{ x }}) |>
            dplyr::mutate(result = .data$variable == y) |>
            dplyr::select(!!varname := "result")
        }
      }
    ) |>
      dplyr::bind_cols(data)
  }

  res <- design |>
    dplyr::mutate(
      type = stringr::str_to_lower(string = .data$type),
      type2 = stringr::str_to_lower(string = .data$type2),
      index = dplyr::row_number(),
      result = purrr::pmap(
        .l = list(
          .data$event,
          .data$time,
          .data$time2,
          .data$outcome,
          .data$exposure,
          .data$effect_modifier,
          .data$stratum,
          .data$confounders,
          .data$weights,
          .data$type,
          .data$trend,
          .data$digits,
          .data$nmin,
          .data$na_rm,
          .data$ci,
          .data$to,
          .data$arguments
        ),
        .f = fill_cells,
        data = data,
        factor = factor,
        risk_percent = risk_percent,
        risk_digits = risk_digits,
        diff_digits = diff_digits,
        ratio_digits = ratio_digits,
        ratio_digits_decrease = ratio_digits_decrease,
        rate_digits = rate_digits,
        reference = reference,
        exposure_levels = exposure_levels
      )
    )

  # simple reshaping if only "type" alone
  if (all(design$type2 == "")) {
    res <- res |>
      dplyr::select(
        "index",
        "label",
        "result"
      ) |>
      tidyr::unnest(
        cols = "result",
        keep_empty = TRUE
      )
    if (".exposure" %in% names(res)) {
      res <- res |>
        dplyr::mutate(
          .exposure = dplyr::if_else(
            is.na(.data$.exposure) & .data$res == "",
            true = stats::na.omit(.data$.exposure)[1],
            false = .data$.exposure
          )
        )
    } else {
      res$.exposure <- "Overall"
    }
    if (layout == "rows") {
      res <- res |>
        tidyr::pivot_wider(
          names_from = ".exposure",
          values_from = "res",
          values_fill = ""
        ) |>
        dplyr::rename(!!name := "label") |>
        dplyr::select(-"index") |>
        dplyr::relocate(
          dplyr::any_of("Trend"),
          .after = dplyr::last_col()
        )
      return(res)
    } else {
      if (sum(duplicated(design$label)) > 0 |
        "" %in% design$label) {
        res |>
          tidyr::pivot_wider(
            names_from = c("index", "label"),
            values_from = "res",
            values_fill = ""
          )
      } else {
        res |>
          dplyr::select(-"index") |>
          tidyr::pivot_wider(
            names_from = "label",
            values_from = "res",
            values_fill = ""
          ) |>
          dplyr::rename(!!name := ".exposure")
      }
    }

    # handle "type" and "type2" together
  } else {
    if (any(is.na(design$exposure))) {
      stop("If using 'type2', 'exposure' must be specified for each row of the 'design', and 'overall = TRUE' cannot be used.")
    }
    res <- res |>
      dplyr::mutate(
        result2 = purrr::pmap(
          .l = list(
            .data$event,
            .data$time,
            .data$time2,
            .data$outcome,
            .data$exposure,
            .data$effect_modifier,
            .data$stratum,
            .data$confounders,
            .data$weights,
            .data$type2, # !
            .data$trend,
            .data$digits2, # !
            .data$nmin,
            .data$na_rm,
            .data$ci,
            .data$to,
            .data$arguments
          ),
          .f = fill_cells,
          data = data,
          factor = factor,
          risk_percent = risk_percent,
          risk_digits = risk_digits,
          diff_digits = diff_digits,
          ratio_digits = ratio_digits,
          ratio_digits_decrease = ratio_digits_decrease,
          rate_digits = rate_digits,
          reference = reference,
          exposure_levels = exposure_levels
        ),
        result = purrr::map2(
          .x = .data$result,
          .y = .data$result2,
          .f = \(x, y) dplyr::full_join(
            x,
            y,
            by = ".exposure",
            suffix = c(".1", ".2")
          )
        )
      ) |>
      dplyr::select(
        "index",
        "label",
        "result"
      ) |>
      tidyr::unnest(cols = "result") |>
      tidyr::pivot_longer(
        cols = c("res.1", "res.2"),
        names_to = "whichres",
        values_to = "value"
      ) |>
      dplyr::mutate(
        value = dplyr::if_else(
          is.na(.data$value),
          true = "",
          false = .data$value
        )
      )
    if (layout == "rows") {
      if (type2_layout == "rows") {
        res <- res |>
          dplyr::filter(
            !(.data$whichres == "res.2" &
              .data$value == "")
          ) |>
          tidyr::pivot_wider(
            names_from = ".exposure",
            values_from = "value",
            values_fill = ""
          ) |>
          dplyr::group_by(.data$index) |>
          dplyr::mutate(
            label = dplyr::if_else(
              dplyr::row_number() == 1,
              true = .data$label,
              false = ""
            )
          ) |>
          dplyr::ungroup() |>
          dplyr::rename(!!name := "label") |>
          dplyr::select(-"index", -"whichres") |>
          dplyr::relocate(
            dplyr::any_of("Trend"),
            .after = dplyr::last_col()
          )
      } else {
        res <- res |>
          dplyr::mutate(
            .exposure = dplyr::if_else(
              .data$whichres == "res.1",
              true = paste0(.data$.exposure),
              false = paste0(
                .data$.exposure,
                " "
              )
            )
          ) |>
          dplyr::select(-"whichres") |>
          tidyr::pivot_wider(
            names_from = ".exposure",
            values_from = "value",
            values_fill = ""
          ) |>
          dplyr::rename(!!name := "label") |>
          dplyr::select(-"index")
      }
      return(res)
    } else {
      if (type2_layout == "rows") {
        if (sum(duplicated(design$label)) > 0 | "" %in% design$label) {
          res <- res |>
            tidyr::pivot_wider(
              names_from = c("index", "label"),
              values_from = "value",
              values_fill = ""
            )
        } else {
          res <- res |>
            dplyr::select(-"index") |>
            tidyr::pivot_wider(
              names_from = "label",
              values_from = "value",
              values_fill = ""
            )
        }
        res |>
          dplyr::group_by(.data$.exposure) |>
          dplyr::mutate(
            .exposure = dplyr::if_else(
              .data$whichres == "res.1",
              true = paste0(.data$.exposure),
              false = ""
            )
          ) |>
          dplyr::ungroup() |>
          dplyr::select(-"whichres") |>
          dplyr::rename(!!name := ".exposure")
      } else {
        res <- res |>
          dplyr::filter(
            !(.data$whichres == "res.2" &
              .data$value == "")
          ) |>
          dplyr::mutate(
            label = dplyr::if_else(
              .data$whichres == "res.1",
              true = .data$label,
              false = paste0(
                .data$label,
                " "
              )
            )
          )
        if (sum(duplicated(design$label)) > 0 | "" %in% design$label) {
          res |>
            dplyr::select(-"whichres") |>
            tidyr::pivot_wider(
              names_from = c("index", "label"),
              values_from = "value",
              values_fill = ""
            ) |>
            dplyr::rename(!!name := ".exposure")
        } else {
          res |>
            dplyr::select(-"whichres", -"index") |>
            tidyr::pivot_wider(
              names_from = "label",
              values_from = "value",
              values_fill = ""
            ) |>
            dplyr::rename(!!name := ".exposure")
        }
      }
    }
  }
}
