% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ee_imagecollection.R
\name{ee_imagecollection_to_local}
\alias{ee_imagecollection_to_local}
\title{Save an EE ImageCollection to the local system.}
\usage{
ee_imagecollection_to_local(
  ic,
  region,
  dsn = NULL,
  via = "drive",
  container = "rgee_backup",
  scale = NULL,
  maxPixels = 1e+09,
  lazy = FALSE,
  public = TRUE,
  add_metadata = TRUE,
  timePrefix = TRUE,
  quiet = FALSE,
  ...
)
}
\arguments{
\item{ic}{ee$ImageCollection to be saved to the system.}

\item{region}{EE Geometry (ee$Geometry$Polygon). The
CRS needs to be the same that the \code{ic} argument. Otherwise, it will be
forced.}

\item{dsn}{Character. Output filename. If missing, a temporary file will
be created for each image.}

\item{via}{Character. Method to export the image. Two methods are available:
"drive", "gcs". See details.}

\item{container}{Character. Name of the folder ('drive') or bucket ('gcs')
to be exported into (ignored if \code{via} is not defined as "drive" or
"gcs").}

\item{scale}{Numeric. The resolution in meters per pixel. Defaults
to the native resolution of the image.}

\item{maxPixels}{Numeric. The maximum allowable number of pixels in the exported image.
If the exported region covers more pixels than the specified limit in the given projection,
the task will fail. Defaults to 100,000,000.}

\item{lazy}{Logical. If TRUE, a \code{\link[future:sequential]{
future::sequential}} object is created to evaluate the task in the future.
See details.}

\item{public}{Logical. If TRUE, a public link to the image is created.}

\item{add_metadata}{Add metadata to the stars_proxy object. See details.}

\item{timePrefix}{Logical. Add current date and time (\code{Sys.time()}) as
a prefix to export files. This parameter helps to avoid exporting files
with the same name. By default TRUE.}

\item{quiet}{Logical. Suppress info message}

\item{...}{Extra exporting argument. See \link{ee_image_to_drive} and}
}
\value{
If add_metadata is FALSE, \code{ee_imagecollection_to_local} will
return a character vector containing the filename of the images downloaded.
Otherwise, if add_metadata is TRUE, will return a list with extra information
related to the exportation (see details).
}
\description{
Save an EE ImageCollection to the local system.
}
\details{
\code{ee_imagecollection_to_local} supports the download of \code{ee$Images}
using two different options: "drive"
(\href{https://CRAN.R-project.org/package=googledrive}{Google Drive}) and "gcs"
(\href{https://CRAN.R-project.org/package=googleCloudStorageR}{
Google Cloud Storage}). In both cases, \code{ee_imagecollection_to_local}
works as follow:
\describe{
\item{1.}{A task  is initiate (i.e., \code{ee$batch$Task$start()}) to
transfer the \code{ee$Image} from Earth Engine to the intermediate container
specified in the argument \code{via}.}
\item{2.}{If the argument \code{lazy} is TRUE, the task will not be
monitored. This is useful to lunch several tasks simultaneously and
calls them later using \code{\link{ee_utils_future_value}} or
\code{\link[future:value]{future::value}}. At the end of this step,
the \code{ee$Images} are stored on the path specified in the argument
\code{dsn}.}
\item{3.}{Finally, if the \code{add_metadata}  argument is set to TRUE,
a list containing the following elements will be appended to the \code{dsn} argument.
\describe{
\item{\bold{if via is "drive":}}{
\describe{
\item{\bold{ee_id: }}{Name of the Earth Engine task.}
\item{\bold{drive_name:}}{Name of the Image in Google Drive.}
\item{\bold{drive_id:}}{Id of the Image in Google Drive.}
\item{\bold{drive_download_link:}}{Download link to the image.}
}}
}
\describe{
\item{\bold{if via is "gcs":}}{
\describe{
\item{\bold{ee_id:}}{Name of the Earth Engine task.}
\item{\bold{gcs_name:}}{Name of the Image in Google Cloud Storage.}
\item{\bold{gcs_bucket:}}{Name of the bucket.}
\item{\bold{gcs_fileFormat:}}{Format of the image.}
\item{\bold{gcs_public_link:}}{Download link to the image.}
\item{\bold{gcs_URI:}}{gs:// link to the image.}
}}
}
}
}

For getting more information about exporting data from Earth Engine, take
a look at the
\href{https://developers.google.com/earth-engine/guides/exporting}{Google
Earth Engine Guide - Export data}.
}
\examples{
\dontrun{
library(rgee)
library(raster)
ee_Initialize(drive = TRUE, gcs = TRUE)

# USDA example
loc <- ee$Geometry$Point(-99.2222, 46.7816)
collection <- ee$ImageCollection('USDA/NAIP/DOQQ')$
  filterBounds(loc)$
  filterDate('2008-01-01', '2020-01-01')$
  filter(ee$Filter$listContains("system:band_names", "N"))

# From ImageCollection to local directory
ee_crs <- collection$first()$projection()$getInfo()$crs
geometry <- collection$first()$geometry(proj = ee_crs)$bounds()
tmp <- tempdir()

## Using drive
# one by once
ic_drive_files_1 <- ee_imagecollection_to_local(
  ic = collection,
  region = geometry,
  scale = 250,
  dsn = file.path(tmp, "drive_")
)

# all at once
ic_drive_files_2 <- ee_imagecollection_to_local(
  ic = collection,
  region = geometry,
  scale = 250,
  lazy = TRUE,
  dsn = file.path(tmp, "drive_")
)

# From Google Drive to client-side
doqq_dsn <- ic_drive_files_2 \%>\% ee_utils_future_value()
sapply(doqq_dsn, '[[', 1)
}
}
\seealso{
Other image download functions: 
\code{\link{ee_as_rast}()},
\code{\link{ee_as_raster}()},
\code{\link{ee_as_stars}()},
\code{\link{ee_as_thumbnail}()}
}
\concept{image download functions}
