% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/interpolate_spherical.R
\name{interpolate_spherical}
\alias{interpolate_spherical}
\title{Interpolate in spherical space}
\usage{
interpolate_spherical(
  sky_points,
  z,
  a,
  filling_source = NULL,
  w = 1,
  k = 3,
  p = 2,
  angular_radius = 20,
  rule = "any",
  size = 50
)
}
\arguments{
\item{sky_points}{\code{data.frame} returned by \code{\link[=extract_rr]{extract_rr()}}. If generated by
other means, it must contain columns \code{row}, \code{col}, \code{z}, \code{a}, and \code{rr},
where the first four define geometry (degrees) and \code{rr} is the value to
interpolate.}

\item{z}{\link[terra:SpatRaster-class]{terra::SpatRaster} generated with \code{\link[=zenith_image]{zenith_image()}}.}

\item{a}{\link[terra:SpatRaster-class]{terra::SpatRaster} generated with \code{\link[=azimuth_image]{azimuth_image()}}.}

\item{filling_source}{optional numeric \link[terra:SpatRaster-class]{terra::SpatRaster} with one
layer. Surface used to complement \code{rr} when neighbors are insufficient
(e.g., output of \code{\link[=fit_cie_model]{fit_cie_model()}}). If \code{NULL} (default), no filling is
applied.}

\item{w}{numeric vector of length one. Weight assigned to \code{filling_source}
in the blend with local estimates (see Eq. 6 in
\insertCite{Lang2010;textual}{rcaiman}).}

\item{k}{numeric vector of length one. Number of neighbors.}

\item{p}{numeric vector of length one. Inverse distance weighting exponent.}

\item{angular_radius}{numeric vector of length one. The maximum radius for
searching k-nearest neighbors (KNN) in degrees.}

\item{rule}{character vector of length one. Either \code{"any"} or \code{"all"}.
With \code{"any"}, pixels within \code{angular_radius} of at least one sample are
interpolated. With \code{"all"}, pixels are interpolated only if the \code{k}
nearest neighbors lie within \code{angular_radius}. If \code{k = 1}, both are
equivalent.}

\item{size}{numeric vector of length one. Number of rows and columns of the
low-resolution grid used before resampling to full resolution.}
}
\value{
Numeric \link[terra:SpatRaster-class]{terra::SpatRaster} with one layer of interpolated
values and the geometry of \code{z}.
}
\description{
Interpolate values from canopy photographs using inverse distance weighting
(IDW) with k-nearest neighbors, computing distances in spherical coordinates
that map the sky vault. Optionally blend with a model surface to fill voids.
}
\details{
Distances are great-circle distances on the sky vault. When \code{filling_source} is
provided, local IDW estimates are blended with that surface following Eq. 6
in \insertCite{Lang2010;textual}{rcaiman}. For efficiency, interpolation runs
on a temporary low-resolution grid of size \code{size}.
}
\note{
This function assumes that \code{sky_points} and the
\link[terra:SpatRaster-class]{terra::SpatRaster} inputs are spatially aligned and share the same
geometry. No checks are performed to enforce this.
}
\examples{
\dontrun{
caim <- read_caim()
z <- zenith_image(ncol(caim), lens())
a <- azimuth_image(z)

# Manual method following Lang et al. (2010)
path <- system.file("external/sky_points.csv",
                    package = "rcaiman")
sky_points <- read.csv(path)
sky_points <- sky_points[c("Y", "X")]
colnames(sky_points) <- c("row", "col")
head(sky_points)
plot(caim$Blue)
points(sky_points$col, nrow(caim) - sky_points$row, col = 2, pch = 10)

# x11()
# plot(caim$Blue)
# sun_angles <- click(c(z, a), 1) \%>\% as.numeric()
sun_angles <- c(z = 49.5, a = 27.4) #taken with above lines then hardcoded

sun_row_col <- row_col_from_zenith_azimuth(z, a,
                                           sun_angles["z"],
                                           sun_angles["a"])
points(sun_row_col[2], nrow(caim) - sun_row_col[1],
       col = "yellow", pch = 8, cex = 3)

rr <- extract_rr(caim$Blue, z, a, sky_points)

set.seed(7)
model <- fit_cie_model(rr, sun_angles,
                       general_sky_type = "Clear")

sky_cie <- cie_image(z, a, model$sun_angles, model$coef)

sky_rr <- interpolate_spherical(rr$sky_points, z, a,
                                filling_source = sky_cie,
                                w = 1,
                                k = 10,
                                p = 2,
                                angular_radius = 20,
                                rule = "any",
                                size = 50)
plot(r/sky_rr/rr$zenith_dn)
}
}
\references{
\insertAllCited{}
}
