% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/apply_by_direction.R
\name{apply_by_direction}
\alias{apply_by_direction}
\title{Apply a method by direction using a constant field of view}
\usage{
apply_by_direction(
  r,
  z,
  a,
  m,
  spacing = 10,
  laxity = 2.5,
  fov = c(30, 40, 50),
  method = c("thr_isodata", "detect_bg_dn", "fit_coneshaped_model",
    "fit_trend_surface_np1", "fit_trend_surface_np6"),
  fun = NULL,
  parallel = FALSE
)
}
\arguments{
\item{r}{\link[terra:SpatRaster-class]{terra::SpatRaster} of one or more layers (e.g., RGB channels or
binary masks) in fisheye projection.}

\item{z}{\link[terra:SpatRaster-class]{terra::SpatRaster} generated with \code{\link[=zenith_image]{zenith_image()}}.}

\item{a}{\link[terra:SpatRaster-class]{terra::SpatRaster} generated with \code{\link[=azimuth_image]{azimuth_image()}}.}

\item{m}{logical \link[terra:SpatRaster-class]{terra::SpatRaster} with one layer. A binary mask with
\code{TRUE} for selected pixels.}

\item{spacing}{numeric vector of length one. Angular spacing (in degrees)
between directions to process.}

\item{laxity}{numeric vector of length one.}

\item{fov}{numeric vector. Field of view in degrees. If more than one value
is provided, they are tried in order when a method fails.}

\item{method}{character vector of length one. Built-in method to apply.
Available options are \code{"thr_isodata"}, \code{"detect_bg_dn"},
\code{"fit_coneshaped_model"}, \code{"fit_trend_surface_np1"}, and
\code{"fit_trend_surface_np6"}. Ignored if \code{fun} is provided.}

\item{fun}{\code{NULL} (default) or a function accepting \code{r}, \code{z}, \code{a}, and \code{m} as
input and returning a single-layer \link[terra:SpatRaster-class]{terra::SpatRaster} object  with
the same number of rows and columns as its first input, \code{r}.}

\item{parallel}{logical vector of length one. If \code{TRUE}, operations are
executed in parallel.}
}
\value{
\link[terra:SpatRaster-class]{terra::SpatRaster} object with two layers: \code{"dn"} for digital
number values and \code{"n"} for the number of valid pixels used in each
directional estimate.
}
\description{
Applies a method to each set of pixels defined by a direction and a constant
field of view (FOV). By default, several built-in methods are available
(see \code{method}), but a custom function can also be provided via the \code{fun}
argument.
}
\note{
This function is part of a manuscript currently under preparation.
}
\examples{
\dontrun{
caim <- read_caim()
r <- caim$Blue
z <- zenith_image(ncol(caim), lens())
a <- azimuth_image(z)
m <- !is.na(z)

# Automatic sky brightness estimation
sky <- apply_by_direction(r, z, a, m, spacing = 10, fov = c(30, 60),
                          method = "detect_bg_dn", parallel = TRUE)
plot(sky$dn)
plot(r / sky$dn)

# Using cone-shaped model
sky_cs <- apply_by_direction(caim, z, a, m, spacing = 15, fov = 60,
                             method = "fit_coneshaped_model", parallel = TRUE)
plot(sky_cs$dn)

# Using trend surface model
sky_s <- apply_by_direction(caim, z, a, m, spacing = 15, fov = 60,
                            method = "fit_trend_surface_np1", parallel = TRUE)
plot(sky_s$dn)

# Using a custom thresholding function
thr <- apply_by_direction(r, z, a, m, 15, fov = c(30, 40, 50),
  fun = function(r, z, a, m) {
    thr <- tryCatch(thr_twocorner(r[m])$tm, error = function(e) NA)
    r[] <- thr
    r
  },
  parallel = TRUE
)
plot(thr$dn)
plot(binarize_with_thr(r, thr$dn))
}
}
\references{
\insertAllCited{}
}
