## ----setup, include = FALSE---------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>",
  warning = FALSE,
  message = FALSE
)

## ----libraries, message = FALSE, warning = FALSE------------------------------
library(rbmi)
library(rbmiUtils)
library(dplyr)

## ----load-data----------------------------------------------------------------
data("ADEFF", package = "rbmiUtils")

## ----inspect-data-------------------------------------------------------------
str(ADEFF)

## ----factor-prep--------------------------------------------------------------
ADEFF <- ADEFF |>
  mutate(
    TRT = factor(TRT01P, levels = c("Placebo", "Drug A")),
    USUBJID = factor(USUBJID),
    AVISIT = factor(AVISIT, levels = c("Week 24", "Week 48"))
  )

## ----define-vars--------------------------------------------------------------
vars <- set_vars(
  subjid = "USUBJID",
  visit = "AVISIT",
  group = "TRT",
  outcome = "CHG",
  covariates = c("BASE", "STRATA", "REGION")
)

## ----validate-----------------------------------------------------------------
validate_data(ADEFF, vars)

## ----missingness--------------------------------------------------------------
miss <- summarise_missingness(ADEFF, vars)
print(miss$summary)

## ----method-------------------------------------------------------------------
set.seed(1974)

method <- method_bayes(
  n_samples = 100,
  control = control_bayes(warmup = 200, thin = 2)
)

## ----draws, message = FALSE, warning = FALSE----------------------------------
dat <- ADEFF |>
  select(USUBJID, STRATA, REGION, TRT, BASE, CHG, AVISIT)

draws_obj <- draws(data = dat, vars = vars, method = method)

## ----impute-------------------------------------------------------------------
impute_obj <- impute(
  draws_obj,
  references = c("Placebo" = "Placebo", "Drug A" = "Placebo")
)

## ----get-imputed--------------------------------------------------------------
ADMI <- get_imputed_data(impute_obj)

## ----analyse------------------------------------------------------------------
ana_obj <- analyse_mi_data(
  data = ADMI,
  vars = vars,
  method = method,
  fun = ancova
)

## ----pool---------------------------------------------------------------------
pool_obj <- pool(ana_obj)
print(pool_obj)

## ----tidy---------------------------------------------------------------------
tidy_df <- tidy_pool_obj(pool_obj)
print(tidy_df)

## ----efficacy-table-default, eval = requireNamespace("gt", quietly = TRUE)----
tbl <- efficacy_table(pool_obj)
tbl

## ----efficacy-table-custom, eval = requireNamespace("gt", quietly = TRUE)-----
tbl_custom <- efficacy_table(
  pool_obj,
  title = "Table 14.2.1: ANCOVA of Change from Baseline",
  subtitle = "Reference-Based Multiple Imputation (Jump to Reference)",
  arm_labels = c(ref = "Placebo", alt = "Drug A")
)
tbl_custom

## ----forest-trt, fig.width = 10, fig.height = 3, eval = requireNamespace("ggplot2", quietly = TRUE) && requireNamespace("patchwork", quietly = TRUE)----
p <- plot_forest(
  pool_obj,
  title = "Treatment Effect: Change from Baseline (Drug A vs Placebo)"
)
p

## ----forest-lsm, fig.width = 10, fig.height = 4, eval = requireNamespace("ggplot2", quietly = TRUE) && requireNamespace("patchwork", quietly = TRUE)----
p_lsm <- plot_forest(
  pool_obj,
  display = "lsm",
  arm_labels = c(ref = "Placebo", alt = "Drug A"),
  title = "LS Mean Estimates by Visit"
)
p_lsm

