% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/add_manual_locked_constraints.R
\name{add_manual_locked_constraints}
\alias{add_manual_locked_constraints}
\alias{add_manual_locked_constraints,ConservationProblem,data.frame-method}
\alias{add_manual_locked_constraints,ConservationProblem,tbl_df-method}
\title{Add manually specified locked constraints}
\usage{
add_manual_locked_constraints(x, data)

\S4method{add_manual_locked_constraints}{ConservationProblem,data.frame}(x, data)

\S4method{add_manual_locked_constraints}{ConservationProblem,tbl_df}(x, data)
}
\arguments{
\item{x}{\code{\link[=problem]{problem()}} object.}

\item{data}{\code{data.frame} or \code{\link[tibble:tibble]{tibble::tibble()}} object.
See the Data format section for more information.}
}
\value{
An updated \code{\link[=problem]{problem()}} object with the constraints added to it.
}
\description{
Add constraints to a conservation planning problem to ensure
that solutions allocate (or do not allocate) specific planning units to
specific management zones. This function offers more fine-grained control
than the \code{\link[=add_locked_in_constraints]{add_locked_in_constraints()}} and
\code{\link[=add_locked_out_constraints]{add_locked_out_constraints()}} functions.
}
\section{Data format}{


The argument to \code{data} should be a \code{data.frame} with the following columns:

\describe{

\item{pu}{\code{integer} planning unit identifiers.
If \code{x} has \code{data.frame} planning units,
then these values must refer to values in the \code{id} column of the planning
unit data.
Alternatively, if \code{x} has \code{\link[sf:sf]{sf::st_sf()}} or \code{matrix} planning units,
then these values must refer to the row numbers of the planning unit data.
Additionally, if \code{x} has \code{numeric} vector planning units,
then these values must refer to the element indices of the planning unit
data.
Finally, if \code{x} has \code{\link[terra:rast]{terra::rast()}} planning units,
then these values must refer to cell indices.}

\item{zone}{\code{character} names of zones. Note that this
argument is optional for arguments to \code{x} that contain a single
zone.}

\item{status}{\code{numeric} status values. These values indicate how much
of each planning unit should be allocated to each zone in the solution.
For example, the \code{numeric} values could be binary values (i.e., zero
or one) for problems containing binary-type decision variables
(using the \code{\link[=add_binary_decisions]{add_binary_decisions()}} function). Alternatively,
the \code{numeric} values could be proportions (e.g., 0.5) for problems
containing proportion-type decision variables (using the
\code{\link[=add_proportion_decisions]{add_proportion_decisions()}}).}

}
}

\examples{
\dontrun{
# set seed for reproducibility
set.seed(500)

# load data
sim_pu_polygons <- get_sim_pu_polygons()
sim_features <- get_sim_features()
sim_zones_pu_polygons <- get_sim_zones_pu_polygons()
sim_zones_features <- get_sim_zones_features()

# create minimal problem
p1 <-
  problem(sim_pu_polygons, sim_features, "cost") \%>\%
  add_min_set_objective() \%>\%
  add_relative_targets(0.2) \%>\%
  add_binary_decisions() \%>\%
  add_default_solver(verbose = FALSE)

# create problem with locked in constraints using add_locked_constraints
p2 <- p1 \%>\% add_locked_in_constraints("locked_in")

# create identical problem using add_manual_locked_constraints
locked_data <- data.frame(
  pu = which(sim_pu_polygons$locked_in),
  status = 1
)

p3 <- p1 \%>\% add_manual_locked_constraints(locked_data)

# solve problems
s1 <- solve(p1)
s2 <- solve(p2)
s3 <- solve(p3)

# create object with all solutions
s4 <- sf::st_sf(
  tibble::tibble(
    s1 = s1$solution_1,
    s2 = s2$solution_1,
    s3 = s3$solution_1
  ),
  geometry = sf::st_geometry(s1)
)

# plot solutions
## s1 = none locked in
## s2 = locked in constraints
## s3 = manual locked constraints
plot(s4)

# create minimal problem with multiple zones
p5 <-
  problem(
    sim_zones_pu_polygons, sim_zones_features,
    c("cost_1", "cost_2", "cost_3")
  ) \%>\%
  add_min_set_objective() \%>\%
  add_relative_targets(matrix(runif(15, 0.1, 0.2), nrow = 5, ncol = 3)) \%>\%
  add_binary_decisions() \%>\%
  add_default_solver(verbose = FALSE)

# create data.frame with the following constraints:
# planning units 1, 2, and 3 must be allocated to zone 1 in the solution
# planning units 4, and 5 must be allocated to zone 2 in the solution
# planning units 8 and 9 must not be allocated to zone 3 in the solution
locked_data2 <- data.frame(
  pu = c(1, 2, 3, 4, 5, 8, 9),
  zone = c(rep("zone_1", 3), rep("zone_2", 2),rep("zone_3", 2)),
  status = c(rep(1, 5), rep(0, 2))
)

# print locked constraint data
print(locked_data2)

# create problem with added constraints
p6 <- p5 \%>\% add_manual_locked_constraints(locked_data2)

# solve problem
s5 <- solve(p5)
s6 <- solve(p6)

# create two new columns representing the zone id that each planning unit
# was allocated to in the two solutions
s5$solution <- category_vector(sf::st_drop_geometry(
  s5[, c("solution_1_zone_1", "solution_1_zone_2", "solution_1_zone_3")]
))
s5$solution <- factor(s5$solution)

s5$solution_locked <- category_vector(sf::st_drop_geometry(
  s6[, c("solution_1_zone_1", "solution_1_zone_2", "solution_1_zone_3")]
))
s5$solution_locked <- factor(s5$solution_locked)

# plot solutions
plot(s5[, c("solution", "solution_locked")], axes = FALSE)
}

}
\seealso{
See \link{constraints} for an overview of all functions for adding constraints.

Other functions for adding constraints:
\code{\link{add_contiguity_constraints}()},
\code{\link{add_feature_contiguity_constraints}()},
\code{\link{add_linear_constraints}()},
\code{\link{add_locked_in_constraints}()},
\code{\link{add_locked_out_constraints}()},
\code{\link{add_mandatory_allocation_constraints}()},
\code{\link{add_manual_bounded_constraints}()},
\code{\link{add_neighbor_constraints}()}
}
\concept{constraints}
