% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/PWD_RL.r
\name{PWD_RL}
\alias{PWD_RL}
\title{Weighted Deming -- Rocke-Lorenzato - known sigma, kappa}
\usage{
PWD_RL(X, Y, sigma, kappa, lambda=1, alpha=NA, beta=NA, mu=NA, epsilon=1e-8)
}
\arguments{
\item{X}{the vector of predicate readings.}

\item{Y}{the vector of test readings.}

\item{sigma}{the RL \eqn{\sigma} parameter.}

\item{kappa}{the RL \eqn{\kappa} parameter.}

\item{lambda}{\emph{optional} (default of 1) - the ratio of the \code{X} to
the \code{Y} precision profile.}

\item{alpha}{\emph{optional} (default of NA) - numeric, single value, initial estimate of \eqn{\alpha}.}

\item{beta}{\emph{optional} (default of NA) - numeric, single value, initial estimate of \eqn{\beta}.}

\item{mu}{\emph{optional} (default of NA) - numeric, vector of length of \code{X}, initial estimate of \eqn{\mu}.}

\item{epsilon}{\emph{optional}  (default of 1e-8) - convergence tolerance limit.}
}
\value{
A list containing the following components:

\item{alpha }{the fitted intercept}
\item{beta }{the fitted slope}
\item{fity }{the vector of predicted Y}
\item{mu }{the vector of estimated latent true values}
\item{resi }{the vector of residuals}
\item{L }{the -2 log likelihood L}
\item{innr }{the number of inner refinement loops executed}
\item{error }{an error code if the iteration fails}
}
\description{
This code fits the weighted Deming regression on
predicate readings (X) and test readings (Y),
with user-supplied Rocke-Lorenzato ("RL") parameters
sigma (\eqn{\sigma}) and kappa (\eqn{\kappa}).
}
\details{
The Rocke-Lorenzato precision profile model assumes the following
forms for the variances, with proportionality constant \eqn{\lambda}:
\itemize{
\item predicate precision profile model: \eqn{g_i = var(X_i) = \lambda\left(\sigma^2 + \left[\kappa\cdot \mu_i\right]^2\right)} and
\item test precision profile model: \eqn{h_i = var(Y_i) = \sigma^2 + \left[\kappa\cdot (\alpha + \beta\mu_i)\right]^2}.
}

The algorithm uses maximum likelihood estimation.  Proportionality constant
\eqn{\lambda} is assumed to be known or estimated externally.
}
\examples{
# library
library(ppwdeming)

# parameter specifications
sigma <- 1
kappa <- 0.08
alpha <- 1
beta  <- 1.1
true  <- 8*10^((0:99)/99)
truey <- alpha+beta*true
# simulate single sample - set seed for reproducibility
set.seed(1039)
# specifications for predicate method
X     <- sigma*rnorm(100)+true *(1+kappa*rnorm(100))
# specifications for test method
Y     <- sigma*rnorm(100)+truey*(1+kappa*rnorm(100))

# fit RL with given sigma and kappa
RL_results <- PWD_RL(X,Y,sigma,kappa)
cat("\nWith given sigma and kappa, the estimated intercept is",
    signif(RL_results$alpha,4), "and the estimated slope is",
    signif(RL_results$beta,4), "\n")

}
\references{
Hawkins DM and Kraker JJ (in press). Precision Profile Weighted
Deming Regression for Methods Comparison. \emph{The Journal of Applied Laboratory Medicine}.
\url{doi:10.1093/jalm/jfaf183}

Hawkins DM (2014). A Model for Assay Precision.
\emph{Statistics in Biopharmaceutical Research}, \strong{6}, 263-269.
\url{doi:10.1080/19466315.2014.899511}
}
\author{
Douglas M. Hawkins, Jessica J. Kraker \href{mailto:krakerjj@uwec.edu}{krakerjj@uwec.edu}
}
