test_that("calculate_restricted_bounds works with real example data", {
  # Use smaller subset for computational efficiency
  data(estimates_constant)
  data(var_constant)

  # Use first 10 observations for faster testing
  n_test <- min(7, length(estimates_constant))
  estimates_subset <- estimates_constant[1:n_test]
  var_subset <- var_constant[1:n_test, 1:n_test]
  
  result <- calculate_restricted_bounds(estimates_subset, var_subset)
  
  expect_s3_class(result, "restricted_bounds")
  expect_s3_class(result, "plausible_bounds_result")
  expect_type(result, "list")
    
  # Check restricted bounds data frame
  expect_s3_class(result$restricted_bounds, "data.frame")
  expect_equal(nrow(result$restricted_bounds), n_test)
  expect_named(result$restricted_bounds, c("horizon", "unrestr_est", "restr_est", "lower", "upper"))
  expect_equal(result$restricted_bounds$unrestr_est, estimates_subset)
  
  # Check that bounds are properly ordered
  expect_true(all(result$restricted_bounds$lower <= result$restricted_bounds$upper))
  
  # Check metadata
  expect_type(result$metadata, "list")
  expect_equal(result$metadata$alpha, 0.05)

  # Compute width from bounds
  width <- mean(result$restricted_bounds$upper - result$restricted_bounds$lower)
  expect_true(is.numeric(width))
  expect_true(width > 0)
})

test_that("calculate_restricted_bounds works with bighump data", {
  # Test with smaller subset of bighump data
  data(estimates_bighump)
  data(var_bighump)

  # Use first 8 observations for faster testing
  n_test <- min(8, length(estimates_bighump))
  estimates_subset <- estimates_bighump[1:n_test]
  var_subset <- var_bighump[1:n_test, 1:n_test]
  
  result <- calculate_restricted_bounds(estimates_subset, var_subset)
  
  expect_s3_class(result, "restricted_bounds")
  expect_equal(nrow(result$restricted_bounds), n_test)
  
  # Check that restr_est is smooth (restricted)
  restr_est_diff <- diff(result$restricted_bounds$restr_est)
  # Restricted estimate should be relatively smooth compared to original estimates
  estimates_diff <- diff(estimates_subset)
  expect_true(var(restr_est_diff) <= var(estimates_diff))

  # Bounds should contain the restr_est
  expect_true(all(result$restricted_bounds$lower <= result$restricted_bounds$restr_est))
  expect_true(all(result$restricted_bounds$restr_est <= result$restricted_bounds$upper))
})

test_that("calculate_restricted_bounds handles different alpha values", {
  # Use small sample for speed
  set.seed(123)
  n <- 5
  estimates <- rnorm(n)
  var <- diag(n) * 0.1
  
  # Test with different confidence levels
  result_99 <- calculate_restricted_bounds(estimates, var, alpha = 0.01)
  result_95 <- calculate_restricted_bounds(estimates, var, alpha = 0.05)
  result_90 <- calculate_restricted_bounds(estimates, var, alpha = 0.10)
  
  # Check alpha is stored correctly
  expect_equal(result_99$metadata$alpha, 0.01)
  expect_equal(result_95$metadata$alpha, 0.05)
  expect_equal(result_90$metadata$alpha, 0.10)
  
  # Bounds should be wider for higher confidence (lower alpha)
  width_99 <- mean(result_99$restricted_bounds$upper - result_99$restricted_bounds$lower)
  width_95 <- mean(result_95$restricted_bounds$upper - result_95$restricted_bounds$lower)
  width_90 <- mean(result_90$restricted_bounds$upper - result_90$restricted_bounds$lower)
  
  expect_true(width_99 > width_95)
  expect_true(width_95 > width_90)
})

test_that("calculate_restricted_bounds handles parallel parameter", {
  skip_on_cran()
  skip_if_not_installed("doParallel")
  skip_if_not_installed("foreach")

  # Use small sample for speed
  set.seed(456)
  n <- 8
  estimates <- rnorm(n)
  var <- diag(n) * 0.1

  # Test with parallel = FALSE
  set.seed(100)
  result_seq <- calculate_restricted_bounds(estimates, var, parallel = FALSE)

  # Test with parallel = TRUE
  set.seed(100)
  result_par <- calculate_restricted_bounds(estimates, var, parallel = TRUE)
  
  # Results should be very similar
  expect_equal(result_seq$restricted_bounds$horizon,
               result_par$restricted_bounds$horizon)
  expect_equal(result_seq$restricted_bounds$unrestr_est,
               result_par$restricted_bounds$unrestr_est)

  # Restricted estimate and bounds might have small numerical differences
  expect_equal(result_seq$restricted_bounds$restr_est,
               result_par$restricted_bounds$restr_est,
               tolerance = 0.01)
})

test_that("calculate_restricted_bounds validates inputs correctly", {
  # Single estimate should error (need at least 2 for restrictions)
  expect_error(calculate_restricted_bounds(0.5, matrix(.1)),
               "calculate_restricted_bounds requires at least 2 estimates")
  
  # Invalid estimates
  expect_error(calculate_restricted_bounds("not_numeric", diag(3)),
               "estimates must be a numeric vector")
  expect_error(calculate_restricted_bounds(matrix(1:6, 2, 3), diag(3)),
               "estimates must be a numeric vector")
  
  # Invalid variance matrix
  expect_error(calculate_restricted_bounds(1:3, "not_matrix"),
               "var must be a square matrix")
  expect_error(calculate_restricted_bounds(1:3, diag(4)),
               "var must be a square matrix with dimensions matching")
  expect_error(calculate_restricted_bounds(1:3, matrix(1:6, 2, 3)),
               "var must be a square matrix")
  
  # Invalid alpha
  expect_error(calculate_restricted_bounds(1:3, diag(3), alpha = 0),
               "alpha must be a number between 0 and 1")
  expect_error(calculate_restricted_bounds(1:3, diag(3), alpha = 1),
               "alpha must be a number between 0 and 1")
  expect_error(calculate_restricted_bounds(1:3, diag(3), alpha = "not_numeric"),
               "alpha must be a number between 0 and 1")
})

test_that("calculate_restricted_bounds works with edge cases", {

  # Two estimates
  estimates_two <- c(0.5, -0.3)
  var_two <- diag(2) * 0.1
  
  result_two <- calculate_restricted_bounds(estimates_two, var_two)
  expect_s3_class(result_two, "restricted_bounds")
  expect_equal(nrow(result_two$restricted_bounds), 2)
  expect_equal(result_two$restricted_bounds$unrestr_est, estimates_two)
  
  # Three estimates (minimum for meaningful restriction)
  estimates_three <- c(0.5, -0.3, 0.2)
  var_three <- diag(3) * 0.1
  
  result_three <- calculate_restricted_bounds(estimates_three, var_three)
  expect_s3_class(result_three, "restricted_bounds")
  expect_equal(nrow(result_three$restricted_bounds), 3)
})

test_that("calculate_restricted_bounds handles different variance structures", {
  set.seed(789)
  n <- 5  # Small sample for speed
  estimates <- rnorm(n)
  
  # Identity variance
  var_identity <- diag(n)
  result_identity <- calculate_restricted_bounds(estimates, var_identity)
  
  # Scaled identity (larger variance)
  var_scaled <- diag(n) * 4
  result_scaled <- calculate_restricted_bounds(estimates, var_scaled)
  
  # Bounds should be wider with larger variance
  width_identity <- mean(result_identity$restricted_bounds$upper - 
                        result_identity$restricted_bounds$lower)
  width_scaled <- mean(result_scaled$restricted_bounds$upper - 
                      result_scaled$restricted_bounds$lower)
  expect_true(width_scaled > width_identity)
  
  # Correlated variance
  var_corr <- diag(n) * 0.5
  for (i in 1:(n-1)) {
    for (j in (i+1):n) {
      var_corr[i, j] <- var_corr[j, i] <- 0.2 * exp(-abs(i - j) / 2)
    }
  }
  
  result_corr <- calculate_restricted_bounds(estimates, var_corr)
  expect_s3_class(result_corr, "restricted_bounds")
})

test_that("calculate_restricted_bounds produces smooth restricted estimates", {
  # Generate wiggly estimates
  set.seed(111)
  n <- 8
  t <- seq(0, 1, length.out = n)
  estimates <- sin(4 * pi * t) + rnorm(n, 0, 0.2)
  var <- diag(n) * 0.1

  result <- calculate_restricted_bounds(estimates, var)

  # Restricted estimate should be smoother than original estimates
  restr_est_roughness <- sum(diff(result$restricted_bounds$restr_est)^2)
  estimates_roughness <- sum(diff(estimates)^2)

  expect_true(restr_est_roughness < estimates_roughness)

  # Check that restr_est is within reasonable range of estimates
  expect_true(all(abs(result$restricted_bounds$restr_est) <=
                 max(abs(estimates)) + 2 * sqrt(max(diag(var)))))
})

test_that("calculate_restricted_bounds metadata is correct", {
  set.seed(222)
  n <- 5
  estimates <- rnorm(n)
  var <- diag(n) * 0.15

  result <- calculate_restricted_bounds(estimates, var, alpha = 0.10)

  # Check metadata structure
  expect_type(result$metadata, "list")
  expect_true("alpha" %in% names(result$metadata))
  expect_true("suptb" %in% names(result$metadata))
  expect_true("df" %in% names(result$metadata))

  # Check metadata values
  expect_equal(result$metadata$alpha, 0.10)
  expect_true(is.numeric(result$metadata$suptb))
  expect_true(result$metadata$suptb > 0)
  expect_true(is.numeric(result$metadata$df))

  # Compute width from actual bounds
  actual_width <- mean(result$restricted_bounds$upper - result$restricted_bounds$lower)
  expect_true(is.numeric(actual_width))
  expect_true(actual_width > 0)
})


test_that("calculate_restricted_bounds optimization works correctly", {
  # Test that optimization finds a reasonable solution
  set.seed(444)
  n <- 5
  # Create estimates with clear trend
  estimates <- c(1, 1.5, 2, 2.3, 2.5)
  var <- diag(n) * 0.1
  
  result <- calculate_restricted_bounds(estimates, var)

  # Restricted estimate should capture the trend
  expect_true(all(diff(result$restricted_bounds$restr_est) >= -0.5))
  
  # Check that metadata contains expected fields
  expect_true(!is.null(result$metadata$restr_class))
  expect_true(is.character(result$metadata$restr_class))
})

test_that("calculate_restricted_bounds handles constant estimates", {
  # All estimates the same
  n <- 5
  estimates_const <- rep(2, n)
  var <- diag(n) * 0.1
  
  result <- calculate_restricted_bounds(estimates_const, var)

  # Restricted estimate should be approximately constant
  expect_true(all(abs(diff(result$restricted_bounds$restr_est)) < 0.1))
  
  # Bounds should be relatively uniform
  widths <- result$restricted_bounds$upper - result$restricted_bounds$lower
  expect_true(sd(widths) / mean(widths) < 0.1)
})

test_that("calculate_restricted_bounds handles zero variance edge case", {
  # Very small variance
  n <- 4
  estimates <- rnorm(n)
  var_small <- diag(n) * 1e-7
  
  # Should still work but with very narrow bounds
  result <- calculate_restricted_bounds(estimates, var_small)
  expect_s3_class(result, "restricted_bounds")
  
  widths <- result$restricted_bounds$upper - result$restricted_bounds$lower
  expect_true(all(widths < 0.01))
})

test_that("calculate_restricted_bounds produces reproducible results", {
  set.seed(555)
  n <- 5
  estimates <- rnorm(n)
  var <- diag(n) * 0.1
  
  set.seed(999)
  result1 <- calculate_restricted_bounds(estimates, var)
  
  set.seed(999)
  result2 <- calculate_restricted_bounds(estimates, var)
  
  expect_identical(result1$restricted_bounds, result2$restricted_bounds)
  expect_identical(result1$metadata, result2$metadata)
})

test_that("calculate_restricted_bounds validates n_cores parameter", {
  skip_on_cran()
  skip_if_not_installed("doParallel")
  skip_if_not_installed("foreach")

  set.seed(123)
  n <- 6
  estimates <- rnorm(n)
  var <- diag(n) * 0.5

  # Invalid: non-numeric
  expect_error(
    calculate_restricted_bounds(estimates, var, parallel = TRUE, n_cores = "two"),
    "n_cores must be a positive integer"
  )

  # Invalid: negative
  expect_error(
    calculate_restricted_bounds(estimates, var, parallel = TRUE, n_cores = -1),
    "n_cores must be a positive integer"
  )

  # Invalid: zero
  expect_error(
    calculate_restricted_bounds(estimates, var, parallel = TRUE, n_cores = 0),
    "n_cores must be a positive integer"
  )

  # Invalid: non-integer
  expect_error(
    calculate_restricted_bounds(estimates, var, parallel = TRUE, n_cores = 2.5),
    "n_cores must be a positive integer"
  )

  # Invalid: vector
  expect_error(
    calculate_restricted_bounds(estimates, var, parallel = TRUE, n_cores = c(2, 4)),
    "n_cores must be a positive integer"
  )
})

test_that("calculate_restricted_bounds works with n_cores = 1", {
  skip_on_cran()
  skip_if_not_installed("doParallel")
  skip_if_not_installed("foreach")

  set.seed(789)
  n <- 8
  estimates <- rnorm(n)
  var <- diag(n) * 0.1

  # Test with n_cores = 1
  set.seed(100)
  result_1core <- calculate_restricted_bounds(estimates, var, parallel = TRUE, n_cores = 1)

  # Test with parallel = FALSE for comparison
  set.seed(100)
  result_seq <- calculate_restricted_bounds(estimates, var, parallel = FALSE)

  # Results should be very similar
  expect_s3_class(result_1core, "restricted_bounds")
  expect_equal(result_1core$restricted_bounds$horizon,
               result_seq$restricted_bounds$horizon)
  expect_equal(result_1core$restricted_bounds$unrestr_est,
               result_seq$restricted_bounds$unrestr_est)
  expect_equal(result_1core$restricted_bounds$restr_est,
               result_seq$restricted_bounds$restr_est,
               tolerance = 0.01)
})
