% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/one_compartment_iv_bolus.R
\name{one_compartment_iv_bolus}
\alias{one_compartment_iv_bolus}
\title{One-Compartment IV Bolus Pharmacokinetic Model (Linear)}
\arguments{
\item{data}{A data frame containing plasma concentration-time data.}

\item{time_col}{Character string specifying the column name for time.}

\item{conc_col}{Character string specifying the column name for plasma concentration.}

\item{dose}{Numeric value specifying the administered IV bolus dose.}

\item{group_col}{Optional character string specifying a grouping variable
(e.g., formulation, subject).}

\item{plot}{Logical; if TRUE, generates a concentration-time plot with fitted lines.}

\item{annotate}{Logical; if TRUE, annotates the plot with PK parameters
(only if <= 2 groups).}
}
\value{
A list containing:
\describe{
  \item{\code{fitted_parameters}}{Data frame with C0, k_el, t1/2, Vd, CL, and R2.}
  \item{\code{data}}{Processed data used for fitting and plotting.}
}
}
\description{
Fits plasma concentration-time data to the one-compartment intravenous (IV) bolus
pharmacokinetic model. The model assumes instantaneous drug distribution throughout
a single, well-mixed compartment and first-order elimination kinetics.

The function performs linear regression on log-transformed plasma concentration
versus time to estimate the elimination rate constant (k_el), elimination half-life
(t1/2), initial concentration (C0), apparent volume of distribution (Vd), and
clearance (CL). Optional grouping (e.g., formulation, subject) and pH-dependent
analysis are supported. Publication-quality plots with fitted regression lines
and parameter annotations are generated.

Model:
C(t) = C0 * exp(-k_el * t)

Linearized form:
log(C) = log(C0) - k_el * t

where:
\itemize{
  \item C(t) is plasma concentration at time t
  \item C0 is the initial plasma concentration
  \item k_el is the elimination rate constant
}

Pharmacokinetic parameters:
\itemize{
  \item Elimination half-life: t1/2 = ln(2) / k_el
  \item Clearance: CL = Dose / AUC
  \item Volume of distribution: Vd = CL / k_el
}
}
\examples{
# Example I: Single subject IV bolus data
df <- data.frame(
  time = c(0.25, 0.5, 1, 2, 4, 6, 8, 12),
  concentration = c(18.2, 16.1, 13.5, 10.2, 6.8, 4.9, 3.6, 2.1)
)
one_compartment_iv_bolus(
  data = df,
  time_col = "time",
  conc_col = "concentration",
  dose = 100
)

# Example II: Condition-dependent pharmacokinetics (e.g., pH or physiological state)
df_cond <- data.frame(
  time = rep(c(0.25, 0.5, 1, 2, 4, 6), 2),
  concentration = c(
    17.8, 15.6, 13.1, 9.8, 6.4, 4.8,   # Condition A
    14.9, 13.0, 10.9, 8.0, 5.2, 3.9    # Condition B
  ),
  condition = rep(c("Condition A", "Condition B"), each = 6)
)
one_compartment_iv_bolus(
  data = df_cond,
  time_col = "time",
  conc_col = "concentration",
  dose = 100,
  group_col = "condition"
)

# Example III: Multiple subjects (population-style IV bolus pharmacokinetics)
df_subjects <- data.frame(
  time = rep(c(0.25, 0.5, 1, 2, 4, 6, 8), 6),
  concentration = c(
    18.6, 16.3, 13.9, 10.5, 7.0, 5.1, 3.8,   # Subject 1
    17.9, 15.7, 13.2, 9.9, 6.6, 4.9, 3.6,    # Subject 2
    17.1, 15.0, 12.6, 9.4, 6.3, 4.7, 3.4,    # Subject 3
    16.4, 14.4, 12.0, 9.0, 6.0, 4.4, 3.2,    # Subject 4
    15.8, 13.9, 11.6, 8.7, 5.8, 4.2, 3.1,    # Subject 5
    15.2, 13.3, 11.0, 8.3, 5.5, 4.0, 2.9     # Subject 6
  ),
  subject = rep(paste0("S", 1:6), each = 7)
)
one_compartment_iv_bolus(
  data = df_subjects,
  time_col = "time",
  conc_col = "concentration",
  dose = 100,
  group_col = "subject"
)
}
\references{
Widmark, E. M. P. (1919) Studies in the concentration of indifferent
narcotics in blood and tissues. Acta Medica Scandinavica, 52(1), 87–164.

Gibaldi, M. & Perrier, D. (1982) <isbn:9780824710422> Pharmacokinetics,
2nd Edition. Marcel Dekker, New York.

Gabrielsson, J. & Weiner, D. (2000) <isbn:9186274929> Pharmacokinetic/Pharmacodynamic
Data Analysis: Concepts and Applications, 3rd Edition, Revised and Expanded.
Swedish Pharmaceutical Press, Stockholm.
}
\author{
Paul Angelo C. Manlapaz
}
