% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rrmc.R
\name{rrmc}
\alias{rrmc}
\title{Rank-based robust matrix completion (non-convex PCP)}
\usage{
rrmc(D, r, eta = NULL, LOD = -Inf)
}
\arguments{
\item{D}{The input data matrix (can contain \code{NA} values). Note that PCP will
converge much more quickly when \code{D} has been standardized in some way (e.g.
scaling columns by their standard deviations, or column-wise min-max
normalization).}

\item{r}{An integer \verb{>= 1} specifying the maximum rank PCP model to return.
All models from rank \code{1} through \code{r} will be returned.}

\item{eta}{(Optional) A double in the range \verb{[0, Inf)} defining the ratio
between the model's sensitivity to sparse and dense noise.
Larger values of \code{eta} will place a greater emphasis on penalizing the
non-zero entries in \code{S} over penalizing dense noise \code{Z}, i.e. errors
between the predicted and observed data \code{Z = L + S - D}. It is recommended
to tune \code{eta} using \code{\link[=grid_search_cv]{grid_search_cv()}} for each unique data matrix \code{D}. By
default, \code{eta = NULL}, in which case \code{eta} is retrieved using
\code{\link[=get_pcp_defaults]{get_pcp_defaults()}}.}

\item{LOD}{(Optional) The limit of detection (LOD) data. Entries in \code{D} that
satisfy \code{D >= LOD} are understood to be above the LOD, otherwise those
entries are treated as below the LOD. \code{LOD} can be either:
\itemize{
\item A double, implying a universal LOD common across all measurements in \code{D};
\item A vector of length \code{ncol(D)}, signifying a column-specific LOD, where
each entry in the \code{LOD} vector corresponds to the LOD for each column in
\code{D}; or
\item A matrix of dimension \code{dim(D)}, indicating an observation-specific LOD,
where each entry in the \code{LOD} matrix corresponds to the LOD for each
entry in \code{D}.
}

By default, \code{LOD = -Inf}, indicating there are no known LODs for PCP to
leverage.}
}
\value{
A list containing:
\itemize{
\item \code{L}: The rank-\code{r} low-rank matrix encoding the \code{r}-many latent patterns
governing the observed input data matrix \code{D}. \code{dim(L)} will be the same
as \code{dim(D)}. To explicitly obtain the underlying patterns, \code{L} can be
used as the input to any matrix factorization technique of choice, e.g.
PCA, factor analysis, or non-negative matrix factorization.
\item \code{S}: The sparse matrix containing the rare outlying or extreme
observations in \code{D} that are not explained by the underlying patterns in
the corresponding \code{L} matrix. \code{dim(S)} will be the same as \code{dim(D)}.
Most entries in \code{S} are \code{0}, while non-zero entries identify the extreme
outlying observations in \code{D}.
\item \code{L_list}: A list of the \code{r}-many \code{L} matrices recovered over the course
of \code{rrmc()}'s iterative optimization procedure. The first element in
\code{L_list} corresponds to the rank-\code{1} \code{L} matrix, the second to the
rank-\code{2} \code{L} matrix, and so on.
\item \code{S_list}: A list of the \code{r}-many corresponding \code{S} matrices recovered
over the course of \code{rrmc()}'s iterative optimization procedure. The first
element in \code{S_list} corresponds to the rank-\code{1} solution's \code{S} matrix,
the second to the rank-\code{2} solution's \code{S} matrix, and so on.
\item \code{objective}: A vector containing the values of \code{rrmc()}'s objective
function over the course of optimization.
}
}
\description{
\code{rrmc()} implements the non-convex PCP algorithm "Rank-based robust matrix
completion" as described in
\href{https://proceedings.mlr.press/v70/cherapanamjeri17a.html}{Cherapanamjeri et al. (2017)}
(see Algorithm 3), outfitted with environmental health (EH)-specific
extensions as described in Gibson et al. (2022).

Given an observed data matrix \code{D}, maximum rank to search up to \code{r}, and
regularization parameter \code{eta}, \code{rrmc()} seeks to find the best low-rank
and sparse estimates \code{L} and \code{S} using an incremental rank-based strategy.
The \code{L} matrix encodes latent patterns that govern the observed data.
The \code{S} matrix captures any extreme events in the data unexplained by the
underlying patterns in \code{L}.

\code{rrmc()}'s incremental rank-based strategy first estimates a rank-\code{1} model
\eqn{(L^{(1)}, S^{(1)})}, before using the rank-\code{1} model as the initialization
point to then construct a rank-\code{2} model \eqn{(L^{(2)}, S^{(2)})}, and so on,
until the desired rank-\code{r} model \eqn{(L^{(r)}, S^{(r)})} is recovered. All
models from ranks \code{1} through \code{r} are returned by \code{rrmc()} in this way.

Experimentally, the \code{rrmc()} approach to PCP has best been able to handle
those datasets that are governed by complex underlying patterns characterized
by slowly decaying singular values, such as EH data. For observed data with a
well-defined low rank structure (rapidly decaying singular values),
\code{\link[=root_pcp]{root_pcp()}} may offer a better model estimate.

Two EH-specific extensions are currently supported by \code{rrmc()}:
\enumerate{
\item The model can handle missing values in the input data matrix \code{D}; and
\item The model can also handle measurements that fall below the limit of
detection (LOD), if provided \code{LOD} information by the user.
}

Support for a non-negativity constraint on \code{rrmc()}'s output will be added in
a future release of \code{pcpr}.
}
\section{The objective function}{

\code{rrmc()} implicitly optimizes the following objective function:
\deqn{\min_{L, S} I_{rank(L) \leq r} + \eta ||S||_0 + ||L + S - D||_F^2}
The first term is the indicator function checking that the \code{L} matrix is
strictly rank \code{r} or less, implemented using a rank \code{r} projection operator
\code{\link[=proj_rank_r]{proj_rank_r()}}. The second term is the \eqn{\ell_0} norm applied to the \code{S}
matrix to encourage sparsity, and is implemented with the help of an adaptive
hard-thresholding operator \code{\link[=hard_threshold]{hard_threshold()}}. The third term is the squared
Frobenius norm applied to the model's noise.
}

\section{The \code{eta} parameter}{

The \code{eta} parameter scales the sparse penalty applied to \code{rrmc()}'s output
sparse \code{S} matrix. Larger values of \code{eta} penalize non-zero entries in \code{S}
more stringently, driving the recovery of sparser \code{S} matrices.

Because there are no other parameters scaling the other terms in \code{rrmc()}'s
objective function, \code{eta} can intuitively be thought of as the dial that
balances the model's sensitivity to extreme events (placed in \code{S}) and
its sensitivity to noise \code{Z} (captured by the last term in the objective,
which measures the discrepancy between the between the predicted model
and the observed data). Larger values of \code{eta} will place a
greater emphasis on penalizing the non-zero entries in \code{S} over penalizing
the errors between the predicted and observed data \code{Z = L + S - D}.
}

\section{Environmental health specific extensions}{

We refer interested readers to
Gibson et al. (2022) for the complete details regarding the EH-specific
extensions.

\strong{Missing value functionality:} PCP assumes that the same data generating
mechanisms govern both the missing and the observed entries in \code{D}. Because
PCP primarily seeks accurate estimation of \emph{patterns} rather than
individual \emph{observations}, this assumption is reasonable, but in some edge
cases may not always be justified. Missing values in \code{D} are therefore
reconstructed in the recovered low-rank \code{L} matrix according to the
underlying patterns in \code{L}. There are three corollaries to keep in mind
regarding the quality of recovered missing observations:
\enumerate{
\item Recovery of missing entries in \code{D} relies on accurate estimation of
\code{L};
\item The fewer observations there are in \code{D}, the harder it is to accurately
reconstruct \code{L} (therefore estimation of \emph{both} unobserved \emph{and} observed
measurements in \code{L} degrades); and
\item Greater proportions of missingness in \code{D} artifically drive up the
sparsity of the estimated \code{S} matrix. This is because it is not possible
to recover a sparse event in \code{S} when the corresponding entry in \code{D} is
unobserved. By definition, sparse events in \code{S} cannot be explained by
the consistent patterns in \code{L}. Practically, if 20\% of the entries in \code{D}
are missing, then at least 20\% of the entries in \code{S} will be 0.
}

\strong{Handling measurements below the limit of detection:} When equipped with
LOD information, PCP treats any estimations of values known to be below the
LOD as equally valid if their approximations fall between 0 and the LOD. Over
the course of optimization, observations below the LOD are pushed into this
known range \eqn{[0, LOD]} using penalties from above and below: should a
\eqn{< LOD} estimate be \eqn{< 0}, it is stringently penalized, since
measured observations cannot be negative. On the other hand, if a \eqn{< LOD}
estimate is \eqn{>} the LOD, it is also heavily penalized: less so than when
\eqn{< 0}, but more so than observations known to be above the LOD, because
we have prior information that these observations must be below LOD.
Observations known to be above the LOD are penalized as usual, using the
Frobenius norm in the above objective function.

Gibson et al. (2022) demonstrates that
in experimental settings with up to 50\% of the data corrupted below the LOD,
PCP with the LOD extension boasts superior accuracy of recovered \code{L} models
compared to PCA coupled with \eqn{LOD / \sqrt{2}} imputation. PCP even
outperforms PCA in low-noise scenarios with as much as 75\% of the data
corrupted below the LOD. The few situations in which PCA bettered PCP were
those pathological cases in which \code{D} was characterized by extreme noise and
huge proportions (i.e., 75\%) of observations falling below the LOD.
}

\examples{
#### -------Simple simulated PCP problem-------####
# First we will simulate a simple dataset with the sim_data() function.
# The dataset will be a 100x10 matrix comprised of:
# 1. A rank-3 component as the ground truth L matrix;
# 2. A ground truth sparse component S w/outliers along the diagonal; and
# 3. A dense Gaussian noise component
data <- sim_data()
# Best practice is to conduct a grid search with grid_search_cv() function,
# but we skip that here for brevity.
pcp_model <- rrmc(data$D, r = 3, eta = 0.35)
data.frame(
  "Observed_relative_error" = norm(data$L - data$D, "F") / norm(data$L, "F"),
  "PCA_error" = norm(data$L - proj_rank_r(data$D, r = 3), "F") / norm(data$L, "F"),
  "PCP_L_error" = norm(data$L - pcp_model$L, "F") / norm(data$L, "F"),
  "PCP_S_error" = norm(data$S - pcp_model$S, "F") / norm(data$S, "F")
)
}
\references{
Cherapanamjeri, Yeshwanth, Kartik Gupta, and Prateek Jain.
"Nearly optimal robust matrix completion."
International Conference on Machine Learning. PMLR, 2017. [available
\href{https://proceedings.mlr.press/v70/cherapanamjeri17a.html}{here}]

Gibson, Elizabeth A., Junhui Zhang, Jingkai Yan, Lawrence
Chillrud, Jaime Benavides, Yanelli Nunez, Julie B. Herbstman, Jeff
Goldsmith, John Wright, and Marianthi-Anna Kioumourtzoglou.
"Principal component pursuit for pattern identification in
environmental mixtures." Environmental Health Perspectives 130, no.
11 (2022): 117008.
}
\seealso{
\code{\link[=root_pcp]{root_pcp()}}
}
