#' Fits Goodman's L. A. (1979) Simple Models for the Analysis of Association in
#' Cross-Classifications Having Ordered Categories
#'
#' null association model
#'
#' @param n matrix of observed counts
#' @param max_iter maximum number of iterations. Default is 10
#' @param verbose should cycle-by-cycle info be printed? Default is FALSE
#' @param exclude_diagonal logical, Should the diagonal be excluded from
#' the computations. Default is FALSE
#' @returns a list containing
#'    alpha: row effects
#'    beta: column effects
#'    log_likelihood: log(likelihood)
#'    g_squared: G^2 fit measure
#'    chisq: X^2 fit measure
#'    df: degrees of freedom
#' @export
Goodman_null_association <- function(n, max_iter=25, verbose=FALSE,
                                     exclude_diagonal=FALSE) {
  values <- model_i_starting_values(n)
  alpha <- values$alpha
  beta <- values$beta

  fHat <- null_association_fHat(alpha, beta)
  if (verbose) {
    pi <- fHat / sum(fHat)
    if (exclude_diagonal) {
      pi <- fHat / (sum(fHat) - sum(diag(fHat)))
    }
    message(paste("Iter = 0, ", log_likelihood(n, pi, exclude_diagonal), "\n"))
  }

  for (iter in 1:max_iter) {
    fHat <- null_association_fHat(alpha, beta)

    alpha <- model_i_update_alpha(alpha, n, fHat, exclude_diagonal)
    fHat <- null_association_fHat(alpha, beta)

    beta <- model_i_update_beta(beta, n, fHat, exclude_diagonal)
    fHat <- null_association_fHat(alpha, beta)

    if (verbose) {
      pi <- model_i_normalize_fHat(fHat, exclude_diagonal)
      message(paste("Iter = ", iter, ",  ", log_likelihood(n, pi, exclude_diagonal), "\n"))
    }
  }

  fHat <- null_association_fHat(alpha, beta)
  pi <- model_i_normalize_fHat(fHat, exclude_diagonal)
  logL <- log_likelihood(n, pi, exclude_diagonal)
  chisq <- pearson_chisq(n, pi, exclude_diagonal)
  g_squared <- likelihood_ratio_chisq(n, pi, exclude_diagonal)

  I <- nrow(n)
  J <- ncol(n)
  df <- (I - 1) * (J - 1)
  if (exclude_diagonal) {
    df <- df - I
  }

  list(alpha=alpha, beta=beta, log_likelihood=logL,
       g_squared=g_squared, chisq=chisq, df=df)
}


#' Computes expected counts for null association model
#'
#' @param alpha row effects
#' @param beta column effects
#' @returns matrix of model-based expected counts
#' @export
null_association_fHat <- function(alpha, beta) {
  I <- length(alpha)
  J <- length(beta)
  epsilon = 1.0e-6
  fHat <- matrix(nrow=I, ncol=J)
  for (i in 1:I) {
    for (j in 1:J) {
      fHat[i, j] <- alpha[i] * beta[j]
      if (fHat[i, j] < epsilon) {
        fHat[i, j] <- epsilon
      }
    }
  }
  fHat
}


#' Fits Goodman's (1979) uniform association model
#'
#' @param n matrix of observed counts
#' @param max_iter maximum number of iterations. Default is 10.
#' @param verbose should cycle-by-cycle info be printed out? Default is FALSE
#' @param exclude_diagonal logical. Should the cells of the main diagonal be
#' excluded from the computations? Default is FALSE, include all cells.
#' @returns a list containing
#'    alpha: row effects
#'    beta: column effects
#'    theta: uniform association parameter
#'    log_likelihood: log(likelihood)
#'    g_squared: G^2 fit measure
#'    chisq: X^2 fit measure
#'    df: degrees of freedom
#' @export
Goodman_uniform_association <- function(n, max_iter=25, verbose=FALSE,
                                        exclude_diagonal=FALSE) {
  values <- model_i_starting_values(n)
  alpha <- values$alpha
  beta <- values$beta
  theta <- 1.0

  for (iter in 1:max_iter) {
    fHat <- uniform_association_fHat(alpha, beta, theta)

    alpha <- model_i_update_alpha(alpha, n, fHat, exclude_diagonal)
    fHat <- uniform_association_fHat(alpha, beta, theta)

    beta <- model_i_update_beta(beta, n, fHat, exclude_diagonal)
    fHat <- uniform_association_fHat(alpha, beta, theta)

    theta <- uniform_association_update_theta(theta, n, fHat, exclude_diagonal)
    fHat <- uniform_association_fHat(alpha, beta, theta)

    if (verbose) {
      pi <- model_i_normalize_fHat(fHat, exclude_diagonal)
      message(log_likelihood(n, pi, exclude_diagonal))
    }
  }

  fHat <- uniform_association_fHat(alpha, beta, theta)
  pi <- model_i_normalize_fHat(fHat, exclude_diagonal)
  logL <- log_likelihood(n, pi, exclude_diagonal)
  chisq <- pearson_chisq(n, pi, exclude_diagonal)
  g_squared <- likelihood_ratio_chisq(n, pi, exclude_diagonal)

  I <- nrow(n)
  J <- ncol(n)
  df <- (I - 1) * (J - 1) - 1
  if (exclude_diagonal) {
    df <- df - I
  }

  list(alpha=alpha, beta=beta, theta=theta, log_likelihood=logL,
       g_squared=g_squared, chisq=chisq, df=df)
}


#' Computes expected counts for uniform association model
#'
#' @param alpha row effects
#' @param beta column effects
#' @param theta association parameter
#' @returns matrix of model-based expected counts
#' @export
uniform_association_fHat <- function(alpha, beta, theta) {
  I <- length(alpha)
  J <- length(beta)
  epsilon = 1.0e-6
  fHat <- matrix(nrow=I, ncol=J)
  for (i in 1:I) {
    for (j in 1:J) {
      fHat[i, j] <- alpha[i] * beta[j] * theta^(i * j)
      if (fHat[i, j] < epsilon) {
        fHat[i, j] <- epsilon
      }
    }
  }
  fHat
}


#' Updates estimate of theta value of the uniform association model
#'
#' @param theta current estimate of theta
#' @param n matrix of observed counts
#' @param fHat current model-based counts for each cell
#' @param exclude_diagonal logical. Should the cells of the main diagonal be
#' excluded from the computations? Defualt is FALSE, include all cells.
#' @returns updated estimate of theta parameter
uniform_association_update_theta <- function(theta, n, fHat, exclude_diagonal=FALSE) {
  I <- nrow(n)
  J <- ncol(n)
  numer <- 0.0
  denom <- 0.0
  for (i in 1:I) {
    rho <- i - (I + 1) / 2
    for (j in 1:J) {
      if (exclude_diagonal && j == i) {
        next
      }
      sigma <- j - (J + 1) / 2
      numer <- numer + rho * sigma * (n[i, j] - fHat[i, j])
      denom <- denom + (rho * sigma)^2 * fHat[i, j]
    }
  }
  theta <- theta * (1.0 + numer / denom)
  theta
}


#' Fits Goodman's (1979) Model I
#'
#' @param n matrix of observed counts
#' @param row_effects should row effects be included in the model? Default is TRUE
#' @param column_effects should column effects be included in the model? Default is TRUE
#' @param max_iter maximum number of iterations. Default is 10
#' @param verbose logical. Should cycle-by-cycle output be printed? Default is no
#' @param exclude_diagonal logical. For square tables, should the cells on the
#' diagonal be excluded?  Default is FALSE, include all cells
#' @returns a list containing
#'    alpha: row effects
#'    beta: column effects
#'    gamma: row location weights
#'    delta: column location weights
#'    log_likelihood: log(likelihood)
#'    g_squared: G^2 fit measure
#'    chisq: X^2 fit measure
#'    df: degrees of freedom
#' @export
Goodman_model_i <- function(n, row_effects=TRUE, column_effects=TRUE, max_iter=25,
                            verbose=FALSE, exclude_diagonal=FALSE) {
  values <- model_i_starting_values(n)
  alpha <- values$alpha
  beta <- values$beta
  gamma <- values$gamma
  delta <- values$delta
  if (!row_effects) {
    gamma <- rep(1.0, nrow(n))
  }
  if (!column_effects) {
    delta <- rep(1.0, ncol(n))
  }

  # the following is kludgy to get the order of updates
  # specified in the article
  for (iter in 1:max_iter) {
    fHat <- model_i_fHat(alpha, beta, gamma, delta)

    if (column_effects) {
      alpha <- model_i_update_alpha(alpha, n, fHat, exclude_diagonal)
      fHat <- model_i_fHat(alpha, beta, gamma, delta)

      delta <- model_i_update_delta(delta, n, fHat, exclude_diagonal)
      fHat <- model_i_fHat(alpha, beta, gamma, delta)

      beta <- model_i_update_beta(beta, n, fHat, exclude_diagonal)
      fHat <- model_i_fHat(alpha, beta, gamma, delta)

      if (row_effects) {
        gamma <- model_i_update_gamma(gamma, n, fHat, exclude_diagonal)
        fHat <- model_i_fHat(alpha, beta, gamma, delta)
      }
    } else {
      beta <- model_i_update_beta(beta, n, fHat, exclude_diagonal)
      fHat <- model_i_fHat(alpha, beta, gamma, delta)

      if (row_effects) {
        gamma <- model_i_update_gamma(gamma, n, fHat, exclude_diagonal)
        fHat <- model_i_fHat(alpha, beta, gamma, delta)
      }

      alpha <- model_i_update_alpha(alpha, n, fHat, exclude_diagonal)
      fHat <- model_i_fHat(alpha, beta, gamma, delta)
    }

    if (verbose) {
      pi <- model_i_normalize_fHat(fHat, exclude_diagonal)
      message(log_likelihood(n, pi, exclude_diagonal))
    }
  }

  fHat <- model_i_fHat(alpha, beta, gamma, delta)
  pi <- model_i_normalize_fHat(fHat, exclude_diagonal)
  logL <- log_likelihood(n, pi, exclude_diagonal)
  chisq <- pearson_chisq(n, pi, exclude_diagonal)
  g_squared <- likelihood_ratio_chisq(n, pi, exclude_diagonal)

  I <- nrow(n)
  J <- ncol(n)
  df <- (I - 2) * (J - 2)
  if (!row_effects) {
    df <- (I - 2) * (J - 1)
  }
  if (!column_effects) {
    df <- (I - 1) * (J - 2)
  }
  if (!row_effects && !column_effects) {
    df <- (I - 1) * (J - 1)
  }
  if (exclude_diagonal) {
    df <- df - I
  }

  list(alpha=alpha, beta=beta, gamma=gamma, delta=delta, log_likelihood=logL,
       g_squared=g_squared, chisq=chisq, df=df)
}


#' Normalizes pi(fHat) to sum to 1.0. If exclude_diagonal is TRUE,
#' the sum of the off-diagonal terms sums to 1.0.
#'
#' @param fHat matrix of model-based cell frequencies
#' @param exclude_diagonal logical. Should the cells on the main diagonal
#' be excluded? Default is FALSE, include all cells
#' @returns matrix of model-based proportions pi
#' @export
model_i_normalize_fHat <- function(fHat, exclude_diagonal=FALSE) {
  pi <- fHat / sum(fHat)
  if (exclude_diagonal) {
    pi <- fHat / (sum(fHat) - sum(diag(fHat)))
  }
  pi
}

#' Computes crude starting values for Model I.
#'
#' @param n matrix of observed counts
#' @return a list containing
#'    alpha: vector of row parameters
#'    beta: vector of column parameters
#'    gamma: vector of row locations
#'    delta: vector of column locations
model_i_starting_values <- function(n) {
  I <- nrow(n)
  J <- ncol(n)
  alpha <- rep(1.0, I)
  beta <- rep(1.0, J)
  gamma <- rep(1.0, I)
  delta <- rep(1.0, J)
  list(alpha=alpha, beta=beta, gamma=gamma, delta=delta)
}


#' Computes model-based expected cell counts for Model I
#'
#' @param alpha row effects
#' @param beta column effects
#' @param gamma row location weights
#' @param delta column location weights
#' @returns matrix of model-based expected counts
#' @export
model_i_fHat <- function(alpha, beta, gamma, delta) {
  I <- length(alpha)
  J <- length(beta)
  epsilon = 1.0e-6
  fHat <- matrix(0.0, nrow=I, ncol=J)
  for (i in 1:I) {
    for (j in 1:J) {
      fHat[i, j] <- alpha[i] * beta[j] * gamma[i]^j * delta[j]^i
      if (fHat[i, j] < epsilon) {
        fHat[i, j] <- epsilon
      }
    }
  }
  fHat
}


#' Updates the estimate of the alpha vector for Model I
#'
#' @param alpha current estimate of beta
#' @param n matrix of observed counts
#' @param fHat current model-based counts for each cell
#' @param exclude_diagonal logical. Should the diagonal be excluded
#' from the computation? Default is FALSE, use all cells.
#' @returns updated estimate of alpha vector
model_i_update_alpha <- function(alpha, n, fHat, exclude_diagonal=FALSE) {
  f_i_dot <- rowSums(n)
  fHat_i_dot <- rowSums(fHat)
  if (exclude_diagonal) {
    r <- nrow(n)
    for (i in 1:r) {
      f_i_dot[i] <- f_i_dot[i] - n[i, i]
      fHat_i_dot[i] <- fHat_i_dot[i] - fHat[i, i]
    }
  }
  alpha <- alpha * f_i_dot / fHat_i_dot
  alpha
}


#' Updates the estimate of the beta vector for Model I
#'
#' @param beta current estimate of alpha
#' @param n matrix of observed counts
#' @param fHat current model-based counts for each cell
#' @param exclude_diagonal logical. Should the diagonal be excluded
#' from the computation? Default is FALSE, use all cells
#' @returns updated estimate of beta vector
model_i_update_beta <- function(beta, n, fHat, exclude_diagonal=FALSE) {
  f_dot_j <- colSums(n)
  fHat_dot_j <- colSums(fHat)
  if (exclude_diagonal) {
    r <- ncol(n)
    for (j in 1:r) {
      f_dot_j[j] <- f_dot_j[j] - n[j, j]
      fHat_dot_j[j] <- fHat_dot_j[j] - fHat[j, j]
    }
  }
  beta <- beta * f_dot_j / fHat_dot_j
  beta
}


#' Updates the estimate of the gamma vector for Model I
#'
#' @param gamma current estimate of gamma
#' @param n matrix of observed counts
#' @param fHat current model-based counts for each cell
#' @param exclude_diagonal logical. Should the diagonal be excluded
#' from the computation? Default is FALSE, use all cells
#' @returns updated estimate of gamma vector
model_i_update_gamma <- function(gamma, n, fHat, exclude_diagonal=FALSE) {
  I <- nrow(n)
  J <- ncol(n)
  for (i in 1:I) {
    numer <- 0.0
    denom <- 0.0
    for (j in 1:J) {
      if (exclude_diagonal && j == i) {
        next
      }
      sigma <- j - (J + 1) / 2
      numer <- numer + sigma * (n[i, j] - fHat[i, j])
      denom <- denom + sigma^2 * fHat[i, j]
    }
    gamma[i] <- gamma[i] * (1.0 + numer / denom)
  }
  gamma
}


#' Updates the estimate of the delta vector for Model I
#'
#' @param delta current estimate of delta
#' @param n matrix of observed counts
#' @param fHat current model-based counts for each cell
#' @param exclude_diagonal logical. Should the diagonal be excluded
#' from the computation? Default is FALSE, use all cells
#' @returns updated estimate of delta vector
model_i_update_delta <- function(delta, n, fHat, exclude_diagonal=FALSE) {
  I <- nrow(n)
  J <- ncol(n)
  for (j in 1:J) {
    numer <- 0.0
    denom <- 0.0
    for (i in 1:I) {
      if (exclude_diagonal && i == j) {
        next
      }
      rho <- i - (I + 1) / 2
      numer <- numer + rho * (n[i, j] - fHat[i, j])
      denom <- denom + rho^2 * fHat[i, j]
    }
    delta[j] <- delta[j] * (1.0 + numer / denom)
  }
  delta
}


#' Computes the row association values theta-hat
#'
#' @param fHat matrix of model-based expected counts
#' @returns thetaHat vector of association parameters
#' @export
model_i_row_theta <- function(fHat) {
  rows <- nrow(fHat)
  thetaHat <- vector("double", rows - 1)
  for (row in 1:(rows - 1)) {
    thetaHat[row] <- fHat[row, 1] * fHat[row + 1, 2] / (fHat[row, 2] * fHat[row + 1, 1])
  }
  thetaHat
}


#' Computes the column association values theta-hat
#'
#'@param fHat matrix of model-based expected counts
#'@returns thetaHat vector of association parameters
#'@export
model_i_column_theta <- function(fHat) {
  cols <- ncol(fHat)
  thetaHat <- vector("double", cols - 1)
  for (col in 1:(cols - 1)) {
    thetaHat[col] <- fHat[1, col] * fHat[2, col + 1] / (fHat[2, col] * fHat[1, col + 1])
  }
  thetaHat
}


#' Computes the table of adjacent odds-ratios theta-hat.
#'
#' @param fHat matrix of model-based expected counts
#' @return thetaHat matrix of adjacent odds-ratios
#' @export
model_i_row_column_odds_ratios <- function(fHat) {
  rows <- nrow(fHat)
  cols <- ncol(fHat)
  thetaHat <- matrix(0.0, nrow=rows - 1, ncol=cols-1)
  for (row in 1:(rows - 1)) {
    for (col in 1:(cols - 1)) {
      thetaHat[row, col] <- (fHat[row, col] * fHat[row + 1, col + 1]
                             / (fHat[row, col + 1] * fHat[row + 1, col]))
    }
  }
  thetaHat
}


#' Gets the overall effects for Model I.
#'
#' @param result a Model I result object
#' @returns a list containing
#'    theta: the overall association
#'    zeta_i_dot: row effects for association
#'    zeta_dot_j: column effects for association
#' @export
model_i_effects <- function(result) {
  fHat <- model_i_fHat(result$alpha, result$bbeta, result$gamma, result$delta)
  odds <- model_i_row_column_odds_ratios(fHat)
  model_i_zeta(odds)
}


#' Computes the overall association theta and the row and column effects zeta
#'
#' @param odds matrix of adjacent odds-ratios
#' @returns a list containing
#'    theta: the overall association
#'    zeta_i_dot: row effects for association
#'    zeta_dot_j: column effects for association
#' @export
model_i_zeta <- function(odds) {
  log_odds <- log(odds)
  psi <- mean(log_odds)
  log_odds_star <- log_odds - psi
  eta_i_dot <- rowMeans(log_odds_star)
  eta_dot_j <- colMeans(log_odds_star)

  theta <- exp(psi)
  zeta_i_dot <- exp(eta_i_dot)
  zeta_dot_j <- exp(eta_dot_j)

  list(theta=theta, zeta_i_dot=zeta_i_dot, zeta_dot_j=zeta_dot_j)
}


#' Fits Goodman's (1979) Model I*
#'
#' @param n matrix of observed counts
#' @param max_iter maximum number of iterations
#' @param exclude_diagonal should the cells along the main diagonal be excluded?
#' Default is FALSE, include all cells
#' @param verbose should cycle-by-cycle information be printed out? Default
#' is FALSE, do not print
#' @returns a list containing
#'    alpha: vector of row parameters
#'    beta: vector of column parameters
#'    theta: vector of common row/column estimates
#'    log_likelihood: log(likelihood) at completion
#'    g_squared: G^2 fit measure
#'    chisq: X^2 fit measure
#'    df: degrees of freedom
#' @export
Goodman_model_i_star <- function(n, max_iter=25, verbose=FALSE, exclude_diagonal=FALSE) {
  if (nrow(n) != ncol(n)) {
    stop("Table must be square")
  }
  values <- model_i_starting_values(n)
  alpha <- values$alpha
  beta <- values$beta
  theta <- values$gamma

  for (iter in 1:max_iter) {
    fHat <- model_i_star_fHat(alpha, beta, theta)

    alpha <- model_i_update_alpha(alpha, n, fHat, exclude_diagonal)
    fHat <- model_i_star_fHat(alpha, beta, theta)

    theta <- model_i_star_update_theta(theta, n, fHat, exclude_diagonal)
    fHat <- model_i_star_fHat(alpha, beta, theta)

    beta <- model_i_update_beta(beta, n, fHat, exclude_diagonal)
    fHat <- model_i_star_fHat(alpha, beta, theta)

    if (verbose) {
      pi <- model_i_normalize_fHat(fHat, exclude_diagonal)
      message(log_likelihood(n, pi, exclude_diagonal))
    }
  }

  fHat <- model_i_star_fHat(alpha, beta, theta)
  pi <- model_i_normalize_fHat(fHat, exclude_diagonal)
  logL <- log_likelihood(n, pi, exclude_diagonal)
  chisq <- pearson_chisq(n, pi, exclude_diagonal)
  g_squared <- likelihood_ratio_chisq(n, pi, exclude_diagonal)
  I <- nrow(n)
  J <- ncol(n)
  df <- (I - 2) * (J - 1)
  if (exclude_diagonal) {
    df <- df - I
  }

  list(alpha=alpha, beta=beta, theta=theta, log_likelihood=logL,
       g_squared=g_squared, chisq=chisq, df=df)
}


#' Computes expected frequencies for Model I*
#'
#' @param alpha row effect parameters
#' @param beta column effect parameters
#' @param theta row/column parameters
#' @returns matrix of model-based expected cell counts
#' @export
model_i_star_fHat <- function(alpha, beta, theta) {
  I <- length(alpha)
  J <- length(beta)
  if (I != J) {
    stop("Matrix must be square")
  }
  epsilon = 1.0e-6
  fHat <- matrix(0.0, nrow=I, ncol=J)
  for (i in 1:I) {
    for (j in 1:J) {
      fHat[i, j] <- alpha[i] * beta[j] * theta[i]^j * theta[j]^i
      if (fHat[i, j] < epsilon) {
        fHat[i, j] <- epsilon
      }
    }
  }
  fHat
}


#' Updates the row/column parameters for Model I*.
#'
#' @param theta vector of estimated row/column effects
#' @param n matrix of observed counts
#' @param fHat matrix of model-based expected frequencies
#' @param exclude_diagonal should the cells of the main diagonal be excluded?
#' Default is FALSE, include all cells
#' @returns new value of theta vector
model_i_star_update_theta <- function(theta, n, fHat, exclude_diagonal=FALSE) {
  I <- nrow(n)
  J <- ncol(n)
  if (I != J) {
    stop("Matrix must be square")
  }

  for (i in 1:I) {
    numer <- 0.0
    denom <- 0.0
    for (j in 1:J) {
      if (exclude_diagonal && j == i) {
        next
      }
      sigma <- j - (J + 1) / 2
      numer <- numer + sigma * (n[j, i] - fHat[j, i])
      numer <- numer + sigma * (n[i, j] - fHat[i, j])
      denom <- denom + sigma^2 * fHat[j, i]
      denom <- denom + sigma^2 * fHat[i, j]
    }
    theta[i] <- theta[i] * (1.0 + (numer / denom))
  }
  theta
}


#' Gets the Model I* effects.
#'
#' @param result a Model I* effect object
#' @returns a list containing
#'    theta: the overall association
#'    zeta: the row/column effect
#' @export
model_i_star_effects <- function(result) {
  fHat <- model_i_star_fHat(result$alpha, result$beta, result$theta)
  odds <- model_i_row_column_odds_ratios(fHat)
  res <- model_i_zeta(odds)
  list(theta=res$theta, zeta=res$zeta_i_dot)
}


#' Fits Goodman's (1979) Model II
#'
#' @param n matrix of observed counts
#' @param rho values of row locations. Default is 1:nrow(n) - (nrow(n) + 1) / 2
#' @param sigma values of column locations. Default is 1:ncol(n) - (ncol(n) + 1) / 2
#' @param update_rows should values of row locations be updated? Default is TRUE, update
#' @param update_columns should value of column locations be updated? Default is TRUE, update
#' @param max_iter maximum number of iterations to perform. Default is 10
#' @param verbose should cycle-by-cycle output be produced? Default is FALSE
#' @param exclude_diagonal logical. Should the diagonal be excluded
#' from the computation. Default is FALSE.
#' @returns a list containing
#'    alpha: row effects
#'    beta: column effects
#'    rho: centered row locations
#'    mu: row locations
#'    sigma: centered column locations
#'    nu: column locations
#'    log_likelihood: log(likelihood)
#'    g_squared: G^2 fit measure
#'    chisq: X^2 fit measure
#'    df: degrees of freedom
#' @export
Goodman_model_ii <- function(n, rho=1:nrow(n) - (nrow(n) + 1) / 2,
                             sigma=1:ncol(n) - (ncol(n) + 1) / 2,
                             update_rows=TRUE, update_columns=TRUE,
                             max_iter=25, verbose=FALSE, exclude_diagonal=FALSE) {
  values <- model_ii_starting_values(n)
  alpha <- values$alpha
  beta <- values$beta
  mu <- rep(0.0, nrow(n))
  rho <- rep(1.0, nrow(n))
  nu <- rep(0.0, ncol(n))
  sigma <- rep(1.0, ncol(n))

  for (iter in 1:max_iter) {
    fHat <- model_ii_fHat(alpha, beta, rho, sigma)

    alpha <- model_ii_update_alpha(alpha, n, fHat, exclude_diagonal)
    fHat <- model_ii_fHat(alpha, beta, rho, sigma)

    if (update_rows) {
      result1 <- model_ii_update_rho(n, fHat, mu, sigma, exclude_diagonal)
      mu <- result1$mu
      rho <- result1$rho
      fHat <- model_ii_fHat(alpha, beta, rho, sigma)
    }

    beta <- model_ii_update_beta(beta, n, fHat, exclude_diagonal)
    fHat <- model_ii_fHat(alpha, beta, rho, sigma)

    if (update_columns) {
      result2 <- model_ii_update_sigma(n, fHat, nu, rho, exclude_diagonal)
      nu <- result2$nu
      sigma <- result2$sigma
      fHat <- model_ii_fHat(alpha, beta, rho, sigma)
    }

    if (verbose) {
      pi <- model_i_normalize_fHat(fHat, exclude_diagonal)
      message(paste(iter, ": ", log_likelihood(n, pi, exclude_diagonal), "\n"))
    }
  }
  fHat <- model_ii_fHat(alpha, beta, rho, sigma)
  pi <- model_i_normalize_fHat(fHat, exclude_diagonal)
  logL = log_likelihood(n, pi, exclude_diagonal)
  chisq <- pearson_chisq(n, pi, exclude_diagonal)
  g_squared <- likelihood_ratio_chisq(n, pi, exclude_diagonal)

  I <- nrow(n)
  J <- ncol(n)
  df <- (I - 2) * (J - 2)
  if (!update_rows) {
    df <- df + I - 1
  }
  if (!update_columns) {
    df <- df + J - 1
  }
  if (exclude_diagonal) {
    df = df - I
  }

  list(alpha=alpha, beta=beta, rho=rho, mu=mu, sigma=sigma, nu=nu, log_likelihood=logL,
       g_squared=g_squared, chisq=chisq, df=df)
}


#' Computes crude starting values for Model II
#'
#' @param n matrix of observed counts
#' @returns a list containing
#'    alpha: vector of row parameters
#'    beta: vector of column parameters
#'    rho: row coefficients
#'    sigma: column coefficients
#'    mu: alternative row coefficients
#'    nu: alternative column coefficients
model_ii_starting_values <- function(n) {
  I <- nrow(n)
  J <- ncol(n)
  alpha <- 1:I
  beta <- 1:J
  mu <- rep(0.0, I)
  nu <- rep(0.0, J)
  rho <-  1:I  - (I + 1) / 2
  sigma <- 1:J  - (J + 1) / 2
  list(alpha=alpha, beta=beta, rho=rho, sigma=sigma, mu=mu, nu=nu)
}


#' Computes expected counts for Model II
#'
#' @param alpha row effects
#' @param beta column effects
#' @param rho row locations
#' @param sigma column locations
#' @returns matrix of model-based expected counts
#' @export
model_ii_fHat <- function(alpha, beta, rho, sigma) {
  I <- length(alpha)
  J <- length(beta)
  epsilon <- 1.0e-6
  fHat <- matrix(0.0, nrow=I, ncol=J)
  for (i in 1:I) {
    for (j in 1:J) {
      fHat[i, j] <- alpha[i] * beta[j] * exp(rho[i] * sigma[j])
      if (fHat[i, j] < epsilon) {
        fHat[i, j] <- epsilon
      }
    }
  }
  fHat
}


#' Updates the estimate of the rho vector for Model II
#'
#' @param n matrix of observed counts
#' @param fHat current model-based counts for each cell
#' @param mu alternative row coefficients
#' @param sigma vector of column location parameters
#' @param exclude_diagonal logical, Should the cells on the main diagonal be
#' excluded? Default is FALSE, use all cells
#' @returns updated estimate of alpha vector
model_ii_update_rho <- function(n, fHat, mu, sigma, exclude_diagonal=FALSE) {
  I <- nrow(n)
  J <- ncol(n)
  mu_star <- vector("double", I)
  for (i in 1:I) {
    numer <- 0.0
    denom <- 0.0
    for (j in 1:J) {
      if (exclude_diagonal && j == i) {
        next
      }
      numer <- numer + sigma[j] * (n[i,j] - fHat[i, j])
      denom <- denom + fHat[i, j] * sigma[j]**2
    }
    mu_star[i] <- mu[i] + numer / denom
  }
  rho_star <- mu_star - mean(mu_star)
  list(rho=rho_star, mu=mu_star)
}


#' Updates the estimate of the sigma vector for Model II
#'
#' @param n matrix of observed counts
#' @param fHat current model-based counts for each cell
#' @param nu vector of column coefficients
#' @param rho vector of row location parameters
#' @param exclude_diagonal logical, Should the cells on the main diagonal be
#' excluded? Default is FALSE, use all cells
#' @returns updated estimate of sigma vector
model_ii_update_sigma <- function(n, fHat, nu, rho, exclude_diagonal=FALSE) {
  I <- nrow(n)
  J <- ncol(n)
  nu_star <- vector("double", J)
  for (j in 1:J) {
    numer <- 0.0
    denom <- 0.0
    for (i in 1:I) {
      if (exclude_diagonal && i == j) {
        next
      }
      numer <- numer + rho[i] * (n[i,j] - fHat[i, j])
      denom <- denom + fHat[i, j] * rho[i]**2
    }
    nu_star[j] <- nu[j] + numer / denom
  }
  sigma_star <- nu_star - mean(nu_star)
  list(sigma=sigma_star, nu=nu_star)
}


#' Updates the estimate of the alpha vector for Model II
#'
#' @param alpha current estimate of alpha
#' @param n matrix of observed counts
#' @param fHat current model-based counts for each cell
#' @param exclude_diagonal logical, Should the cells on the main diagonal be
#' excluded? Default is FALSE, use all cells
#' @returns updated estimate of alpha vector
model_ii_update_alpha <- function(alpha, n, fHat, exclude_diagonal=FALSE) {
  f_i_dot <- rowSums(n)
  fHat_i_dot <- rowSums(fHat)
  if (exclude_diagonal) {
    r <- nrow(n)
    for (i in 1:r) {
      f_i_dot[i] <- f_i_dot[i] - n[i, i]
      fHat_i_dot[i] <- fHat_i_dot[i] - fHat[i, i]
    }
  }
  alpha_star <- alpha * f_i_dot / fHat_i_dot
  alpha_star
}


#' Updates the estimate of the beta vector for Model II
#'
#' @param beta current estimate of beta
#' @param n matrix of observed counts
#' @param fHat current model-based counts for each cell
#' @param exclude_diagonal logical, Should the cells on the main diagonal be
#' excluded? Default is FALSE, use all cells
#' @returns updated estimate of beta vector
model_ii_update_beta <- function(beta, n, fHat, exclude_diagonal=FALSE) {
  f_dot_j <- colSums(n)
  fHat_dot_j <- colSums(fHat)
  if (exclude_diagonal) {
    r <- ncol(n)
    for (j in 1:r) {
      f_dot_j[j] <- f_dot_j[j] - n[j, j]
      fHat_dot_j[j] <- fHat_dot_j[j] - fHat[j, j]
    }
  }
  beta_star <- beta * f_dot_j / fHat_dot_j
  beta_star
}


#' Gets the effects phi, ksi_i_dot and ksi_dot_j for Model II results.
#'
#' @param result a result object from Model II
#' @returns a list containing:
#'    phi: the overall effect
#'    ksi_i_dot: the row effects
#'    ksi_dot_j: the column effects
#' @export
model_ii_effects <- function(result) {
  fHat <- model_ii_fHat(result$alpha, result$beta, result$rho, result$sigma)
  odds <- model_i_row_column_odds_ratios(fHat)
  model_ii_ksi(odds)
}


#' Gets the effects phi, ksi_i_dot and ksi_dot_j for Model II matrix of odds-ratios.
#'
#' @param odds matrix of adjacent odds-ratios
#' @returns a list containing:
#'    phi: the overall effect in log metric
#'    ksi_i_dot: the row effects
#'    ksi_dot_j: the column effects
#' @export
model_ii_ksi <- function(odds) {
  log_odds <- log(odds)
  phi <- sign(prod(log_odds)) * abs(prod(log(odds))) ^(1.0 / length(log(odds)))
  log_odds <- log_odds / phi

  rows <- nrow(odds)
  ksi_i_dot <- vector("double", rows)
  for (i in 1:rows) {
    ksi_i_dot[i] <- sign(prod(log_odds[i,])) * abs(prod(log_odds[i,]))^(1.0 / length(log_odds[i,]))
  }

  cols <- ncol(odds)
  ksi_dot_j <- vector("double", cols)
  for (j in 1:cols) {
    ksi_dot_j[j] <- sign(prod(log_odds[,j])) * abs(prod(log_odds[,j]))^(1.0 / length(log_odds[,j]))
  }

  list(phi=phi, ksi_i_dot=ksi_i_dot, ksi_dot_j=ksi_dot_j)
}


#' Fits Goodman's (1979) model II*, where row and column effects are equal.
#'
#' @param n matrix of observed counts
#' @param exclude_diagonal should the cells of the main diagonal be excluded?
#' Default is FALSE, include all cells
#' @param max_iter maximum number of iterations
#' @param verbose should cycle-by-cycle information be printed out?
#' Default is FALSE, do not print
#' @returns a list containing
#'    alpha: vector of alpha (row) parameters
#'    beta: vector of beta (column) parameters
#'    phi: vector of common row/column effects
#'    log_likelihood: value of the log(likelihood) function at completion
#'    g_squared: G^2 fit measure
#'    chisq: X^2 fit measure
#'    df: degrees of freedom
#' @export
Goodman_model_ii_star <- function(n, exclude_diagonal=FALSE,max_iter=25, verbose=FALSE) {
  I <- nrow(n)
  J <- ncol(n)
  if (I != J) {
    stop("Matrix must be square")
  }
  values <- model_ii_starting_values(n)
  alpha <- values$alpha
  beta <- values$beta
  mu <- rep(0.0, I)
  phi <- 1:I - (I + 1) / 2

  for (iter in 1:max_iter) {
    fHat <- model_ii_star_fHat(alpha, beta, phi)

    alpha <- model_ii_update_alpha(alpha, n, fHat, exclude_diagonal)
    fHat <- model_ii_star_fHat(alpha, beta, phi)

    result1 <- model_ii_star_update_phi(n, fHat, mu, phi, exclude_diagonal)
    mu <- result1$mu
    phi <- result1$phi
    fHat <- model_ii_star_fHat(alpha, beta, phi)

    beta <- model_ii_update_beta(beta, n, fHat, exclude_diagonal)
    fHat <- model_ii_star_fHat(alpha, beta, phi)

    if (verbose) {
      pi <- model_i_normalize_fHat(fHat, exclude_diagonal)
      message(paste(iter, ": ", log_likelihood(n, pi, exclude_diagonal), "\n"))
    }
  }
  fHat <- model_ii_star_fHat(alpha, beta, phi)
  pi <- model_i_normalize_fHat(fHat, exclude_diagonal)
  logL = log_likelihood(n, pi, exclude_diagonal)
  chisq <- pearson_chisq(n, pi, exclude_diagonal)
  g_squared <- likelihood_ratio_chisq(n, pi, exclude_diagonal)

  I <- nrow(n)
  J <- ncol(n)
  df <- (I - 2) * (J - 2) + I - 2
  if (exclude_diagonal) {
    df = df - I
  }

  list(alpha=alpha, beta=beta, phi=phi, log_likelihood=logL,
       g_squared=g_squared, chisq=chisq, df=df)
}

#' Computes expected counts for Model II*
#'
#' @param alpha row effects
#' @param beta column effects
#' @param phi row/column locations
#' @returns matrix of model-based expected counts
#' @export
model_ii_star_fHat <- function(alpha, beta, phi) {
  I <- length(alpha)
  J <- length(beta)
  epsilon <- 1.0e-6
  fHat <- matrix(0.0, nrow=I, ncol=J)
  for (i in 1:I) {
    for (j in 1:J) {
      fHat[i, j] <- alpha[i] * beta[j] * exp(phi[i] * phi[j])
      if (fHat[i, j] < epsilon) {
        fHat[i, j] <- epsilon
      }
    }
  }
  fHat
}


#' Updates estimate of phi vector
#'
#' @param n matrix of observed counts
#' @param fHat current model-based counts for each cell
#' @param mu alternative row coefficients
#' @param phi vector of column location parameters
#' @param exclude_diagonal logical, Should the cells on the main diagonal be
#' excluded? Default is FALSE, use all cells
#' @returns list containing:
#'    phi: updated estimate of the phi vector
#'    mu: updated estimate of vector mu
model_ii_star_update_phi <- function(n, fHat, mu, phi, exclude_diagonal=FALSE) {
  I <- nrow(n)
  J <- ncol(n)
  mu_star <- vector("double", I)
  for (i in 1:I) {
    numer <- 0.0
    denom <- 0.0
    for (j in 1:J) {
      if (exclude_diagonal && j == i) {
        next
      }
      numer <- numer + phi[j] * (n[i, j] - fHat[i, j])
      numer <- numer + phi[j] * (n[j, i] - fHat[j, i])
      denom <- denom + fHat[i, j] * phi[j]**2
      denom <- denom + fHat[j, i] * phi[j]**2
    }
    mu_star[i] <- mu[i] + numer / denom
  }
  phi_star <- mu_star - mean(mu_star)
  list(phi=phi_star, mu=mu_star)
}


#' Gets the effects for Model II*
#'
#' @param result a Model II* result object
#' @returns a list containing
#'    phi: common effect in log metric
#'    ksi: vector of ksi parameters
#' @export
model_ii_star_effects <- function(result) {
  fHat <- model_ii_star_fHat(result$alpha, result$beta, result$phi)
  odds <- model_i_row_column_odds_ratios(fHat)
  res <- model_ii_effects(odds)
  list(phi=res$phi, ksi=res$ksi_i_dot)
}


#' Fits the symmetric association model from Goodman (1979).
#' Note the model is a reparameterized version of the quasi-symmetry model,
#' so the quasi-symmetry model has the same fit indices.
#'
#' @param n matrix of observed counts
#' @returns a list containing
#'    x: design matrix used for the glm() regression
#'    beta: parameter estimates
#'    se: standard errors of beta
#'    g_squared: G^2 measure of fit
#'    chisq: X^2 measure of fit
#'    df: degrees of freedom
#'    expected: model-based expected cell counts
#' @export
Goodman_symmetric_association_model <- function(n) {
  x <- log_linear_quasi_symmetry_model_design(n)
  result <- log_linear_fit(n, x)
  result
}
