#' Compute T values (sum of block totals) for a given h x w plot
#'
#' @param df_mat numeric matrix of data
#' @param h rows in plot
#' @param w cols in plot
#' @return numeric vector of block totals
#' @export
get_Tvals <- function(df_mat, h, w) {
  r <- nrow(df_mat)
  c <- ncol(df_mat)
  nr <- r / h
  nc <- c / w
  if (nr != floor(nr) || nc != floor(nc)) stop("Plot does not divide matrix evenly")

  Tvals <- c()
  for (j in 1:nc) {
    for (i in 0:(nr-1)) {
      block <- df_mat[(i*h+1):((i+1)*h), ((j-1)*w+1):(j*w)]
      Tvals <- c(Tvals, sum(block))
    }
  }
  return(Tvals)
}

#' Compute population variance for given h x w plot
#'
#' @param df_mat numeric matrix of data
#' @param h rows in plot
#' @param w cols in plot
#' @return numeric variance
#' @export
population_variance <- function(df_mat, h, w) {
  Tvals <- get_Tvals(df_mat, h, w)
  V <- sum((Tvals - mean(Tvals))^2) / length(Tvals)
  return(V)
}

#' Generate valid plot sizes and shapes
#'
#' @param df_mat numeric matrix of data
#' @importFrom stats var.test bartlett.test
#' @return data.frame of possible plot sizes and shapes
#' @export
generate_plot_shapes <- function(df_mat) {
  r <- nrow(df_mat)
  c <- ncol(df_mat)
  area <- r * c
  G <- sum(df_mat)
  GM <- G / area
  valid_sizes <- (1:27)[area %% (1:27) == 0]

  res <- data.frame(
    plot_size_units = integer(0),
    shape_h = integer(0),
    shape_w = integer(0),
    num_plots = integer(0),
    shape = character(0),
    V_between = numeric(0),
    V_per_unit_area = numeric(0),
    CV_percent = numeric(0),
    p_value = numeric(0),
    test_used = character(0),
    stringsAsFactors = FALSE
  )

  seen <- character(0)
  for (x in valid_sizes) {
    num_plots <- area / x
    for (h in 1:x) {
      if (x %% h != 0) next
      w <- x / h
      if ((r %% h == 0) && (c %% w == 0)) {
        key1 <- paste0(h, "x", w)
        if (!(key1 %in% seen)) {
          Vb <- population_variance(df_mat, h, w)
          res <- rbind(res, data.frame(
            plot_size_units = x,
            shape_h = h,
            shape_w = w,
            num_plots = num_plots,
            shape = key1,
            V_between = Vb,
            V_per_unit_area = Vb / (x^2),
            CV_percent = (sqrt(Vb / (x^2)) / GM) * 100,
            p_value = NA,
            test_used = NA,
            stringsAsFactors = FALSE
          ))
          seen <- c(seen, key1)
        }
      }
      if (h != w && (r %% w == 0) && (c %% h == 0)) {
        key2 <- paste0(w, "x", h)
        if (!(key2 %in% seen)) {
          Vb <- population_variance(df_mat, w, h)
          res <- rbind(res, data.frame(
            plot_size_units = x,
            shape_h = w,
            shape_w = h,
            num_plots = num_plots,
            shape = key2,
            V_between = Vb,
            V_per_unit_area = Vb / (x^2),
            CV_percent = (sqrt(Vb / (x^2)) / GM) * 100,
            p_value = NA,
            test_used = NA,
            stringsAsFactors = FALSE
          ))
          seen <- c(seen, key2)
        }
      }
    }
  }

  # Apply variance homogeneity tests
  for (size in unique(res$plot_size_units)) {
    subset_res <- subset(res, plot_size_units == size)

    # collect all Tvals
    Tlist <- list()
    for (i in 1:nrow(subset_res)) {
      h <- subset_res$shape_h[i]
      w <- subset_res$shape_w[i]
      Tlist[[subset_res$shape[i]]] <- get_Tvals(df_mat, h, w)
    }

    if (length(Tlist) == 2) {
      test <- var.test(Tlist[[1]], Tlist[[2]])
      pval <- test$p.value
      method <- "F-test"
    } else if (length(Tlist) > 2) {
      dfTall <- do.call(rbind, lapply(names(Tlist), function(name) {
        data.frame(value = Tlist[[name]], shape = name)
      }))
      test <- bartlett.test(value ~ shape, data = dfTall)
      pval <- test$p.value
      method <- "Bartlett"
    } else {
      next
    }

    res$p_value[res$plot_size_units == size] <- pval
    res$test_used[res$plot_size_units == size] <- method
  }

  row.names(res) <- NULL
  return(res)
}
