#' Load a list of macroeconomic data from WDI
#'
#' This function downloads a set of ten macroeconomic variables via API from the World Development Indicators (WDI) of the World Bank (https://databank.worldbank.org/source/world-development-indicators).
#' The variables are suspected to have a predictive capacity for the load data.
#'
#' @param longterm_data The long-term data series resulting from the function \code{\link{decompose_load_data}}. Contains information on country (longterm$country) and years (longterm$year).
#'
#' @return Data frame with the original time series and 10 additional columns with macroeconomic indicators.
#' @export
#'
#' @seealso See function \code{\link{decompose_load_data}} for the generation of the long-term series and \code{\link{long_term_lm}} for the selection of covariates.
#'
#' @examples
#' example_longterm_and_macro_data <- get_macro_economic_data(example_longterm_data)
#' print("Macro economic variables are added from the World Bank Developer Indicators:")
#' example_longterm_and_macro_data
#'
get_macro_economic_data <- function(longterm_data) {
  if ("example" %in% colnames(longterm_data)) {
    if (unique(longterm_data$example) == TRUE) {
      return(oRaklE::example_longterm_and_macro_data)
    }
  }


  longterm <- longterm_data
  country <- unique(longterm$country)
  start_year <- min(longterm$year)
  end_year <- max(longterm$year)

  res_pop <- tryCatch(
    {
      httr::GET(
        paste0(
          "https://api.worldbank.org/v2/country/", country,
          "/indicator/SP.POP.TOTL?date=", start_year, ":", end_year,
          "&format=json"
        )
      )
    },
    error = function(e) {
      message(
        "Error during GET request to api.worldbank.org: ",
        conditionMessage(e),
        "\nAre you connected to the internet?"
      )
      return(NULL)
    }
  )

  if (is.null(res_pop)) {
    return(invisible(NULL))
  }

  if (res_pop$status_code == 502) {
    message("The World Development Indicator database of the World Bank is currently unreachable. Please try again in a minute.")
    return(invisible(NULL))
  }
  if (httr::http_error(res_pop)) {
    status <- httr::status_code(res_pop)
    error_info <- httr::http_status(res_pop)$message
    message("HTTP request to api.worldbank.org failed (status ", status, "): ", error_info)
    return(invisible(NULL))
  }

  data_pop <- jsonlite::fromJSON(rawToChar(res_pop$content))
  df_pop <- as.data.frame(data_pop[2])
  df_pop <- df_pop[order(df_pop$date), ]
  longterm$population <- rep(NA, nrow(longterm))
  for (i in seq_len(nrow(df_pop))) {
    longterm$population[i] <- df_pop$value[i]
  }

      res_gdp <- tryCatch(
        {httr::GET(paste0("https://api.worldbank.org/v2/country/", country, "/indicator/NY.GDP.MKTP.KD?date=", start_year, ":", end_year, "&format=json"))
    },
    error = function(e) {
     message("Error during GET request to api.worldbank.org", e$message,
        "\nAre you connected to the internet?"
      )
      return(NULL)
    }
  )

      if (is.null(res_gdp)) {
        return(invisible(NULL))
      }

      if (res_gdp$status_code == 502) {
        message("The World Development Indicator database of the World Bank is currently unreachable. Please try again in a minute.")
        return(invisible(NULL))
      }
      if (httr::http_error(res_gdp)) {
        status <- httr::status_code(res_gdp)
        error_info <- httr::http_status(res_gdp)$message
        message("HTTP request to api.worldbank.org failed (status ", status, "): ", error_info)
        return(invisible(NULL))
      }

  data_gdp <- jsonlite::fromJSON(rawToChar(res_gdp$content))
  df_gdp <- as.data.frame(data_gdp[2])
  df_gdp <- df_gdp[order(df_gdp$date), ]
  longterm$GDP <- rep(NA, nrow(longterm))
  for (i in seq_len(nrow(df_gdp))) {
    longterm$GDP[i] <- df_gdp$value[i]
  }


      res_ind <-   tryCatch(
        {httr::GET(paste0("https://api.worldbank.org/v2/country/", country, "/indicator/NV.IND.TOTL.ZS?date=", start_year, ":", end_year, "&format=json"))
    },
    error = function(e) {
      message("Error during GET request to api.worldbank.org", e$message,
        "\nAre you connected to the internet?"
      )
      return(NULL)
    }
  )
      if (is.null(res_ind)) {
        return(invisible(NULL))
      }

      if (res_ind$status_code == 502) {
        message("The World Development Indicator database of the World Bank is currently unreachable. Please try again in a minute.")
        return(invisible(NULL))
      }
      if (httr::http_error(res_ind)) {
        status <- httr::status_code(res_ind)
        error_info <- httr::http_status(res_ind)$message
        message("HTTP request to api.worldbank.org failed (status ", status, "): ", error_info)
        return(invisible(NULL))
      }
  data_ind <- jsonlite::fromJSON(rawToChar(res_ind$content))
  df_ind <- as.data.frame(data_ind[2])
  df_ind <- df_ind[order(df_ind$date), ]

  longterm$industrial_value_added <- rep(NA, nrow(longterm))
  for (i in seq_len(nrow(df_ind))) {
    longterm$industrial_value_added[i] <- df_ind$value[i]
  }


      res_man <-  tryCatch(
        {httr::GET(paste0("https://api.worldbank.org/v2/country/", country, "/indicator/NV.IND.MANF.ZS?date=", start_year, ":", end_year, "&format=json"))
    },
    error = function(e) {
      message("Error during GET request to api.worldbank.org", e$message,
        "\nAre you connected to the internet?"
      )
      return(NULL)
    }
  )

      if (is.null(res_man)) {
        return(invisible(NULL))
      }

      if (res_man$status_code == 502) {
        message("The World Development Indicator database of the World Bank is currently unreachable. Please try again in a minute.")
        return(invisible(NULL))
      }
      if (httr::http_error(res_man)) {
        status <- httr::status_code(res_man)
        error_info <- httr::http_status(res_man)$message
        message("HTTP request to api.worldbank.org failed (status ", status, "): ", error_info)
        return(invisible(NULL))
      }
  data_man <- jsonlite::fromJSON(rawToChar(res_man$content))
  df_man <- as.data.frame(data_man[2])
  df_man <- df_man[order(df_man$date), ]

  longterm$manufacturing_value_added <- rep(NA, nrow(longterm))
  for (i in seq_len(nrow(df_man))) {
    longterm$manufacturing_value_added[i] <- df_man$value[i]
  }


      res_gro <-  tryCatch(
        { httr::GET(paste0("https://api.worldbank.org/v2/country/", country, "/indicator/NY.GDP.MKTP.KD.ZG?date=", start_year, ":", end_year, "&format=json"))
    },
    error = function(e) {
      message("Error during GET request to api.worldbank.org", e$message,
        "\nAre you connected to the internet?"
      )
      return(NULL)
    }
  )

      if (is.null(res_gro)) {
        return(invisible(NULL))
      }

      if (res_gro$status_code == 502) {
        message("The World Development Indicator database of the World Bank is currently unreachable. Please try again in a minute.")
        return(invisible(NULL))
      }
      if (httr::http_error(res_gro)) {
        status <- httr::status_code(res_gro)
        error_info <- httr::http_status(res_gro)$message
        message("HTTP request to api.worldbank.org failed (status ", status, "): ", error_info)
        return(invisible(NULL))
      }
  data_gro <- jsonlite::fromJSON(rawToChar(res_gro$content))
  df_gro <- as.data.frame(data_gro[2])
  df_gro <- df_gro[order(df_gro$date), ]

  longterm$GDP_growth <- rep(NA, nrow(longterm))
  for (i in seq_len(nrow(df_gro))) {
    longterm$GDP_growth[i] <- df_gro$value[i]
  }

      res_gdp_defl <-
        tryCatch(
          {httr::GET(paste0("https://api.worldbank.org/v2/country/", country, "/indicator/NY.GDP.DEFL.KD.ZG?date=", start_year, ":", end_year, "&format=json"))
    },
    error = function(e) {
      message("Error during GET request to api.worldbank.org", e$message,
        "\nAre you connected to the internet?")
      return(NULL)
    }
  )
      if (is.null(res_gdp_defl)) {
        return(invisible(NULL))
      }

      if (res_gdp_defl$status_code == 502) {
        message("The World Development Indicator database of the World Bank is currently unreachable. Please try again in a minute.")
        return(invisible(NULL))
      }
      if (httr::http_error(res_gdp_defl)) {
        status <- httr::status_code(res_gdp_defl)
        error_info <- httr::http_status(res_gdp_defl)$message
        message("HTTP request to api.worldbank.org failed (status ", status, "): ", error_info)
        return(invisible(NULL))
      }

  data_gdp_defl <- jsonlite::fromJSON(rawToChar(res_gdp_defl$content))
  df_gdp_defl <- as.data.frame(data_gdp_defl[2])
  df_gdp_defl <- df_gdp_defl[order(df_gdp_defl$date), ]

  longterm$GDP_deflator <- rep(NA, nrow(longterm))
  for (i in seq_len(nrow(df_gdp_defl))) {
    longterm$GDP_deflator[i] <- df_gdp_defl$value[i]
  }



      res_serv <-   tryCatch(
        {httr::GET(paste0("https://api.worldbank.org/v2/country/", country, "/indicator/NV.SRV.TOTL.ZS?date=", start_year, ":", end_year, "&format=json"))
    },
    error = function(e) {
      message("Error during GET request to api.worldbank.org", e$message,
        "\nAre you connected to the internet?"
      )
      return(NULL)
    }
  )

      if (is.null(res_serv)) {
        return(invisible(NULL))
      }

      if (res_serv$status_code == 502) {
        message("The World Development Indicator database of the World Bank is currently unreachable. Please try again in a minute.")
        return(invisible(NULL))
      }
      if (httr::http_error(res_serv)) {
        status <- httr::status_code(res_serv)
        error_info <- httr::http_status(res_serv)$message
        message("HTTP request to api.worldbank.org failed (status ", status, "): ", error_info)
        return(invisible(NULL))
      }
  data_serv <- jsonlite::fromJSON(rawToChar(res_serv$content))
  df_serv <- as.data.frame(data_serv[2])
  df_serv <- df_serv[order(df_serv$date), ]


  longterm$service_value_added <- rep(NA, nrow(longterm))
  for (i in seq_len(nrow(df_serv))) {
    longterm$service_value_added[i] <- df_serv$value[i]
  }



      res_gni <-   tryCatch(
        {httr::GET(paste0("https://api.worldbank.org/v2/country/", country, "/indicator/NY.GNP.MKTP.KD?date=", start_year, ":", end_year, "&format=json"))
    },
    error = function(e) {
      message("Error during GET request to api.worldbank.org", e$message,
        "\nAre you connected to the internet?"
      )
      return(NULL)
    }
  )
      if (is.null(res_gni)) {
        return(invisible(NULL))
      }

      if (res_gni$status_code == 502) {
        message("The World Development Indicator database of the World Bank is currently unreachable. Please try again in a minute.")
        return(invisible(NULL))
      }
      if (httr::http_error(res_gni)) {
        status <- httr::status_code(res_gni)
        error_info <- httr::http_status(res_gni)$message
        message("HTTP request to api.worldbank.org failed (status ", status, "): ", error_info)
        return(invisible(NULL))
      }
  data_gni <- jsonlite::fromJSON(rawToChar(res_gni$content))
  df_gni <- as.data.frame(data_gni[2])
  df_gni <- df_gni[order(df_gni$date), ]

  longterm$GNI <- rep(NA, nrow(longterm))
  for (i in seq_len(nrow(df_gni))) {
    longterm$GNI[i] <- df_gni$value[i]
  }



      res_hou <-  tryCatch(
        { httr::GET(paste0("https://api.worldbank.org/v2/country/", country, "/indicator/NE.CON.PRVT.ZS?date=", start_year, ":", end_year, "&format=json"))

    },
    error = function(e) {
      message("Error during GET request to api.worldbank.org", e$message,
        "\nAre you connected to the internet?"
      )
      return(NULL)
    }
  )
      if (is.null(res_hou)) {
        return(invisible(NULL))
      }

      if (res_hou$status_code == 502) {
        message("The World Development Indicator database of the World Bank is currently unreachable. Please try again in a minute.")
        return(invisible(NULL))
      }
      if (httr::http_error(res_hou)) {
        status <- httr::status_code(res_hou)
        error_info <- httr::http_status(res_hou)$message
        message("HTTP request to api.worldbank.org failed (status ", status, "): ", error_info)
        return(invisible(NULL))
      }

      data_hou <- jsonlite::fromJSON(rawToChar(res_hou$content))
      df_hou <- as.data.frame(data_hou[2])
      df_hou <- df_hou[order(df_hou$date), ]
  longterm$household_consumption_expenditure <- rep(NA, nrow(longterm))
  for (i in seq_len(nrow(df_hou))) {
    longterm$household_consumption_expenditure[i] <- df_hou$value[i]
  }


      res_rural <-  tryCatch(
        { httr::GET(paste0("https://api.worldbank.org/v2/country/", country, "/indicator/SP.RUR.TOTL?date=", start_year, ":", end_year, "&format=json"))
    },
    error = function(e) {
      message("Error during GET request to api.worldbank.org", e$message,
        "\nAre you connected to the internet?"
      )
      return(NULL)
    }
  )
      if (is.null(res_rural)) {
        return(invisible(NULL))
      }

      if (res_rural$status_code == 502) {
        message("The World Development Indicator database of the World Bank is currently unreachable. Please try again in a minute.")
        return(invisible(NULL))
      }
      if (httr::http_error(res_rural)) {
        status <- httr::status_code(res_rural)
        error_info <- httr::http_status(res_rural)$message
        message("HTTP request to api.worldbank.org failed (status ", status, "): ", error_info)
        return(invisible(NULL))
      }
  data_rural <- jsonlite::fromJSON(rawToChar(res_rural$content))
  df_rural <- as.data.frame(data_rural[2])
  df_rural <- df_rural[order(df_rural$date), ]


  longterm$rural_population <- rep(NA, nrow(longterm))
  for (i in seq_len(nrow(df_serv))) {
    longterm$rural_population[i] <- df_rural$value[i]
  }

  # consumer_price_inflation_pct= httr::GET(paste0("https://api.worldbank.org/v2/country/",country,"/indicator/FP.CPI.TOTL.ZG?date=",start_year,":",end_year,"&format=json"))
  #
  # data_cpi=jsonlite::fromJSON(rawToChar(consumer_price_inflation_pct$content))
  # df_cpi<- as.data.frame(data_cpi[2])
  # df_cpi<- df_cpi[order(df_cpi$date),]
  #
  # longterm$consumer_price_inflation_pct <- df_cpi$value




  return(longterm)
}

