% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tabu.operator.R
\name{tabu.operator}
\alias{tabu.operator}
\title{Tabu search operator for model selection}
\usage{
tabu.operator(
  dat,
  param_table = NULL,
  start.mod = NULL,
  search.space = c("ivbase", "oralbase"),
  no.cores = NULL,
  tabu.control = tabuControl(),
  penalty.control = penaltyControl(),
  precomputed_results_file = NULL,
  foldername = NULL,
  filename = "test",
  seed = 1234,
  .modEnv = NULL,
  verbose = TRUE,
  ...
)
}
\arguments{
\item{dat}{A data frame containing pharmacokinetic data in standard
nlmixr2 format, including "ID", "TIME", "EVID", and "DV", and may include
additional columns.}

\item{param_table}{Optional data frame of initial parameter estimates. If NULL,
the table is generated by \code{auto_param_table()}.}

\item{start.mod}{A named integer vector specifying the starting model
code. If NULL, a base model is generated using \code{base_model()}.}

\item{search.space}{Character, one of "ivbase" or "oralbase".
Default is "ivbase".}

\item{no.cores}{Integer. Number of CPU cores to use. If NULL, uses
\code{rxode2::getRxThreads()}.}

\item{tabu.control}{A list of Tabu Search control parameters from
\code{\link{tabuControl}}:
\describe{
\item{tenure}{Integer. Number of iterations a move remains tabu.}
\item{niter}{Integer. Maximum number of search iterations.}
\item{start.point}{Optional initial model code vector.}
\item{aspiration}{Logical. If TRUE, allows aspiration criterion.}
\item{policy}{Character. Tabu restriction policy: \code{move} or
\code{attribute}. See Details.}
\item{nsize}{Optional integer. Maximum number of neighbors randomly
sampled from the full neighborhood (candidate list strategy).}
}}

\item{penalty.control}{A list of penalty control parameters defined by
\code{penaltyControl()}, specifying penalty values used for model diagnostics
during fitness evaluation.}

\item{precomputed_results_file}{Optional path to a CSV file of previously computed
model results used for caching.}

\item{foldername}{Character string specifying the folder name for storing
intermediate results. If \code{NULL} (default), \code{tempdir()}
is used for temporary storage. If specified, a cache directory
is created in the current working directory.}

\item{filename}{Optional character string used as a prefix for output files.
Defaults to "test".}

\item{seed}{Integer. Random seed controlling the random sampling steps of the
tabu operator for reproducible runs. Default is 1234.}

\item{.modEnv}{Environment for storing intermediate results. If \code{NULL},
a new environment is created.}

\item{verbose}{Logical. If TRUE, print progress messages.}

\item{...}{Additional arguments passed to \code{mod.run()}.}
}
\value{
An object of class \code{tabuOperatorResult}, containing:
\item{Final Selected Code}{Vector representation of the best model.}
\item{Final Selected Model Name}{Selected best model (human-readable).}
\item{Model Run History}{Data frame of all model evaluations with fitness values.}
\item{Search History}{List with iteration-level history:
\code{starting.points.history}, \code{local.best.history},
\code{tabu.elements.history}, \code{neighbors.history}.}
}
\description{
Performs tabu search to explore the pharmacometric model space and
identify the best-performing model. Supports both IV and Oral search spaces.
}
\details{
This function implements tabu search for pharmacometric model structure
optimization. Models are encoded as bit vectors representing structural and
statistical components.

\strong{Neighbor Generation and Validation}

Each iteration generates neighbors by one-bit flips, then validates them
using \code{validStringcat()}. The algorithm maintains both:
\itemize{
\item \code{neighbors_orig}: Original neighbors (before validation) →
used to detect intended moves
\item \code{neighbors_val}: Validated neighbors (after validation) →
used for fitness evaluation
}

This separation is critical because validation may introduce secondary changes.
For example, changing \code{no.cmpt} from 2 to 3 might force \code{eta.vp = 0}
to maintain model legality. The tabu list records only the intended change
(\code{no.cmpt}), not validation side effects (\code{eta.vp}).

\strong{Tabu List Policies}

Two restriction policies are available:
\itemize{
\item \code{"move"}: Forbids specific transitions (e.g., \code{no.cmpt: 2 -> 3}
and \code{3 -> 2}). Stores both forward and reverse moves.
\item \code{"attribute"}: Forbids setting a parameter to a specific value
regardless of origin (e.g., any move setting \code{no.cmpt = 3}).
}

Both policies use the same data structure (\code{element}, \code{from},
\code{to}, \code{tabu.iteration.left}). For attribute-based policy, the
\code{from} field is stored for record-keeping but only \code{to} is used
in tabu checking.

\strong{Aspiration Criterion}
When enabled, tabu moves are allowed if they produce a solution better than
the global best.

\strong{Perturbation}
If the search returns to a previous starting point (cycling detected), a
2-bit perturbation is applied to escape the local region.
}
\examples{
\donttest{
# Example usage with phenotype dataset
outs <- tabu.operator(
  dat = pheno_sd,
  param_table = NULL,
  search.space = "ivbase",
  tabu.control = tabuControl(),
  saem.control = nlmixr2est::saemControl(
    seed = 1234,
    nBurn = 200,
    nEm   = 300,
    logLik = TRUE
  )
)
print(outs)
}

}
\seealso{
\code{\link{tabuControl}} for control parameters,
\code{\link{detect_move}} for move detection,
\code{\link{is_move_tabu}} for tabu checking,
\code{\link{perturb_2bit}} for perturbation
}
\author{
Zhonghui Huang
}
