#' Control Parameters for Tabu Search
#'
#' Creates a list of control settings for the \code{tabu.operator} function.
#'
#' @param tenure Integer. Number of iterations a move remains tabu.
#' @param niter Integer. Maximum number of search iterations.
#' @param aspiration Logical. Whether to apply the aspiration criterion.
#'   If TRUE, tabu moves are allowed if they yield a solution strictly better
#'   than the global best found so far.
#' @param nsize Optional integer. If not NULL, restricts neighborhood sear
#'   to a random subset of this size (candidate list strategy).
#' @param policy Character. Type of tabu restriction:
#'   \itemize{
#'     \item \code{"attribute"} — forbid revisiting a variable value (default).
#'     \item \code{"move"} — forbid only specific from–to transitions.
#'   }
#'
#' @return A named list containing all tabu control parameters.
#'
#' @author Zhonghui Huang
#'
#' @examples
#' tabuControl()
#'
#' @export
tabuControl <- function(tenure = 3,
                        niter = 20,
                        aspiration = TRUE,
                        nsize = NULL,
                        policy = "attribute") {
  list(
    tenure = tenure,
    niter = niter,
    aspiration = aspiration,
    nsize = nsize,
    policy = policy
  )
}

#' Tabu search operator for model selection
#'
#' Performs tabu search to explore the pharmacometric model space and
#' identify the best-performing model. Supports both IV and Oral search spaces.
#'
#' @param dat A data frame containing pharmacokinetic data in standard
#'   nlmixr2 format, including "ID", "TIME", "EVID", and "DV", and may include
#'   additional columns.
#' @param param_table Optional data frame of initial parameter estimates. If NULL,
#'   the table is generated by \code{auto_param_table()}.
#' @param start.mod A named integer vector specifying the starting model
#'   code. If NULL, a base model is generated using \code{base_model()}.
#' @param search.space Character, one of "ivbase" or "oralbase".
#'   Default is "ivbase".
#' @param no.cores Integer. Number of CPU cores to use. If NULL, uses
#'   \code{rxode2::getRxThreads()}.
#' @param tabu.control A list of Tabu Search control parameters from
#'   \code{\link{tabuControl}}:
#'   \describe{
#'     \item{tenure}{Integer. Number of iterations a move remains tabu.}
#'     \item{niter}{Integer. Maximum number of search iterations.}
#'     \item{start.point}{Optional initial model code vector.}
#'     \item{aspiration}{Logical. If TRUE, allows aspiration criterion.}
#'     \item{policy}{Character. Tabu restriction policy: \code{move} or
#'       \code{attribute}. See Details.}
#'     \item{nsize}{Optional integer. Maximum number of neighbors randomly
#'       sampled from the full neighborhood (candidate list strategy).}
#'   }
#' @param penalty.control A list of penalty control parameters defined by
#'   \code{penaltyControl()}, specifying penalty values used for model diagnostics
#'   during fitness evaluation.
#' @param precomputed_results_file Optional path to a CSV file of previously computed
#'   model results used for caching.
#' @param foldername Character string specifying the folder name for storing
#'   intermediate results. If \code{NULL} (default), \code{tempdir()}
#'   is used for temporary storage. If specified, a cache directory
#'   is created in the current working directory.
#' @param filename Optional character string used as a prefix for output files.
#'   Defaults to "test".
#' @param seed Integer. Random seed controlling the random sampling steps of the
#'   tabu operator for reproducible runs. Default is 1234.
#' @param .modEnv Environment for storing intermediate results. If \code{NULL},
#'   a new environment is created.
#' @param verbose Logical. If TRUE, print progress messages.
#' @param ... Additional arguments passed to \code{mod.run()}.
#'
#' @return An object of class \code{tabuOperatorResult}, containing:
#'   \item{Final Selected Code}{Vector representation of the best model.}
#'   \item{Final Selected Model Name}{Selected best model (human-readable).}
#'   \item{Model Run History}{Data frame of all model evaluations with fitness values.}
#'   \item{Search History}{List with iteration-level history:
#'         \code{starting.points.history}, \code{local.best.history},
#'         \code{tabu.elements.history}, \code{neighbors.history}.}
#'
#' @details
#' This function implements tabu search for pharmacometric model structure
#' optimization. Models are encoded as bit vectors representing structural and
#' statistical components.
#'
#' \strong{Neighbor Generation and Validation}
#'
#' Each iteration generates neighbors by one-bit flips, then validates them
#' using \code{validStringcat()}. The algorithm maintains both:
#' \itemize{
#'   \item \code{neighbors_orig}: Original neighbors (before validation) →
#'         used to detect intended moves
#'   \item \code{neighbors_val}: Validated neighbors (after validation) →
#'         used for fitness evaluation
#' }
#'
#' This separation is critical because validation may introduce secondary changes.
#' For example, changing \code{no.cmpt} from 2 to 3 might force \code{eta.vp = 0}
#' to maintain model legality. The tabu list records only the intended change
#' (\code{no.cmpt}), not validation side effects (\code{eta.vp}).
#'
#' \strong{Tabu List Policies}
#'
#' Two restriction policies are available:
#' \itemize{
#'   \item \code{"move"}: Forbids specific transitions (e.g., \code{no.cmpt: 2 -> 3}
#'         and \code{3 -> 2}). Stores both forward and reverse moves.
#'   \item \code{"attribute"}: Forbids setting a parameter to a specific value
#'         regardless of origin (e.g., any move setting \code{no.cmpt = 3}).
#' }
#'
#' Both policies use the same data structure (\code{element}, \code{from},
#' \code{to}, \code{tabu.iteration.left}). For attribute-based policy, the
#' \code{from} field is stored for record-keeping but only \code{to} is used
#' in tabu checking.
#'
#' \strong{Aspiration Criterion}
#' When enabled, tabu moves are allowed if they produce a solution better than
#' the global best.
#'
#' \strong{Perturbation}
#' If the search returns to a previous starting point (cycling detected), a
#' 2-bit perturbation is applied to escape the local region.
#'
#' @author Zhonghui Huang
#'
#' @examples
#' \donttest{
#' # Example usage with phenotype dataset
#' outs <- tabu.operator(
#'   dat = pheno_sd,
#'   param_table = NULL,
#'   search.space = "ivbase",
#'   tabu.control = tabuControl(),
#'   saem.control = nlmixr2est::saemControl(
#'     seed = 1234,
#'     nBurn = 200,
#'     nEm   = 300,
#'     logLik = TRUE
#'   )
#' )
#' print(outs)
#' }
#'
#' @seealso
#' \code{\link{tabuControl}} for control parameters,
#' \code{\link{detect_move}} for move detection,
#' \code{\link{is_move_tabu}} for tabu checking,
#' \code{\link{perturb_2bit}} for perturbation
#'
#' @export
tabu.operator <- function(dat,
                          param_table = NULL,
                          start.mod = NULL,
                          search.space = c("ivbase", "oralbase"),
                          no.cores = NULL,
                          tabu.control = tabuControl(),
                          penalty.control = penaltyControl(),
                          precomputed_results_file = NULL,
                          foldername = NULL,
                          filename = "test",
                          seed = 1234,
                          .modEnv = NULL,
                          verbose = TRUE,
                          ...) {
  if (!is.null(.modEnv)) {
    if (!is.environment(.modEnv)) {
      stop("`.modEnv` must be an environment", call. = FALSE)
    }
  } else {
    .modEnv <- new.env(parent = emptyenv())
  }

  # Ensure essential keys exist in .modEnv
  if (is.null(.modEnv$modi))
    .modEnv$modi <- 1L
  if (is.null(.modEnv$r))
    .modEnv$r <- 1L
  if (is.null(.modEnv$Store.all))
    .modEnv$Store.all <- NULL
  if (is.null(.modEnv$precomputed_cache_loaded))
    .modEnv$precomputed_cache_loaded <- FALSE
  if (is.null(.modEnv$precomputed_results))
    .modEnv$precomputed_results <- NULL
  if (is.null(.modEnv$param_table))
    .modEnv$param_table <- NULL
  if (is.null(.modEnv$saem.control))
    .modEnv$saem.control <- NULL

  if (is.null(no.cores)) {
    no.cores <- rxode2::getRxThreads()
  }

  if (is.null(foldername) || !nzchar(foldername)) {
    # foldername <-
    #   paste0("tabuCache_", filename, "_", digest::digest(dat))
    foldername <- tempdir()
  }
  if (!dir.exists(foldername)) {
    dir.create(foldername,
               showWarnings = FALSE,
               recursive = TRUE)
  }
  # Set random seed
  if (!is.numeric(seed) ||
      length(seed) != 1 || is.na(seed) || is.infinite(seed)) {
    stop("`seed` must be a single finite numeric value", call. = FALSE)
  }
  if (abs(seed) > .Machine$integer.max) {
    stop("`seed` exceeds valid integer range", call. = FALSE)
  }
  seed <- as.integer(seed)
  withr::local_rng_version("4.2.0")
  withr::local_seed(seed)

  # Initial estimates
  if (!is.null(param_table)) {
    param_table_use <- param_table
  } else if (!is.null(.modEnv$param_table)) {
    param_table_use <- .modEnv$param_table
  } else {
    param_table_use <- auto_param_table(
      dat = dat,
      nlmixr2autoinits = TRUE,
      foldername = foldername,
      filename = filename,
      out.inits = TRUE
    )
    .modEnv$param_table <- param_table_use
  }

  param_table <- param_table_use

  # ACO does not support a custom search space.
  custom_config <- NULL

  search.space <-
    match.arg(search.space, choices = c("ivbase", "oralbase"))
  # if (identical(search.space, "custom")) {
  #   stop(
  #     "aco currently does not support search.space = 'custom'. Use 'ivbase' or 'oralbase'.",
  #     call. = FALSE
  #   )
  # }
  cfg <- spaceConfig(search.space)

  bit.names <- if (identical(search.space, "custom")) {
    if (is.null(custom_config) || is.null(custom_config$params)) {
      stop("custom_config$params must be provided when search.space = 'custom'",
           call. = FALSE)
    }
    custom_config$params
  } else {
    cfg$params
  }

  if (!is.null(start.mod)) {
    string_vec <- start.mod
  } else {
    string_vec <-
      base_model(search.space = search.space)
    string_vec["no.cmpt"] <- 2 # enforce 2-compartment default
  }

  # --- Initialize histories ---
  local.best.history <- list()
  starting.points.history <- list()
  tabu.elements.history <- list()
  neighbors.history <- list()
  tabu.elements.all <- NULL
  prev_string <- NULL
  globalbest <- NULL

  progressr::with_progress({
    p <- progressr::progressor(steps = tabu.control$niter)

    for (tabu.iter in 1:tabu.control$niter) {
      # Define current starting point
      if (tabu.iter == 1) {
        start_string <- string_vec
      } else {
        start_string <- current_string
      }
      starting.points.history[[tabu.iter]] <- start_string

      # Generate neighbors (original + validated)
      neighbors_list <-
        generate_neighbors_df(start_string, search.space = search.space)
      neighbors_orig <-
        neighbors_list$original_neighbors   # pre-validation
      neighbors_val  <-
        neighbors_list$validated_neighbors  # post-validation (legal models)

      neighbors_val <- dplyr::distinct(neighbors_val)[, bit.names]

      neighbors_eval <- list()
      aspiration_candidates <- list()
      move_records <- list()  # store primary moves

      if (!is.null(tabu.elements.all) &&
          nrow(tabu.elements.all) > 0) {
        for (row in 1:nrow(neighbors_val)) {
          # detect primary move using original neighbor
          move <- detect_move(start_string,
                              neighbors_val[row, ],
                              original_neighbor = neighbors_orig[row, ])

          if (is_move_tabu(
            move = move,
            tabu_list = tabu.elements.all,
            policy = tabu.control$policy
          )) {
            # tabu move
            if (tabu.control$aspiration) {
              aspiration_candidates[[length(aspiration_candidates) + 1]] <-
                neighbors_val[row, ]
            }
          } else {
            # non-tabu -> keep
            neighbors_eval[[length(neighbors_eval) + 1]] <-
              neighbors_val[row, ]
            move_records[[length(move_records) + 1]] <- move$element
          }
        }
      } else {
        # if no tabu elements, all neighbors are valid
        neighbors_eval <-
          split(neighbors_val, seq_len(nrow(neighbors_val)))
        # detect moves for all neighbors
        move_records <-
          lapply(seq_len(nrow(neighbors_val)), function(row) {
            move <- detect_move(start_string,
                                neighbors_val[row,],
                                original_neighbor = neighbors_orig[row,])
            move$element
          })
      }
      # Convert lists back to data frames
      if (length(neighbors_eval) > 0) {
        neighbors_eval <- do.call(rbind, neighbors_eval)
        neighbors_eval$move.element <- unlist(move_records)
      } else {
        neighbors_eval <- data.frame()
      }

      if (length(aspiration_candidates) > 0) {
        aspiration_candidates <- do.call(rbind, aspiration_candidates)
      } else {
        aspiration_candidates <- data.frame()
      }

      # save (store validated neighbors in history)
      neighbors.history[[tabu.iter]] <- neighbors_val
      # Evaluate neighbors
      if (nrow(neighbors_eval) > 0) {
        neighbors_eval$fitness <- vapply(seq_len(nrow(neighbors_eval)),
                                         function(k) {
                                           string_vec <- as.numeric(neighbors_eval[k, bit.names])
                                           result <- try(mod.run(
                                             string               = string_vec,
                                             dat                  = dat,
                                             search.space         = search.space,
                                             no.cores             = no.cores,
                                             param_table          = param_table,
                                             precomputed_results_file = precomputed_results_file,
                                             filename             = filename,
                                             foldername           = foldername,
                                             .modEnv              = .modEnv,
                                             ...
                                           ),
                                           silent = TRUE)
                                           if (is.numeric(result) &&
                                               length(result) == 1) {
                                             result
                                           } else {
                                             NA_real_
                                           }
                                         },
                                         numeric(1))
        # Aspiration criterion check
        if (tabu.control$aspiration) {
          best.fitness <- min(.modEnv$Store.all$fitness, na.rm = TRUE)
          aspiration_candidates <-
            neighbors_eval[neighbors_eval$fitness < best.fitness,]
        } else {
          aspiration_candidates <- data.frame()
        }
      } else {
        neighbors_eval <- data.frame()
        aspiration_candidates <- data.frame()
      }

      # Update local best
      if (nrow(neighbors_eval) > 0) {
        localbest <-
          neighbors_eval[which.min(neighbors_eval$fitness), , drop = FALSE]
      } else {
        localbest <- start_string  # fallback, in case no neighbors
      }
      local.best.history[[tabu.iter]] <- localbest

      if (tabu.iter == 1) {
        prev_string <- string_vec      # starting point
      } else {
        prev_string <- start_string
      }
      current_string <- localbest[1, bit.names]

      # Update current solution with neighbor or perturbation
      has_been_start <-
        any(vapply(starting.points.history, function(hist) {
          all(hist == current_string)
        }, logical(1)))

      if (has_been_start) {
        if (verbose) {
          message(
            "Iteration ",
            tabu.iter,
            ": candidate already used as a starting point. Applying 2-bit perturbation to avoid cycling."
          )
        }
        perturb <- perturb_2bit(prev_string, search.space)
        current_string <- perturb$validated_neighbor

        tabu.elements <- data.frame(
          tabu.num = tabu.iter,
          element  = "perturbation",
          # not in the tabulist
          from     = NA,
          to       = NA,
          tabu.iteration.left = 0,
          stringsAsFactors = FALSE
        )
      } else {
        # Normal neighbor move update tabu as usual
        idx <-
          match(
            paste0(as.numeric(current_string), collapse = "_"),
            apply(neighbors_val[, bit.names], 1, function(x)
              paste0(as.numeric(x), collapse = "_"))
          )

        if (!is.na(idx)) {
          # Found the matching original neighbor and record the true move
          move <- detect_move(start_string,
                              new_string        = current_string,
                              original_neighbor = neighbors_orig[idx, ])
        } else {
          # Fallback: in rare cases where match fails
          move <- detect_move(start_string,
                              new_string        = current_string,
                              original_neighbor = current_string)
        }

        if (tabu.control$policy == "move") {
          # Move-based tabu:
          # Store both forward and reverse moves (e.g., 2 to 3 and 3 to 2)
          tabu.elements <- rbind(
            data.frame(
              tabu.num = tabu.iter,
              element  = move$element,
              from     = unname(move$from),
              to       = unname(move$to),
              tabu.iteration.left = tabu.control$tenure,
              stringsAsFactors = FALSE
            ),
            data.frame(
              tabu.num = tabu.iter,
              element  = move$element,
              from     = unname(move$to),
              # reverse move
              to       = unname(move$from),
              # reverse move
              tabu.iteration.left = tabu.control$tenure,
              stringsAsFactors = FALSE
            )
          )
        } else if (tabu.control$policy == "attribute") {
          # Attribute-based tabu:
          # Store only the target value (e.g., "element = no.cmpt, to = 3")
          # This forbids any move that sets the element to this value.
          tabu.elements <- data.frame(
            tabu.num = tabu.iter,
            element  = move$element,
            from     = unname(move$from),
            to       = unname(move$to),
            tabu.iteration.left = tabu.control$tenure,
            stringsAsFactors = FALSE
          )
        }
      }

      if (!is.null(tabu.elements.all)) {
        tabu.elements.all$tabu.iteration.left <-
          tabu.elements.all$tabu.iteration.left - 1
      }

      tabu.elements.all <- rbind(tabu.elements.all, tabu.elements)
      tabu.elements.all <-
        tabu.elements.all[tabu.elements.all$tabu.iteration.left > 0,]

      rownames(tabu.elements.all) <- NULL
      tabu.elements.history[[tabu.iter]] <- tabu.elements.all

      p(sprintf("Tabu iteration %d / %d", tabu.iter, tabu.control$niter))
    }
  })

  # Final output (Tabu Search)
  localbestf <-
    .modEnv$Store.all[.modEnv$Store.all$fitness == min(.modEnv$Store.all$fitness), ][1, ]
  best_model_code <- as.numeric(localbestf[, bit.names])
  names(best_model_code) <- bit.names
  best_model_name <- parseName(modcode = best_model_code,
                               search.space  = search.space)

  out <- new.env(parent = emptyenv())
  class(out) <- "tabuOperatorResult"
  out[["Final Selected Code"]] <- best_model_code
  out[["Final Selected Model Name"]] <- best_model_name
  out[["Model Run History"]] <-
    as.data.frame(.modEnv$Store.all, stringsAsFactors = FALSE)
  out[["Search History"]] <- list(
    starting.points.history = starting.points.history,
    local.best.history      = local.best.history,
    tabu.elements.history   = tabu.elements.history,
    neighbors.history       = neighbors.history
  )
  return(out)

}


#' Print method for tabu operator results
#'
#' Print tabu operator results.
#'
#' @param x A "tabuOperatorResult" object.
#' @param ... Additional arguments (currently ignored).
#'
#' @return Invisibly returns x.
#'
#' @seealso \code{\link{tabu.operator}}
#'
#' @export
print.tabuOperatorResult <- function(x, ...) {
  # Print final selected model code
  cat(crayon::green$bold("\n=== Final Selected Model Code (Tabu Search) ===\n"))
  print(x$`Final Selected Code`)

  # Print final selected model name
  cat(crayon::green$bold("\n=== Final Selected Model Name (Tabu Search) ===\n"))
  cat(x$`Final Selected Model Name`, "\n")

  invisible(x)
}
