#' Create control parameters for the ACO algorithm
#'
#' Creates a list of control settings for the \code{aco.operator} function.
#'
#' @param nants Integer. Number of ants (candidate solutions) generated
#'   at each iteration. Defaults to 15.
#' @param niter Integer. Maximum number of ACO iterations. Defaults to 20.
#' @param Q A positive numeric value. Pheromone scaling constant controlling the
#'   amount of pheromone deposited by high-quality solutions during each
#'   iteration. Defaults to 1.
#' @param rho Numeric in (0, 1). Pheromone evaporation rate. Higher values
#'   increase evaporation, encouraging exploration. Defaults to 0.5.
#' @param phi0 A non-negative numeric value. Initial pheromone value assigned
#' to all nodes at the start of the search. Defaults to 2.
#' @param phi_min A non-negative numeric value. Lower bound for pheromone values, preventing
#'   premature convergence. Defaults to 1.
#' @param phi_max A non-negative numeric value. Upper bound for pheromone values, limiting
#'   excessive reinforcement. Defaults to Inf.
#' @param alpha A non-negative numeric value. Exponent controlling the influence
#'   of pheromone values on the probability of selecting a component during
#'   solution construction. Defaults to 1.
#' @param elite Numeric. Elitism rate between 0 and 1. Specifies the proportion
#' of elite ants whose solutions are preserved and directly propagated to the
#' next iteration. Defaults to 0.
#' @param prob_min Numeric. Minimum probability floor between 0 and 1.
#'   Applied during solution construction to avoid zero-probability choices.
#'   Defaults to 0.2.
#' @param diff_tol Numeric. Significance difference threshold used for ranking.
#'   Values within this threshold are considered equal and receive the same rank.
#'   Default is 1.
#' @return A named list containing all ACO control parameters.
#'
#' @author Zhonghui Huang
#'
#' @examples
#' acoControl()
#'
#' @export
acoControl <- function(nants = 15,
                       niter = 20,
                       Q = 1,
                       rho = 0.5,
                       phi0 = 2,
                       phi_min = 1,
                       phi_max = Inf,
                       alpha = 1,
                       elite = 0,
                       prob_min = 0.2,
                       diff_tol = 1) {
  list(
    nants = nants,
    niter = niter,
    Q = Q,
    rho = rho,
    phi0 = phi0,
    phi_min = phi_min,
    phi_max = phi_max,
    alpha = alpha,
    elite = elite,
    prob_min = prob_min,
    diff_tol = diff_tol
  )
}

#' ACO operator for model selection
#'
#' Implements an ant colony optimization algorithm to explore
#' model space and identify the best-performing model given pre-defined fitness
#' function.
#'
#' @param dat A data frame containing pharmacokinetic data in standard
#'   nlmixr2 format, including "ID", "TIME", "EVID", and "DV", and may include
#'   additional columns.
#' @param param_table Optional data frame of initial parameter estimates. If NULL,
#'   the table is generated by \code{auto_param_table()}.
#' @param search.space Character, one of "ivbase" or "oralbase".
#'   Default is "ivbase".
#' @param no.cores Integer. Number of CPU cores to use. If NULL, uses
#'   \code{rxode2::getRxThreads()}.
#' @param aco.control A list of ACO control parameters defined by
#'   \code{acoControl()}. Includes:
#'   \itemize{
#'     \item nants - number of ants per iteration.
#'     \item niter - maximum number of iterations.
#'     \item rho - pheromone evaporation rate.
#'     \item phi0 - initial pheromone level.
#'     \item phi_min, phi_max - bounds for pheromone levels.
#'     \item alpha - pheromone weight exponent.
#'     \item elite - proportion of best solutions preserved each iteration.
#'     \item prob.min - minimum sampling probability.
#'     \item diff_tol - threshold for significant fitness difference.
#'   }
#' @param penalty.control A list of penalty control parameters defined by
#'   \code{penaltyControl()}, specifying penalty values used for model diagnostics
#'   during fitness evaluation.
#' @param precomputed_results_file Optional path to a CSV file of previously computed
#'   model results used for caching.
#' @param foldername Character string specifying the folder name for storing
#'   intermediate results. If \code{NULL} (default), \code{tempdir()}
#'   is used for temporary storage. If specified, a cache directory
#'   is created in the current working directory.
#' @param filename Optional character string used as a prefix for output files.
#'   Defaults to "test".
#' @param seed Integer. Random seed controlling the random sampling steps of the
#'   ant colony optimization operator for reproducible runs. Default is 1234.
#' @param ... Additional arguments passed to the underlying model fitting function
#'   (e.g., \code{mod.run()}).
#' @param .modEnv Optional environment used internally to store model indices,
#'   cached parameter tables, and results across steps.
#' @param verbose Logical. If TRUE, print progress messages.
#' @param ... Additional arguments passed to \code{mod.run()}.
#'
#' @details
#' The ACO approach uses a colony of "ants" to stochastically sample models,
#' evaluate their fitness, and update pheromone trails that guide future searches.
#' This iterative process balances exploration of new models with exploitation
#' of promising candidates.
#'
#' @return An object of class \code{"acoOperatorResult"}, containing:
#' \itemize{
#'   \item \code{$`Final Selected Code`} - Vector representation of the best model.
#'   \item \code{$`Final Selected Model Name`} - Human-readable name of the selected model.
#'   \item \code{$`Model Run History`} - Data frame of model runs across iterations.
#'   \item \code{$`Node Run History`} - History of pheromone probabilities for each iteration.
#' }
#'
#' @author Zhonghui Huang
#'
#' @examples
#' \donttest{
#' # Example usage with phenotype dataset
#' outs <- aco.operator(
#'   dat = pheno_sd,
#'   param_table = NULL,
#'   search.space = "ivbase",
#'   aco.control = acoControl(),
#'   saem.control = nlmixr2est::saemControl(
#'     seed = 1234,
#'     nBurn = 200,
#'     nEm   = 300,
#'     logLik = TRUE
#'   )
#' )
#' print(outs)
#' }
#'
#' @seealso
#' \code{\link{acoControl}},
#' \code{\link{penaltyControl}},
#' \code{\link{auto_param_table}},
#' \code{\link{mod.run}},
#' \code{\link{ppkmodGen}}
#'
#' @export
aco.operator <- function(dat,
                         param_table = NULL,
                         search.space = c("ivbase", "oralbase"),
                         no.cores = NULL,
                         aco.control = acoControl(),
                         penalty.control = penaltyControl(),
                         precomputed_results_file = NULL,
                         foldername = NULL,
                         filename = "test",
                         seed = 1234,
                         .modEnv = NULL,
                         verbose = TRUE,
                         ...) {
  if (!is.null(.modEnv)) {
    if (!is.environment(.modEnv)) {
      stop("`.modEnv` must be an environment", call. = FALSE)
    }
    # .modEnv <- get(".modEnv", inherits = TRUE)
  } else {
    .modEnv <- new.env(parent = emptyenv())
  }

  # Ensure essential keys exist in .modEnv
  if (is.null(.modEnv$modi))
    .modEnv$modi <- 1L
  if (is.null(.modEnv$r))
    .modEnv$r <- 1L
  if (is.null(.modEnv$Store.all))
    .modEnv$Store.all <- NULL
  if (is.null(.modEnv$precomputed_cache_loaded))
    .modEnv$precomputed_cache_loaded <- FALSE
  if (is.null(.modEnv$precomputed_results))
    .modEnv$precomputed_results <- NULL
  if (is.null(.modEnv$param_table))
    .modEnv$param_table <- NULL
  if (is.null(.modEnv$saem.control))
    .modEnv$saem.control <- NULL

  if (is.null(no.cores)) {
    no.cores <- rxode2::getRxThreads()
  }

  if (is.null(foldername) || !nzchar(foldername)) {
    # foldername <-
    #   paste0("acoCache_", filename, "_", digest::digest(dat))
    foldername <- tempdir()
  }
  if (!dir.exists(foldername)) {
    dir.create(foldername,
               showWarnings = FALSE,
               recursive = TRUE)
  }
  # Set random seed
  if (!is.numeric(seed) ||
      length(seed) != 1 || is.na(seed) || is.infinite(seed)) {
    stop("`seed` must be a single finite numeric value", call. = FALSE)
  }
  if (abs(seed) > .Machine$integer.max) {
    stop("`seed` exceeds valid integer range", call. = FALSE)
  }
  seed <- as.integer(seed)
  withr::local_rng_version("4.2.0")
  withr::local_seed(seed)

  # Initial estimates
  if (!is.null(param_table)) {
    param_table_use <- param_table
  } else if (!is.null(.modEnv$param_table)) {
    param_table_use <- .modEnv$param_table
  } else {
    param_table_use <- auto_param_table(
      dat = dat,
      nlmixr2autoinits = TRUE,
      foldername = foldername,
      filename = filename,
      out.inits = TRUE
    )
    .modEnv$param_table <- param_table_use
  }
  param_table <- param_table_use

  # ACO does not support a custom search space.
  custom_config <- NULL

  search.space <-
    match.arg(search.space, choices = c("ivbase", "oralbase"))
  # if (identical(search.space, "custom")) {
  #   stop(
  #     "aco currently does not support search.space = 'custom'. Use 'ivbase' or 'oralbase'.",
  #     call. = FALSE
  #   )
  # }
  cfg <- spaceConfig(search.space)

  bit.names <- if (identical(search.space, "custom")) {
    if (is.null(custom_config) || is.null(custom_config$params)) {
      stop("custom_config$params must be provided when search.space = 'custom'",
           call. = FALSE)
    }
    custom_config$params
  } else {
    cfg$params
  }

  old_handlers <- progressr::handlers()
  old_enable <- getOption("progressr.enable")

  on.exit({
    options(progressr.enable = old_enable)
    if (length(old_handlers) > 0) {
      progressr::handlers(old_handlers)
    } else {
      progressr::handlers(list())
    }
  }, add = TRUE)

  if (isTRUE(verbose)) {
    progressr::handlers("txtprogressbar")
  } else {
    progressr::handlers("void")
  }

  progressr::with_progress({
    p <- progressr::progressor(steps = aco.control$niter)
    for (aco.iter in 1:aco.control$niter) {
      if (aco.iter == 1) {
        nodeslst0 <- initNodeList(search.space = search.space,
                                  phi0 = aco.control$phi0)
        nodeslst.hist <- nodeslst0
        cycle.all.list <- list()

        initial.ants <- createAnts(search.space,
                                   aco.control$nants,
                                   init = TRUE,
                                   nodeslst = nodeslst0)
        initial.ants <- t(vapply(seq_len(ncol(
          initial.ants
        )),  #
        function(i) {
          validStringcat(string = pmax(unname(initial.ants[, i]), 0),
                         search.space = search.space)
        },
        numeric(nrow(
          initial.ants
        ))))

        colnames(initial.ants) <- bit.names
        cycle.all.list[[1]] <- initial.ants

        # Model running and fitness evaluation
        data.ants <- as.data.frame(initial.ants)
        data.ants$fitness <- vapply(seq_len(nrow(data.ants)),
                                    function(k) {
                                      string_vec <- as.vector(initial.ants[k, ])
                                      result <- try(mod.run(
                                        string               = string_vec,
                                        dat                  = dat,
                                        search.space         = search.space,
                                        no.cores             = no.cores,
                                        param_table          = param_table,
                                        precomputed_results_file = precomputed_results_file,
                                        filename             = filename,
                                        foldername           = foldername,
                                        .modEnv              = .modEnv,
                                        verbose =  verbose,
                                        ...
                                      ),
                                      silent = TRUE)
                                      if (is.numeric(result) &&
                                          length(result) == 1)
                                        result
                                      else
                                        NA_real_
                                    },
                                    numeric(1))

        data.ants$round <- aco.iter
        data.ants$mod.no <- seq_len(nrow(data.ants))
        data.ants <-
          data.ants[, c("round", "mod.no", setdiff(names(data.ants), c("round", "mod.no")))]

        fitness_history <- data.frame()
        data.ants$round <-
          aco.iter  # Track which generation each record belongs to
        fitness_history <- rbind(fitness_history, data.ants)

        #  Update pheromone levels and probabilities ---
        nodeslst1 <- phi.calculate(
          r = aco.iter,
          search.space = search.space,
          fitness_history = fitness_history,
          nodeslst.hist = nodeslst.hist,
          Q = aco.control$Q,
          alpha = aco.control$alpha,
          rho = aco.control$rho,
          diff_tol = aco.control$diff_tol,
          phi0 = aco.control$phi0,
          phi_min = aco.control$phi_min,
          phi_max = aco.control$phi_max
        )

        nodeslsts <- p.calculation(nodeslst = nodeslst1,
                                   prob_min = aco.control$prob_min)

        nodeslst.hist <- rbind(nodeslst.hist, nodeslsts)
      }
      else{
        # Identify current best model (elitism)
        bestmodel <-
          fitness_history[fitness_history$fitness == min(fitness_history$fitness),]
        bestmodelcode <- bestmodel[1, bit.names]

        # Generate next generation of ants
        cycle.all <- if (is.null(seed)) {
          createAnts(
            search.space = search.space,
            nants = aco.control$nants,
            init = FALSE,
            nodeslst = nodeslsts,
            custom_config = custom_config
          )
        } else {
          withr::with_seed(seed, {
            createAnts(
              search.space = search.space,
              nants = aco.control$nants,
              init = FALSE,
              nodeslst = nodeslsts,
              custom_config = custom_config
            )
          })
        }
        # Apply elitism: preserve top-performing ants
        no.elitism <-
          max(round(aco.control$nants * aco.control$elite, 0), 1)
        for (loop.no.elitism in 1:no.elitism) {
          cycle.all[, (aco.control$nants - loop.no.elitism + 1)] <-
            as.numeric(bestmodelcode)
        }
        cycle.all.list[[aco.iter]] <- cycle.all

        # Evaluate all ants in current iteration
        data.ants <-
          as.data.frame(t(vapply(seq_len(ncol(cycle.all)),
                                 function(i) {
                                   validStringcat(string = pmax(unname(cycle.all[, i]), 0),
                                                  search.space = search.space)
                                 },
                                 numeric(nrow(cycle.all)))))
        colnames(data.ants) <- bit.names
        data.ants$fitness <- vapply(seq_len(nrow(data.ants)),
                                    function(k) {
                                      string_vec <- as.vector(data.ants[k,])
                                      result <- try(mod.run(
                                        string               = string_vec,
                                        dat                  = dat,
                                        search.space         = search.space,
                                        no.cores             = no.cores,
                                        param_table          = param_table,
                                        precomputed_results_file = precomputed_results_file,
                                        filename             = filename,
                                        foldername           = foldername,
                                        .modEnv              = .modEnv,
                                        verbose = verbose,
                                        ...
                                      ),
                                      silent = TRUE)
                                      if (is.numeric(result) &&
                                          length(result) == 1)
                                        result
                                      else
                                        NA_real_
                                    },
                                    numeric(1))

        # Add round and model IDs
        data.ants$round <- aco.iter
        data.ants$mod.no <- seq_len(nrow(data.ants))
        data.ants <-
          data.ants[, c("round", "mod.no", setdiff(names(data.ants), c("round", "mod.no")))]

        # Append to fitness history
        fitness_history <-
          rbind(fitness_history[, setdiff(names(fitness_history), "allrank")],
                data.ants)

        # Update pheromone trails
        nodeslsts <- phi.calculate(
          r = aco.iter,
          search.space = search.space,
          fitness_history = fitness_history,
          nodeslst.hist = nodeslst.hist,
          Q = aco.control$Q,
          alpha = aco.control$alpha,
          rho = aco.control$rho,
          diff_tol = aco.control$diff_tol,
          phi0 = aco.control$phi0,
          phi_min = aco.control$phi_min,
          phi_max = aco.control$phi_max
        )

        # Update probabilities
        nodeslsts <- p.calculation(nodeslst = nodeslsts,
                                   prob_min = aco.control$prob_min)

        # Extend node history
        nodeslst.hist <- rbind(nodeslst.hist, nodeslsts)
      }

      p(sprintf("ACO iteration %d / %d", aco.iter, aco.control$niter))
    }
  })

  # Final output
  names(bestmodelcode) <- bit.names
  best_model_name <- parseName(modcode = bestmodelcode,
                               search.space  = search.space)

  best_idx <-
    which(apply(.modEnv$Store.all[, names(bestmodelcode)], 1, function(x) {
      all(x == bestmodelcode)
    }))[1]

  best_row <-
    .modEnv$Store.all[best_idx, c("model.num", "round.num")]

  out <- new.env(parent = emptyenv())
  class(out) <- "acoOperatorResult"
  out[["Final Selected Code"]] <- bestmodelcode
  out[["Final Selected Model Name"]] <- best_model_name
  out[["Best Model First Occurrence"]] <- paste0(
    "Best model first appears at model ",
    best_row$model.num,
    ", round ",
    best_row$round.num,
    "."
  )
  out[["Model Run History"]] <-
    as.data.frame(.modEnv$Store.all, stringsAsFactors = FALSE)
  out[["Node Run History"]] <- nodeslst.hist

  return(out)
}

#' Print method for ACO operator results
#'
#' Print ACO operator results.
#'
#' @param x An "acoOperatorResult" object.
#' @param ... Additional arguments (currently ignored).
#'
#' @return Invisibly returns x.
#'
#' @seealso \code{\link{aco.operator}}
#'
#' @author Zhonghui Huang
#'
#' @export
print.acoOperatorResult <- function(x, ...) {
  cat(crayon::green$bold("\n=== Final Selected Model Code (ACO) ===\n"))
  print(x$`Final Selected Code`)

  cat(crayon::green$bold("\n=== Final Selected Model Name (ACO) ===\n"))
  cat(x$`Final Selected Model Name`, "\n")

  invisible(x)
}
