% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/integrate_component.R
\name{integrate_component}
\alias{integrate_component}
\title{Integrate Serial Interval Component Functions for Likelihood Calculation}
\usage{
integrate_component(
  d,
  mu,
  sigma,
  comp,
  dist = c("normal", "gamma"),
  lower = TRUE
)
}
\arguments{
\item{d}{numeric; the index case-to-case (ICC) interval in days for which to calculate the likelihood contribution}

\item{mu}{numeric; the mean of the serial interval distribution in days}

\item{sigma}{numeric; the standard deviation of the serial interval distribution in days}

\item{comp}{integer; the transmission route component number (1 to 7). See Details for component definitions}

\item{dist}{character; the assumed underlying distribution of the serial interval.
Must be either "normal" or "gamma". Defaults to "normal"}

\item{lower}{logical; if \code{TRUE} (default), performs integration using \code{flower} and \code{fupper} functions. If \code{FALSE}, uses \code{f0} function}
}
\value{
numeric; the integrated likelihood value for the specified component and data point. Used in the EM algorithm for serial interval estimation
}
\description{
This function performs numerical integration of serial interval component functions
used in the Vink method for estimating serial interval distributions. It integrates
the probability density functions for different transmission routes over specified
intervals as part of the Expectation-Maximization algorithm.
}
\details{
The function supports two integration modes:
\itemize{
\item \code{lower = TRUE}: Integrates using \code{flower} and \code{fupper} functions
over intervals \code{[d-1, d]} and \code{[d, d+1]} respectively, representing the likelihood
contribution when case occurs at day d
\item \code{lower = FALSE}: Integrates using \code{f0} function over interval \code{[d, d+1]},
representing an alternative likelihood formulation
}

The components represent different transmission routes in outbreak analysis:
\itemize{
\item Component 1: Co-Primary (CP) transmission
\item Components 2+3: Primary-Secondary (PS) transmission
\item Components 4+5: Primary-Tertiary (PT) transmission
\item Components 6+7: Primary-Quaternary (PQ) transmission
}

This function is primarily used internally by \code{si_estim()} as part of the Vink method for estimating serial interval parameters from outbreak data.
}
\examples{
# Basic example with lower integration (default)
# Component 2 represents primary-secondary transmission
integrate_component(d = 15, mu = 12, sigma = 3, comp = 2, dist = "normal", lower = TRUE)

# Upper integration example
integrate_component(d = 15, mu = 12, sigma = 3, comp = 2, dist = "normal", lower = FALSE)

# Using gamma distribution
integrate_component(d = 10, mu = 8, sigma = 2, comp = 1, dist = "gamma", lower = TRUE)

# Component 1 (co-primary transmission) with normal distribution
integrate_component(d = 5, mu = 10, sigma = 3, comp = 1, dist = "normal", lower = TRUE)

# Compare different components for the same data point
d_val <- 20
mu_val <- 15
sigma_val <- 4

# Calculate for components 1, 2, and 4 (different transmission routes)
sapply(c(1, 2, 4), function(comp) {
  integrate_component(d_val, mu_val, sigma_val, comp, "normal", lower = TRUE)
})
}
\references{
Vink MA, Bootsma MCJ, Wallinga J (2014). Serial intervals of respiratory infectious
diseases: A systematic review and analysis. American Journal of Epidemiology,
180(9), 865-875.
}
\seealso{
\code{\link{flower}}, \code{\link{fupper}}, \code{\link{f0}}, \code{\link{si_estim}}
}
