% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/robust.lmer.R
\name{robust.lmer}
\alias{robust.lmer}
\title{Robust Estimation of Multilevel and Linear Mixed-Effects Models}
\usage{
robust.lmer(model, method = c("DAStau", "DASvar"),setting = c("RSEn", "RSEa"),
            digits = 2, p.digits = 3, write = NULL, append = TRUE, check = TRUE,
            output = TRUE)
}
\arguments{
\item{model}{a fitted model of class \code{"lmerMod"} or \code{"lmerModLmerTest"}.}

\item{method}{a character string indicating the method used for estimating
theta and sigma, i.e., \code{"DAStau"} (default) for using numerical
quadrature for computing the consistency factors and \code{"DASvar"}
for computing the consistency factors using a direct approximation.
Note that \code{"DAStau"} is slower than \code{"DASvar"} but
yields more accurate results. However, complex models with
correlated random effects with more than one correlation term
can only estimated using \code{"DASvar"}. See help page of the
\code{rlmer()} function in the R package \code{robustlmm}
for more details.}

\item{setting}{a character string indicating the setting for the parameters
used for computing the robustness weights, i.e., \code{"RSEn"}
(default)  and \code{"RSEa"} for higher asymptotic efficiency
which results in lower robustness. See help page of the
\code{rlmer()} function in the R package \code{robustlmm}
for more details.}

\item{digits}{an integer value indicating the number of decimal places to
be used.}

\item{p.digits}{an integer value indicating the number of decimal places to
be used for displaying \emph{p}-value.}

\item{write}{a character string naming a file for writing the output into
either a text file with file extension \code{".txt"} (e.g.,
\code{"Output.txt"}) or Excel file with file extension
\code{".xlsx"}  (e.g., \code{"Output.xlsx"}). If the file
name does not contain any file extension, an Excel file will
be written.}

\item{append}{logical: if \code{TRUE} (default), output will be appended
to an existing text file with extension \code{.txt} specified
in \code{write}, if \code{FALSE} existing text file will be
overwritten.}

\item{check}{logical: if \code{TRUE} (default), argument specification
is checked.}

\item{output}{logical: if \code{TRUE} (default), output is shown on the console.}
}
\value{
Returns an object of class \code{misty.object}, which is a list with following
entries:
\item{\code{call}}{function call}
\item{\code{type}}{type of analysis}
\item{\code{model}}{object returned from the \code{rlmer} function}
\item{\code{args}}{specification of function arguments}
\item{\code{result}}{list with results, i.e., \code{call} for the the function
call, \code{randeff} for the variance and correlation components, \code{coef}
for the model coefficients, \code{weights} for the robustness weights, and
and \code{converg} for the convergence check, i.e., \code{1} = model converged,
\code{0} = model singular, and \code{-1} model not converged.}
}
\description{
This function estimates a multilevel and linear mixed-effects model based on
a robust estimation method using the \code{rlmer()} function from the
\pkg{robustlmm} package that down-weights observations depending on robustness
weights computed by robustification of the scoring equations and an application
of the Design Adaptive Scale (DAS) approach.
}
\details{
\describe{
\item{\strong{Function specification and Function Arguments}}{The function
\code{rlmer} from the \pkg{robustlmm} package is specified much like the
function \code{lmer} from the \code{lme4 package}, i.e., a formula object
and data frame is specified as the first and second argument. However, the
\code{robust.lmer} function requires a fitted \code{"lmerMod"} or
\code{"lmerModLmerTest"} that is used to re-estimate the model using the
robust estimation method. Note that the function \code{rlmer} provides
the additional arguments \code{rho.e}, \code{rho.b}, \code{rho.sigma.e},
\code{rho.sigma.b}, \code{rel.tol}, \code{max.iter}, \code{verbose},
\code{doFit}, \code{init}, and \code{initTheta} that are not supported by
the \code{robust.lmer} function. See help page of the \code{rlmer()} function
in the R package \code{robustlmm} for more details.}
\item{\strong{Statistical Significance Testing}}{
The function \code{rlmer} from the \pkg{robustlmm} package does not provide
any degrees of freedom or significance values. When specifying a \code{"lmerModLmerTest"}
object for the argument \code{model}, the \code{robust.lmer} function uses the
Satterthwaite or Kenward-Roger degrees of freedom from the \code{"lmerModLmerTest"}
object to compute significance values for the regression coefficients based on
parameter estimates and standard error of the robust multilevel mixed-effects
(see Sleegers et al. (2021).}
}
}
\examples{
\dontrun{
# Load lme4, lmerTest, and misty package
misty::libraries(lme4, lmerTest, misty)

# Load data set "Demo.twolevel" in the lavaan package
data("Demo.twolevel", package = "lavaan")

#----------------------------------------------------------------------------
# Multilevel and Linear Mixed-Effects Model

# Cluster-mean centering, center() from the misty package
Demo.twolevel <- center(Demo.twolevel, x2, type = "CWC", cluster = "cluster")

# Grand-mean centering, center() from the misty package
Demo.twolevel <- center(Demo.twolevel, w1, type = "CGM", cluster = "cluster")

# Estimate two-level mixed-effects model
mod.lmer2 <- lmer(y1 ~ x2.c + w1.c + (1 | cluster), data = Demo.twolevel)

# Example 1a: Default setting
mod.lmer2r <- robust.lmer(mod.lmer2)

# Example 1b: Extract robustness weights
mod.lmer2r$result$weight$iresid
mod.lmer2r$result$weight$iranef

#----------------------------------------------------------------------------
# Write Results

# Example 2a: Write results into a text file
robust.lmer(mod.lmer2, write = "Robust_lmer.txt", output = FALSE)

# Example 2b: Write results into a Excel file
robust.lmer(mod.lmer2, write = "Robust_lmer.xlsx", output = FALSE)
}
}
\references{
Koller, M. (2016). robustlmm: An R Package for Robust Estimation of Linear
Mixed-Effects Models. \emph{Journal of Statistical Software, 75}(6), 1–24.
https://doi.org/10.18637/jss.v075.i06
}
\seealso{
\code{\link{coeff.robust}}, \code{\link{summa}}
}
\author{
Takuya Yanagida \email{takuya.yanagida@univie.ac.at}
}
