% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mgidi.R
\name{mgidi}
\alias{mgidi}
\title{Multitrait Genotype-Ideotype Distance Index}
\usage{
mgidi(
  .data,
  use_data = "blup",
  SI = 15,
  mineval = 1,
  ideotype = NULL,
  weights = NULL,
  use = "complete.obs",
  verbose = TRUE
)
}
\arguments{
\item{.data}{An object fitted with the function \code{\link[=gafem]{gafem()}}, \code{\link[=gamem]{gamem()}} or a
two-way table with BLUPs for genotypes in each trait (genotypes in rows and
traits in columns). In the last case, the first column is assumed to have
the genotype's name.}

\item{use_data}{Define which data to use if \code{.data} is an object of
class \code{gamem}. Defaults to \code{"blup"} (the BLUPs for genotypes).
Use \code{"pheno"} to use phenotypic means instead BLUPs for computing the
index.}

\item{SI}{An integer (0-100). The selection intensity in percentage of the
total number of genotypes.}

\item{mineval}{The minimum value so that an eigenvector is retained in the
factor analysis.}

\item{ideotype}{A vector of length \code{nvar} where \code{nvar} is the number of
traits used to plan the ideotype. Use \code{'h'} to indicate the traits in which
higher values are desired or \code{'l'} to indicate the traits in which lower
values are desired. For example, \code{ideotype = c("h, h, h, h, l")} will
consider that the ideotype has higher values for the first four traits and
lower values for the last trait. ALternatively, one can use a mixed vector,
indicating both h/l values and a numeric value for the target trait(s),
eg., \code{ideotype = c("120, h, 30, h, l")}. In this scenario, a numeric value
to define the ideotype is declared for the first and third traits. For this
traits, the absolute difference between the observed value and the numeric
ideotype will be computed, and after the rescaling procedure, the genotype
with the smallest difference will have 100. If \code{.data}is a model fitted
with the functions \code{\link[=gafem]{gafem()}} or \code{\link[=gamem]{gamem()}}, the order of the traits will be
the declared in the argument \code{resp} in those functions.}

\item{weights}{Optional weights to assign for each trait in the selection
process. It must be a numeric vector of length equal to the number of
traits in \code{.data}. By default (\code{NULL}) a numeric vector of weights equal to
1 is used, i.e., all traits have the same weight in the selection process.
It is suggested weights ranging from 0 to 1. The weights will then shrink
the ideotype vector toward 0. This is useful, for example, to prioritize
grain yield rather than a plant-related trait in the selection process.}

\item{use}{The method for computing covariances in the presence of missing
values. Defaults to \code{complete.obs}, i.e., missing values are handled
by casewise deletion.}

\item{verbose}{If \code{verbose = TRUE} (Default) then some results are
shown in the console.}
}
\value{
An object of class \code{mgidi} with the following items:
\itemize{
\item \strong{data} The data used to compute the factor analysis.
\item \strong{cormat} The correlation matrix among the environments.
\item \strong{PCA} The eigenvalues and explained variance.
\item \strong{FA} The factor analysis.
\item \strong{KMO} The result for the Kaiser-Meyer-Olkin test.
\item \strong{MSA} The measure of sampling adequacy for individual variable.
\item \strong{communalities} The communalities.
\item \strong{communalities_mean} The communalities' mean.
\item \strong{initial_loadings} The initial loadings.
\item \strong{finish_loadings} The final loadings after varimax rotation.
\item \strong{canonical_loadings} The canonical loadings.
\item \strong{scores_gen} The scores for genotypes in all retained factors.
\item \strong{scores_ide} The scores for the ideotype in all retained factors.
\item \strong{gen_ide} The distance between the scores of each genotype with the
ideotype.
\item \strong{MGIDI} The multi-trait genotype-ideotype distance index.
\item \strong{contri_fac} The relative contribution of each factor on the MGIDI
value. The lower the contribution of a factor, the close of the ideotype the
variables in such factor are.
\item \strong{contri_fac_rank, contri_fac_rank_sel} The rank for the contribution
of each factor for all genotypes and selected genotypes, respectively.
\item \strong{complementarity} The complementarity matrix, which is the Euclidean
distance between selected genotypes based on the contribution of each factor
on the MGIDI index (waiting reference).
\item \strong{sel_dif} The selection differential for the variables.
\item \strong{stat_gain} A descriptive statistic for the selection gains. The
minimum, mean, confidence interval, standard deviation, maximum, and sum of
selection gain values are computed. If traits have negative and positive
desired gains, the statistics are computed for by strata.
\item \strong{sel_gen} The selected genotypes.
}
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#stable}{\figure{lifecycle-stable.svg}{options: alt='[Stable]'}}}{\strong{[Stable]}}

Computes the multi-trait genotype-ideotype distance index, MGIDI, (Olivoto
and Nardino, 2020), used to select genotypes in plant breeding programs based
on multiple traits.The MGIDI index is computed as follows:
\loadmathjax
\mjsdeqn{MGIDI_i = \sqrt{\sum\limits_{j = 1}^f(F_{ij} - {F_j})^2}}

where \mjseqn{MGIDI_i} is the multi-trait genotype-ideotype distance index
for the \emph{i}th genotype; \mjseqn{F_{ij}} is the score of the \emph{i}th genotype in
the \emph{j}th factor (\emph{i = 1, 2, ..., g; j = 1, 2, ..., f}), being \emph{g} and \emph{f}
the number of genotypes and factors, respectively, and \mjseqn{F_j} is the
\emph{j}th score of the ideotype. The genotype with the lowest MGIDI is then
closer to the ideotype and therefore should presents desired values for all
the analyzed traits.
}
\examples{
\donttest{
library(metan)

# simulate a data set
# 10 genotypes
# 5 replications
# 4 traits
df <-
 g_simula(ngen = 10,
          nrep = 5,
          nvars = 4,
          gen_eff = 35,
          seed = c(1, 2, 3, 4))

# run a mixed-effect model (genotype as random effect)
mod <-
  gamem(df,
        gen = GEN,
        rep = REP,
        resp = everything())
# BLUPs for genotypes
gmd(mod, "blupg")

# Compute the MGIDI index
# Default options (all traits with positive desired gains)
# Equal weights for all traits
mgidi_ind <- mgidi(mod)
gmd(mgidi_ind, "MGIDI")

# Higher weight for traits V1 and V4
# This will increase the probability of selecting H7 and H9
# 30\% selection pressure
mgidi_ind2 <-
   mgidi(mod,
         weights = c(1, .2, .2, 1),
         SI = 30)
gmd(mgidi_ind2, "MGIDI")

# plot the contribution of each factor on the MGIDI index
p1 <- plot(mgidi_ind, type = "contribution")
p2 <- plot(mgidi_ind2, type = "contribution")
p1 + p2

# Negative desired gains for V1
# Positive desired gains for V2, V3 and V4
mgidi_ind3 <-
  mgidi(mod,
       ideotype = c("h, h, h, l"))


# Extract the BLUPs for each genotype
(blupsg <- gmd(mod, "blupg"))

# Consider the following ideotype that will be close to H4
# Define a numeric ideotype for the first three traits, and the lower values
# for the last trait
ideotype <- c("129.46, 76.8, 89.7, l")

mgidi_ind4 <-
  mgidi(mod,
       ideotype = ideotype)

# Note how the strenghts of H4 are related to FA1 (V1 and V2)
plot(mgidi_ind4, type = "contribution", genotypes = "all")

}
}
\references{
Olivoto, T., and Nardino, M. (2020). MGIDI: toward an effective
multivariate selection in biological experiments. Bioinformatics.
\doi{10.1093/bioinformatics/btaa981}
}
\author{
Tiago Olivoto \email{tiagoolivoto@gmail.com}
}
