#' Six-Hump Camel Back Function (F16)
#'
#' @description
#' A multimodal test function with six local minima, two of which are global.
#' Fixed dimension of 2.
#'
#' @param x Numeric vector of length 2 (2-dimensional input).
#'
#' @return Numeric scalar representing the function value.
#'
#' @details
#' \strong{Formula:}
#' \deqn{f(x) = 4x_1^2 - 2.1x_1^4 + \frac{x_1^6}{3} + x_1 x_2 - 4x_2^2 + 4x_2^4}
#'
#' \strong{Global minimum:} \eqn{f(\pm 0.0898, \mp 0.7126) \approx -1.0316}
#'
#' There are two global minima at approximately \eqn{(0.0898, -0.7126)} and
#' \eqn{(-0.0898, 0.7126)}.
#'
#' \strong{Characteristics:}
#' \itemize{
#'   \item Type: Multimodal
#'   \item Separable: No
#'   \item Differentiable: Yes
#'   \item Fixed dimension: 2
#'   \item Number of local minima: 6
#'   \item Number of global minima: 2
#'   \item Default bounds: \eqn{[-5, 5]^2}
#' }
#'
#' The function has a shape resembling a camel's back with six humps (minima).
#'
#' @seealso
#' \code{\link{test-functions}} for an overview of all test functions,
#' \code{\link{get_function_details}} to retrieve function parameters.
#'
#' @examples
#' F16(c(0.0898, -0.7126))   # Returns approximately -1.0316 (global minimum)
#' F16(c(-0.0898, 0.7126))   # Returns approximately -1.0316 (global minimum)
#' F16(c(0, 0))              # Returns 0
#'
#' @export
F16 <- function(x) {
  # Six-Hump Camel Back
  return(4 * (x[1]^2) - 2.1 * (x[1]^4) + (x[1]^6) / 3 +
           x[1] * x[2] - 4 * (x[2]^2) + 4 * (x[2]^4))
}
