% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/HelperFunctions.R
\name{loglik_weibull}
\alias{loglik_weibull}
\title{Compute Log-Likelihood for Weibull Accelerated Failure Time Model}
\usage{
loglik_weibull(log_y, log_mu, status, scale, weights = 1)
}
\arguments{
\item{log_y}{Numeric vector of logarithmic response/survival times}

\item{log_mu}{Numeric vector of logarithmic predicted survival times}

\item{status}{Numeric vector of censoring indicators
(1 = event, 0 = censored)
Indicates whether an event of interest occurred (1) or the observation was
right-censored (0). In survival analysis, right-censoring occurs when the
full survival time is unknown, typically because the study ended or the
subject was lost to follow-up before the event of interest occurred.}

\item{scale}{Numeric scalar representing the Weibull scale parameter}

\item{weights}{Optional numeric vector of observation weights (default = 1)}
}
\value{
A numeric scalar representing the total log-likelihood of the model
}
\description{
Calculates the log-likelihood for a Weibull accelerated failure time (AFT)
survival model, supporting right-censored survival data.
}
\details{
The function computes log-likelihood contributions for a Weibull AFT model,
explicitly accounting for right-censored observations. It supports optional
observation weighting to accommodate complex sampling designs.

This both provides a tool for actually fitting Weibull AFT models, and
boilerplate code for users who wish to incorporate Lagrangian multiplier
smoothing splines into their own custom models.
}
\examples{

## Minimal example of fitting a Weibull Accelerated Failure Time model
# Simulating survival data with right-censoring
set.seed(1234)
x1 <- rnorm(1000)
x2 <- rbinom(1000, 1, 0.5)
yraw <- rexp(exp(0.01*x1 + 0.01*x2))
# status: 1 = event occurred, 0 = right-censored
status <- rbinom(1000, 1, 0.25)
yobs <- ifelse(status, runif(1, 0, yraw), yraw)
df <- data.frame(
  y = yobs,
  x1 = x1,
  x2 = x2
)

## Fit model using lgspline with Weibull AFT specifics
model_fit <- lgspline(y ~ spl(x1) + x2,
                      df,
                      unconstrained_fit_fxn = unconstrained_fit_weibull,
                      family = weibull_family(),
                      need_dispersion_for_estimation = TRUE,
                      dispersion_function = weibull_dispersion_function,
                      glm_weight_function = weibull_glm_weight_function,
                      shur_correction_function = weibull_shur_correction,
                      status = status,
                      opt = FALSE,
                      K = 1)

loglik_weibull(log(model_fit$y), log(model_fit$ytilde), status,
  sqrt(model_fit$sigmasq_tilde))

}
