% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/nlexpect.R
\name{nlexpect}
\alias{nlexpect}
\title{Compute expectation of a function of the coefficients.}
\usage{
nlexpect(
  est,
  fun,
  coefs,
  ...,
  tol = getOption("lfe.etol"),
  lhs = NULL,
  cv,
  istats = FALSE,
  flags = list(verbose = 0),
  max.eval = 200000L,
  method = c("hcubature", "pcubature", "cuhre", "suave", "vegas", "divonne"),
  vectorize = FALSE
)
}
\arguments{
\item{est}{object of class \code{"felm"} or \code{"lm"}, a result of a call to
\code{\link[=felm]{felm()}} or \code{lm}.}

\item{fun}{function of coefficients to be integrated. Can also be a
\code{quote}d expression.}

\item{coefs}{character. Names of coefficients to test. Only needed if
\code{fun} is a function.}

\item{...}{other arguments passed to fun or the integration routine.}

\item{tol}{numeric. Tolerance for the computed integral.}

\item{lhs}{character. Name of the left hand side, if \code{est} has more
than one.}

\item{cv}{Covariance matrix to use in place of \code{vcov(est)}}

\item{istats}{logical. Should convergence information from the integration
routine be included as attributes?}

\item{flags}{list. Additional flags for the underlying integration routine. Not used after the
package \pkg{R2Cuba} disappeared.}

\item{max.eval}{integer. Maximum number of integral evaluations.}

\item{method}{character. A method specification usable by \code{cubature::cubintegrate}.
The documentation there says that \code{"pcubature"} is good for smooth integrands of low dimensions.}

\item{vectorize}{logical or numeric. Use vectorized function evaluation from package
\pkg{cubature}. This can speed up integration significantly. If method is from the Cuba library
(i.e. not pcubature or hcubature), \code{vectorize} should be specified as a numeric, a vectorization
factor. The default is 128.}
}
\value{
The function \code{nlexpect} computes and returns the expectation of
the function \code{fun(beta)}, with \code{beta} a vector of coefficients.
I.e., if the coefficients \code{beta} are bootstrapped a large number of
times, \code{nlexpect(est, fun)} should be equal to \code{mean(fun(beta))}.
}
\description{
Use integration of the joint distribution of the coefficients to compute the
expectation of some function of the coefficients.  Can be used for
non-linear inference tests.
}
\details{
The function \code{nlexpect} integrates the function \code{fun(x)} over the
multivariate normal distribution specified by the point estimates and the
covariance matrix \code{vcov(est)}.  This is the expectation of
\code{fun(beta)} if we were to bootstrap the data (e.g. by drawing the
residuals anew) and do repeated estimations.

The list of coefficients used by \code{fun} must be specified in
\code{coefs}.

If the function is simple, it can be specified as a quoted expression like
\code{quote(a*b+log(abs(d)))}. In this case, if \code{coefs} is not
specified, it will be set to the list of all the variables occurring in the
expression which are also names of coefficients.

\code{fun} may return a vector of values, in which case a vector of
expectations is computed, like \code{quote(c(a*b, a^3-b))}. However, if the
expressions contain different variables, like \code{quote(c(a*b, d*e))}, a
quite compute intensive 4-dimensional integral will be computed, compared to
two cheap 2-dimensional integrals if you do them separately. There is nothing to gain
from using vector-valued functions compared to multiple calls to \code{nlexpect()}.

You may of course also integrate inequalities like \code{quote(abs(x1-0.2) > 0.2)} to simulate the probability from t-tests or Wald-tests. See the
examples.

The function you provide will get an argument \code{...} if it does not have
one already.  It will also be passed an argument \code{.z} which contains
the actual coefficients in normalized coordinates, i.e. if \code{ch} is the
Cholesky decomposition of the covariance matrix, and \code{pt} are the point
estimates, the coefficients will be \verb{pt + ch \\\%*\\\% .z}. The first argument
is a vector with names corresponding to the coefficients.

If you specify \code{vectorized=TRUE}, your function will be passed a list with vectors
in its first argument. The function must
be able to handle a list, and must return a vector of the same length as the vectors
in the list.  If you pass an expression like \code{x < y}, each variable will be a vector.
If your function is vector valued, it must return a matrix where each
column is the values.

The \code{tol} argument specifies both the relative tolerance and the
absolute tolerance. If these should not be the same, specify \code{tol} as a
vector of length 2. The first value is the relative tolerance, the second is
the absolute tolerance. Termination occurs when at least one of the
tolerances is met.

The \code{...} can be used for passing other arguments to the integration
routine \code{cubature::cubintegrate} and the function to be integrated.
}
\note{
An alternative to this method is to use the \code{bootexpr} argument
with \code{\link[=felm]{felm()}}, to do a Monte Carlo integration.
}
\examples{

N <- 100
x1 <- rnorm(N)
# make some correlation
x2 <- 0.1 * rnorm(N) + 0.1 * x1
y <- 0.1 * x1 + x2 + rnorm(N)
summary(est <- felm(y ~ x1 + x2))
pt1 <- coef(est)["x1"]
pt2 <- coef(est)["x2"]
# expected values of coefficients, should match the summary
# and variance, i.e. square of standard errors in the summary
nlexpect(est, quote(c(x1 = x1, x2 = x2, var = c((x1 - pt1)^2, (x2 - pt2)^2))))
\donttest{
# the covariance matrix:
nlexpect(est, tcrossprod(as.matrix(c(x1 - pt1, x2 - pt2))))
}
# Wald test of single variable
waldtest(est, ~x1)["p.F"]
# the same with nlexpect, i.e. probability for observing abs(x1)>abs(pt1) conditional
# on E(x1) = 0.
nlexpect(est, (x1 - pt1)^2 > pt1^2, tol = 1e-7, vectorize = TRUE)
# which is the same as
2 * nlexpect(est, x1 * sign(pt1) < 0)

# Here's a multivalued, vectorized example
nlexpect(est, rbind(a = x1 * x2 < pt1, b = x1 * x2 > 0), vectorize = TRUE, method = "hcubature")
\donttest{

# Non-linear test:

# A simple one, what's the probability that product x1*x2 is between 0 and |E(x1)|?
nlexpect(est, x1 * x2 > 0 & x1 * x2 < abs(pt1), vectorize = TRUE, method = "hcubature")
# Then a more complicated one with the expected value of a polynomal in the coefficients
f <- function(x) c(poly = x[["x1"]] * (6 * x[["x1"]] - x[["x2"]]^2))
# This is the linearized test:
waldtest(est, f)["p.F"]
# In general, for a function f, the non-linear Wald test is something like
# the following:
# expected value of function
Ef <- nlexpect(est, f, coefs = c("x1", "x2"))
# point value of function
Pf <- f(c(pt1, pt2))
# similar to a Wald test, but non-linear:
nlexpect(est, function(x) (f(x) - Ef)^2 > Pf^2, c("x1", "x2"), vectorize = TRUE)
# one-sided
nlexpect(est, function(x) f(x) - Ef > abs(Pf), c("x1", "x2"), vectorize = TRUE)
# other sided
nlexpect(est, function(x) f(x) - Ef < -abs(Pf), c("x1", "x2"), vectorize = TRUE)
}

}
\seealso{
\code{\link[=waldtest]{waldtest()}}
}
