% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/legend.R
\name{addLegend}
\alias{addLegend}
\alias{labelFormat}
\title{Add a color legend to a map}
\usage{
addLegend(
  map,
  position = c("topright", "bottomright", "bottomleft", "topleft"),
  pal,
  values,
  na.label = "NA",
  bins = 7,
  colors,
  opacity = 0.5,
  labels = NULL,
  labFormat = labelFormat(),
  title = NULL,
  className = "info legend",
  layerId = NULL,
  group = NULL,
  data = getMapData(map)
)

labelFormat(
  prefix = "",
  suffix = "",
  between = " &ndash; ",
  digits = 3,
  big.mark = ",",
  transform = identity
)
}
\arguments{
\item{map}{a map widget object created from \code{\link[=leaflet]{leaflet()}}}

\item{position}{the position of the legend}

\item{pal}{the color palette function, generated from
\code{\link[=colorNumeric]{colorNumeric()}}, \code{colorBin()}, \code{colorQuantile()}, or
\code{colorFactor()}}

\item{values}{the values used to generate colors from the palette function}

\item{na.label}{the legend label for \code{NA}s in \code{values}}

\item{bins}{an approximate number of tick-marks on the color gradient for the
\code{colorNumeric} palette if it is of length one; you can also provide a
numeric vector as the pre-defined breaks (equally spaced)}

\item{colors}{a vector of (HTML) colors to be used in the legend if
\code{pal} is not provided}

\item{opacity}{the opacity of colors}

\item{labels}{a vector of text labels in the legend corresponding to
\code{colors}}

\item{labFormat}{a function to format the labels derived from \code{pal} and
\code{values} (see Details below to know what \code{labelFormat()} returns
by default; you can either use the helper function \code{labelFormat()}, or
write your own function)}

\item{title}{the legend title}

\item{className}{extra CSS classes to append to the control, space separated}

\item{layerId}{the ID of the legend; subsequent calls to \code{addLegend()}
or \code{addControl()} with the same \code{layerId} will replace this
legend. The ID can also be used with \code{removeControl()}.}

\item{group}{\code{group} name of a leaflet layer group.
Supplying this value will tie the legend to the leaflet layer group
with this name and will auto add/remove the legend as the
group is added/removed, for example via \code{layerControl()}.
You will need to set the \code{group} when you add a layer
(e.g., \code{\link[=addPolygons]{addPolygons()}}) and supply the same name here.}

\item{data}{the data object from which the argument values are derived; by
default, it is the \code{data} object provided to \code{leaflet()}
initially, but can be overridden}

\item{prefix}{a prefix of legend labels}

\item{suffix}{a suffix of legend labels}

\item{between}{a separator between \code{x[i]} and \code{x[i + 1]} in legend
labels (by default, it is a dash)}

\item{digits}{the number of digits of numeric values in labels}

\item{big.mark}{the thousand separator}

\item{transform}{a function to transform the label value}
}
\description{
When a color palette function is used in a map (e.g.,
\code{\link[=colorNumeric]{colorNumeric()}}), a color legend can be automatically derived from
the palette function. You can also manually specify the colors and labels for
the legend.
}
\details{
The \code{labFormat} argument is a function that takes the argument
\code{type = c("numeric", "bin", "quantile", "factor")}, plus, arguments for
different types of color palettes. For the \code{colorNumeric()} palette,
\code{labFormat} takes a single argument, which is the breaks of the numeric
vector, and returns a character vector of the same length. For
\code{colorBin()}, \code{labFormat} also takes a vector of breaks of length
\code{n} but should return a character vector of length \code{n - 1}, with
the \code{i}-th element representing the interval \code{c(x[i], x[i + 1])}.
For \code{colorQuantile()}, \code{labFormat} takes two arguments, the quantiles
and the associated probabilities (each of length \code{n}), and should return
a character vector of length \code{n - 1} (similar to the \code{colorBin()}
palette). For \code{colorFactor()}, \code{labFormat} takes one argument, the
unique values of the factor, and should return a character vector of the same
length.

By default, \code{labFormat} is basically \code{format(scientific = FALSE, big.mark = ",")} for the numeric palette, \code{as.character()} for the
factor palette, and a function to return labels of the form \samp{x[i] - x[i
+ 1]} for bin and quantile palettes (in the case of quantile palettes,
\code{x} is the probabilities instead of the values of breaks).
}
\examples{
# !formatR
library(leaflet)
# a manual legend
leaflet() \%>\% addTiles() \%>\% addLegend(
  position = "bottomright",
  colors = rgb(t(col2rgb(palette())) / 255),
  labels = palette(), opacity = 1,
  title = "An Obvious Legend"
)

\donttest{

# an automatic legend derived from the color palette
df <- local({
  n <- 300; x <- rnorm(n); y <- rnorm(n)
  z <- sqrt(x ^ 2 + y ^ 2); z[sample(n, 10)] <- NA
  data.frame(x, y, z)
})
pal <- colorNumeric("OrRd", df$z)
leaflet(df) \%>\%
  addTiles() \%>\%
  addCircleMarkers(~x, ~y, color = ~pal(z), group = "circles") \%>\%
  addLegend(pal = pal, values = ~z, group = "circles", position = "bottomleft") \%>\%
  addLayersControl(overlayGroups = c("circles"))

# format legend labels
df <- data.frame(x = rnorm(100), y = rexp(100, 2), z = runif(100))
pal <- colorBin("PuOr", df$z, bins = c(0, .1, .4, .9, 1))
leaflet(df) \%>\%
  addTiles() \%>\%
  addCircleMarkers(~x, ~y, color = ~pal(z), group = "circles") \%>\%
  addLegend(pal = pal, values = ~z, group = "circles", position = "bottomleft") \%>\%
  addLayersControl(overlayGroups = c("circles"))

leaflet(df) \%>\%
  addTiles() \%>\%
  addCircleMarkers(~x, ~y, color = ~pal(z), group = "circles") \%>\%
  addLegend(pal = pal, values = ~z, labFormat = labelFormat(
    prefix = "(", suffix = ")\%", between = ", ",
    transform = function(x) 100 * x
  ),  group = "circles", position = "bottomleft" ) \%>\%
  addLayersControl(overlayGroups = c("circles"))
}
}
