% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/permshap.R
\name{permshap}
\alias{permshap}
\alias{permshap.default}
\alias{permshap.ranger}
\title{Permutation SHAP}
\usage{
permshap(object, ...)

\method{permshap}{default}(
  object,
  X,
  bg_X = NULL,
  pred_fun = stats::predict,
  feature_names = colnames(X),
  bg_w = NULL,
  bg_n = 200L,
  exact = length(feature_names) <= 8L,
  low_memory = length(feature_names) > 15L,
  tol = 0.01,
  max_iter = 10L * length(feature_names),
  parallel = FALSE,
  parallel_args = NULL,
  verbose = TRUE,
  seed = NULL,
  ...
)

\method{permshap}{ranger}(
  object,
  X,
  bg_X = NULL,
  pred_fun = NULL,
  feature_names = colnames(X),
  bg_w = NULL,
  bg_n = 200L,
  exact = length(feature_names) <= 8L,
  low_memory = length(feature_names) > 15L,
  tol = 0.01,
  max_iter = 10L * length(feature_names),
  parallel = FALSE,
  parallel_args = NULL,
  verbose = TRUE,
  seed = NULL,
  survival = c("chf", "prob"),
  ...
)
}
\arguments{
\item{object}{Fitted model object.}

\item{...}{Additional arguments passed to \code{pred_fun(object, X, ...)}.}

\item{X}{\eqn{(n \times p)} matrix or \code{data.frame} with rows to be explained.
The columns should only represent model features, not the response
(but see \code{feature_names} on how to overrule this).}

\item{bg_X}{Background data used to integrate out "switched off" features,
often a subset of the training data (typically 50 to 500 rows).
In cases with a natural "off" value (like MNIST digits),
this can also be a single row with all values set to the off value.
If no \code{bg_X} is passed (the default) and if \code{X} is sufficiently large,
a random sample of \code{bg_n} rows from \code{X} serves as background data.}

\item{pred_fun}{Prediction function of the form \verb{function(object, X, ...)},
providing \eqn{K \ge 1} predictions per row. Its first argument
represents the model \code{object}, its second argument a data structure like \code{X}.
Additional (named) arguments are passed via \code{...}.
The default, \code{\link[stats:predict]{stats::predict()}}, will work in most cases.}

\item{feature_names}{Optional vector of column names in \code{X} used to calculate
SHAP values. By default, this equals \code{colnames(X)}.}

\item{bg_w}{Optional vector of case weights for each row of \code{bg_X}.
If \code{bg_X = NULL}, must be of same length as \code{X}. Set to \code{NULL} for no weights.}

\item{bg_n}{If \code{bg_X = NULL}: Size of background data to be sampled from \code{X}.}

\item{exact}{If \code{TRUE}, the algorithm will produce exact SHAP values
with respect to the background data.
The default is \code{TRUE} for up to eight features, and \code{FALSE} otherwise.}

\item{low_memory}{If \code{FALSE} (default up to p = 15), the algorithm does p
iterations in one chunk, evaluating Shapley's formula 2p^2 times.
For models with interactions up to order two, you can set this to \code{TRUE}
to save time.}

\item{tol}{Tolerance determining when to stop. As in CL21, the algorithm keeps
iterating until \eqn{\textrm{max}(\sigma_n)/(\textrm{max}(\beta_n) - \textrm{min}(\beta_n)) < \textrm{tol}},
where the \eqn{\beta_n} are the SHAP values of a given observation,
and \eqn{\sigma_n} their standard errors.
For multidimensional predictions, the criterion must be satisfied for each
dimension separately. The stopping criterion uses the fact that standard errors
and SHAP values are all on the same scale. Ignored if \code{exact = TRUE}.
For \code{permshap()}, the default is 0.01, while for \code{kernelshap()} it is set to 0.005.}

\item{max_iter}{If the stopping criterion (see \code{tol}) is not reached after
\code{max_iter} iterations, the algorithm stops. Ignored if \code{exact = TRUE}.}

\item{parallel}{If \code{TRUE}, use \code{\link[foreach:foreach]{foreach::foreach()}} and \verb{\%dofuture\%} to loop over rows
to be explained. Must register backend beforehand, e.g., \code{plan(multisession)},
see README for an example. Currently disables the progress bar.}

\item{parallel_args}{Named list of arguments passed to
\code{foreach::foreach(.options.future = ...)}, ideally \code{NULL} (default).
Only relevant if \code{parallel = TRUE}.
Example on Windows: if \code{object} is a GAM fitted with package 'mgcv',
then one might need to set \code{parallel_args = list(packages = "mgcv")}.
Similarly, if the model has been fitted with \code{ranger()}, then it might be necessary
to pass \code{parallel_args = list(packages = "ranger")}.}

\item{verbose}{Set to \code{FALSE} to suppress messages and the progress bar.}

\item{seed}{Optional integer random seed. Note that it changes the global seed.}

\item{survival}{Should cumulative hazards ("chf", default) or survival
probabilities ("prob") per time be predicted? Only in \code{ranger()} survival models.}
}
\value{
An object of class "kernelshap" with the following components:
\itemize{
\item \code{S}: \eqn{(n \times p)} matrix with SHAP values or, if the model output has
dimension \eqn{K > 1}, a list of \eqn{K} such matrices.
\item \code{X}: Same as input argument \code{X}.
\item \code{baseline}: Vector of length K representing the average prediction on the
background data.
\item \code{bg_X}: The background data.
\item \code{bg_w}: The background case weights.
\item \code{m_exact}: Number of on-off vectors evaluated once per row of \code{X}.
\item \code{exact}: Logical flag indicating whether calculations are exact or not.
\item \code{txt}: Summary text.
\item \code{predictions}: \eqn{(n \times K)} matrix with predictions of \code{X}.
\item \code{algorithm}: "permshap".
\item \code{m}: Number of sampled on-off vectors evaluated per iteration (if not exact).
\item \code{SE}: Standard errors corresponding to \code{S} (if not exact).
\item \code{n_iter}: Integer vector of length n providing the number of iterations
per row of \code{X} (if not exact).
\item \code{converged}: Logical vector of length n indicating convergence per row of \code{X}
(if not exact).
}
}
\description{
Permutation SHAP algorithm with respect to a background dataset,
see Strumbelj and Kononenko (2014) for the basic idea.

By default, for up to p=8 features, exact SHAP values are returned
(exact with respect to the selected background data).
Otherwise, the sampling process iterates until the resulting values
are sufficiently precise, and standard errors are provided.
}
\details{
During each iteration, the algorithm cycles twice through a random permutation:
It starts with all feature components "turned on" (i.e., taking them
from the observation to be explained), then gradually turning off components
according to the permutation.
When all components are turned off, the algorithm - one by one - turns the components
back on, until all components are turned on again. This antithetic scheme allows to
evaluate Shapley's formula twice per feature using a single permutation and a total
of 2p disjoint evaluations of the contribution function.

For models with interactions up to order two, one can show that
even a single iteration provides exact SHAP values for all features
(with respect to the given background dataset).

The Python implementation "shap" uses a similar approach, but without
providing standard errors, and without early stopping.

For faster convergence, we use balanced permutations in the sense that
p subsequent permutations each start with a different feature.
Furthermore, the 2p on-off vectors with sum <=1 or >=p-1 are evaluated only once,
similar to the degree 1 hybrid in \code{\link[=kernelshap]{kernelshap()}}.
}
\section{Methods (by class)}{
\itemize{
\item \code{permshap(default)}: Default permutation SHAP method.

\item \code{permshap(ranger)}: Permutation SHAP method for "ranger" models, see Readme for an example.

}}
\examples{
# MODEL ONE: Linear regression
fit <- lm(Sepal.Length ~ ., data = iris)

# Select rows to explain (only feature columns)
X_explain <- iris[-1]

# Calculate SHAP values
s <- permshap(fit, X_explain)
s

# MODEL TWO: Multi-response linear regression
fit <- lm(as.matrix(iris[, 1:2]) ~ Petal.Length + Petal.Width + Species, data = iris)
s <- permshap(fit, iris[3:5])
s

# Note 1: Feature columns can also be selected 'feature_names'
# Note 2: Especially when X is small, pass a sufficiently large background data bg_X
s <- permshap(
  fit,
  iris[1:4, ],
  bg_X = iris,
  feature_names = c("Petal.Length", "Petal.Width", "Species")
)
s
}
\references{
\enumerate{
\item Erik Strumbelj and Igor Kononenko. Explaining prediction models and individual
predictions with feature contributions. Knowledge and Information Systems 41, 2014.
}
}
