# Data processing functions to clean and annotate data ========

#' @title Function replaced by `orbi_flag_satellite_peaks()`
#' @param ... parameters passed on to the new function [orbi_flag_satellite_peaks()].
#' @export
orbi_filter_satellite_peaks <- function(...) {
  lifecycle::deprecate_soft(
    "1.2.0",
    "orbi_filter_satellite_peaks()",
    "orbi_flag_satellite_peaks()"
  )
  orbi_flag_satellite_peaks(...) |>
    orbi_filter_flagged_data()
}

#' @title Flag minor satellite peaks
#' Flags minor signals for an isotopocule that matches multiple peaks within its exact mass +/- tolerance interval in the same scan. These are often small satellite peaks generated by the Fourier transform.
#' However, if there are satelite peaks of high intensity or very many satellite peaks, it can indicate that the m/z and tolerance setting used for identifying isotopcules need to be revisited.
#' Visualize the flagged satellite peaks with [orbi_plot_satellite_peaks()].
#'
#' @param dataset An aggregated dataset or a data frame of peaks (i.e. works directly after [orbi_identify_isotopocules()] as well as with a tibble from \link[=orbi_get_data]{orbi_get_data(peaks = everything())} or when reading from an IsoX file)
#' @examples
#' fpath <- system.file("extdata", "testfile_flow.isox", package = "isoorbi")
#' df <-
#'   orbi_read_isox(file = fpath) |>
#'   orbi_simplify_isox() |>
#'   orbi_flag_satellite_peaks()
#'
#' @return same object as provided in `dataset` with new column `is_satellite_peak` that flags satellite peaks
#' @export
orbi_flag_satellite_peaks <- function(dataset) {
  # safety checks
  check_dataset_arg(dataset)

  # keep track for later
  is_agg <- is(dataset, "orbi_aggregated_data")
  peaks <- if (is_agg) dataset$peaks else dataset

  # check columns
  check_tibble(
    peaks,
    c("uidx|filename", "scan.no", "isotopocule", "ions.incremental|intensity"),
    regexps = TRUE,
    .arg = "dataset"
  )

  # info
  start <- start_info("is running")

  # y column
  y_col <- names(tidyselect::eval_select(
    any_of(c("ions.incremental", "intensity")),
    peaks
  ))[1]

  # calculation
  out <-
    peaks |>
    dplyr::mutate(
      .by = dplyr::any_of(
        c("uidx", "filename", "scan.no", "itc_uidx", "isotopocule", "compound")
      ),
      # ignore missing and unidentified for this
      is_satellite_peak = !is.na(.data$isotopocule) &
        !is.na(!!sym(y_col)) &
        !!expr(!!sym(y_col) < max(!!sym(y_col)))
    ) |>
    try_catch_cnds()

  # abort if issues
  abort_cnds(
    out$conditions,
    message = "something went wrong tying to flag satellite peaks:"
  )
  peaks <- out$result

  # info
  sat_peaks <- sum(peaks$is_satellite_peak)
  isotopocules <- peaks |>
    dplyr::summarise(
      has_sat_peaks = any(.data$is_satellite_peak),
      .by = "isotopocule"
    ) |>
    dplyr::filter(.data$has_sat_peaks) |>
    dplyr::pull(.data$isotopocule)

  if (sat_peaks > 0) {
    finish_info(
      "flagged {format_number(sat_peaks)}/{format_number(nrow(peaks))} peaks ",
      "in {length(isotopocules)} isotopocule{?s} ({.field {isotopocules}}) ",
      "as {cli::col_yellow('satellite')} peaks ({signif(100 * sat_peaks / nrow(peaks), 2)}%)",
      start = start
    )
  } else {
    finish_info(
      "confirmed there are no {cli::col_yellow('satellite')} peaks",
      start = start
    )
  }

  if (is_agg) {
    # got aggregated data to begin with --> return aggregated data
    dataset$peaks <- peaks
    return(dataset)
  } else {
    # got a plain peaks tibble
    return(peaks)
  }
}

#' @title Function replaced by `orbi_flag_weak_isotopocules()`
#' @param ... parameters passed on to the new function orbi_flag_weak_isotopocules().
#' @export
orbi_filter_weak_isotopocules <- function(...) {
  lifecycle::deprecate_soft(
    "1.2.0",
    "orbi_filter_weak_isotopocules()",
    "orbi_flag_weak_isotopocules()"
  )
  orbi_flag_weak_isotopocules(...) |>
    orbi_filter_flagged_data()
}

#' Flag weak isotopocules
#'
#' This function flags isotopocules that are not detected in a minimum of `min_percent` of scans that then can be easily visualized with [orbi_plot_isotopocule_coverage()].
#' It evaluates weak isotopocules within each "uidx", "filename", "block", "segment" and "injection" (for those of the columns that exist),
#' in addition to any groupings already defined before calling this function using dplyr's `group_by()`. It restores the original groupings in the returned data.
#'
#' @param dataset A simplified IsoX data frame to be processed
#' @param min_percent A number between 0 and 100 (inclusive). Isotopocule must be observed in at least this percentage of scans (please note: the percentage
#' is defined relative to the most commonly observed isotopocule of each compound). The default is 100, the most stringent condition to ensure reliable
#' isotpocule coverage and ratio calculations across data blocks. If you lower the default, be mindful of potential misinterprations from using isotopotcules
#' that are very close to their detection limit within a datablock. For continuous flow operations it may be necessary to make data blocks smaller using
#' [orbi_define_block_for_flow_injection()] and [orbi_adjust_block()].
#'
#' @examples
#' fpath <- system.file("extdata", "testfile_flow.isox", package = "isoorbi")
#' df <- orbi_read_isox(file = fpath) |>
#'       orbi_simplify_isox() |>
#'       orbi_flag_weak_isotopocules(min_percent = 100)
#'
#' @return same object as provided in `dataset` with new column `is_weak_isotopocule` that flags weak isotopocules.
#' @export
orbi_flag_weak_isotopocules <- function(dataset, min_percent = 100) {
  # safety checks
  check_dataset_arg(dataset)
  check_arg(
    min_percent,
    !missing(min_percent) &&
      is.numeric(min_percent) &&
      length(min_percent) == 1L &&
      min_percent > 0 &&
      min_percent <= 100,
    "must be a single number between 0 and 100 (inclusive)"
  )

  # keep track for later
  is_agg <- is(dataset, "orbi_aggregated_data")
  added_scan_cols <- c("block", "segment")
  peaks <- if (is_agg) {
    dataset$peaks |>
      dplyr::left_join(
        dataset$scans |>
          dplyr::select(
            "uidx",
            "scan.no",
            dplyr::any_of(added_scan_cols)
          ),
        by = c("uidx", "scan.no")
      )
  } else {
    dataset
  }

  # check columns
  check_tibble(
    peaks,
    c("uidx|filename", "scan.no", "isotopocule", "ions.incremental|intensity"),
    regexps = TRUE,
    .arg = "dataset"
  )

  # info
  start <- start_info("is running")

  # y column
  y_col <- names(tidyselect::eval_select(
    any_of(c("ions.incremental", "intensity")),
    peaks
  ))[1]

  # add groupings if they exist
  peaks_out <- peaks |>
    group_if_exists(
      c("uidx", "filename", "compound", "block", "segment", "injection"),
      add = TRUE
    )

  # info
  starting_isotopocules <- peaks_out |> count_grouped_distinct("isotopocule")

  # flag weak isotopocules
  out <-
    peaks_out |>
    dplyr::mutate(max.scans = length(unique(.data$scan.no))) |>
    dplyr::group_by(.data$isotopocule, .add = TRUE) |>
    dplyr::mutate(
      obs.scans = length(unique(.data$scan.no[
        !is.na(.data$isotopocule) &
          !is.na(!!sym(y_col))
      ])),
      is_weak_isotopocule = !is.na(.data$isotopocule) &
        .data$obs.scans < min_percent / 100 * .data$max.scans
    ) |>
    dplyr::select(-"obs.scans", -"max.scans") |>
    try_catch_cnds()

  # stop if there are errors
  abort_cnds(
    out$conditions,
    message = "something went wrong flagging weak isotopocules:"
  )
  peaks_out <- out$result

  # info
  removed_isotopocules <- peaks_out |>
    dplyr::filter(.data$is_weak_isotopocule) |>
    count_grouped_distinct("isotopocule")
  n_groups <- dplyr::n_groups(peaks_out)
  if (removed_isotopocules > 0) {
    finish_info(
      "flagged {removed_isotopocules} of {starting_isotopocules} isotopocules as {cli::col_yellow('weak')} ",
      "because they were NOT present in at least {min_percent}% of scans ",
      if (n_groups > 0) {
        "in each of the {n_groups} data groups (based on {.field {dplyr::group_vars(peaks_out)}}) "
      },
      "{cli::symbol$arrow_right} use {.strong orbi_plot_isotopocule_coverage()} to visualize them",
      start = start
    )
  } else {
    finish_info(
      "confirmed there are no {cli::col_yellow('weak')} isotopocules: ",
      "all are detected in at least {min_percent}% of scans ",
      if (n_groups > 0) {
        "in each of the {n_groups} data groups (based on {.field {dplyr::group_vars(peaks_out)}})"
      },
      start = start
    )
  }

  # restore groupings from original dataset
  peaks_out <- peaks_out |> group_by_same_groups(peaks)

  if (is_agg) {
    # got aggregated data to begin with --> return aggregated data
    dataset$peaks <- peaks_out |>
      dplyr::select(-dplyr::any_of(added_scan_cols))
    return(dataset)
  } else {
    # got a plain peaks tibble
    return(peaks_out)
  }
}

#' Isotopocule coverage
#'
#' The coverage of each isotopcule across scans/time is an important indicator for data completeness. These functions provide ways to summarize and visualize the isotopocule coverage in a dataset.
#'
#' @inheritParams orbi_flag_satellite_peaks
#' @describeIn orbi_isotopocule_coverage calculates which stretches of the data have data for which isotopocules. This function is usually used indicrectly by [orbi_plot_isotopocule_coverage()] but can be called directly to investigate isotopocule coverage.
#'
#' @return summary data frame
#' @export
orbi_get_isotopocule_coverage <- function(dataset) {
  # safety checks
  check_dataset_arg(dataset)

  # get peaks tibble
  peaks <- if (is(dataset, "orbi_aggregated_data")) {
    dataset$peaks |>
      dplyr::left_join(
        dataset$file_info |> dplyr::select("uidx", "filename"),
        by = "uidx"
      ) |>
      dplyr::left_join(
        dataset$scans |>
          dplyr::select(
            "uidx",
            "scan.no",
            "time.min",
            dplyr::any_of("data_group")
          ),
        by = c("uidx", "scan.no")
      )
  } else {
    dataset
  }

  # check columns
  check_tibble(
    peaks,
    c(
      "uidx|filename",
      "scan.no",
      "isotopocule",
      "ions.incremental|intensity"
    ),
    regexps = TRUE,
    .arg = "dataset"
  )

  # y column (only used for filtering)
  y_col <- names(tidyselect::eval_select(
    any_of(c("ions.incremental", "intensity")),
    peaks
  ))[1]

  # prep peaks
  peaks <- peaks |>
    # filter out missing and unidentified
    orbi_filter_isotopocules() |>
    # will only factorize if they exist
    factorize_dataset(c("filename", "compound", "isotopocule")) |>
    suppressMessages()
  isotopocule_levels <- levels(peaks$isotopocule)

  # grouping colums
  by_cols <- tidyselect::eval_select(
    any_of(c(
      "uidx",
      "filename",
      "compound",
      "isotopocule",
      # make sure a data group column is included if it exists
      "data_group",
      # make sure a weak isotopocule column is included if it exists
      "is_weak_isotopocule"
    )),
    peaks
  ) |>
    names()

  # arrange colums
  arrange_cols <- tidyselect::eval_select(
    any_of(c(
      "filename",
      "uidx",
      "compound",
      "isotopocule"
    )),
    peaks
  ) |>
    names()

  # calculate coverage
  output <-
    peaks |>
    # need isotopocule as char otherwise will always complete for all levels
    dplyr::mutate(isotopocule = as.character(.data$isotopocule)) |>
    # find data stretches
    dplyr::arrange(!!!map(c(arrange_cols, "scan.no"), sym)) |>
    dplyr::mutate(
      .by = dplyr::all_of(by_cols),
      # re introduce factor
      isotopocule = factor(.data$isotopocule, levels = isotopocule_levels),
      data_stretch = c(0L, cumsum(diff(.data$scan.no) > 1L)),
    ) |>
    # summarize
    dplyr::summarize(
      .by = dplyr::all_of(c(by_cols, "data_stretch")),
      n_points = dplyr::n(),
      start_scan.no = .data$scan.no[1],
      end_scan.no = utils::tail(.data$scan.no, 1),
      start_time.min = if ("time.min" %in% names(peaks)) {
        .data$time.min[1]
      } else {
        list(NULL)
      },
      end_time.min = if ("time.min" %in% names(peaks)) {
        utils::tail(.data$time.min, 1)
      } else {
        list(NULL)
      }
    ) |>
    dplyr::arrange(
      !!!map(
        c(arrange_cols, if ("data_group" %in% names(peaks)) "data_group"),
        sym
      )
    )
  if (!"time.min" %in% names(peaks)) {
    output <- output |> dplyr::select(-"start_time.min", -"end_time.min")
  }
  return(output)
}


#' @title Function replaced by `orbi_flag_outliers()`
#' @param ... parameters passed on to the new function [orbi_flag_outliers()].
#' @param outlier_percent outlier_percent needs to be between 0 and 10, flags extreme scans based on TIC x injection time (i.e., ion intensity)
#' @export
orbi_filter_scan_intensity <- function(..., outlier_percent) {
  lifecycle::deprecate_soft(
    "1.2.0",
    "orbi_filter_scan_intensity()",
    "orbi_flag_outliers()"
  )
  lifecycle::deprecate_soft(
    "1.2.0",
    "orbi_filter_scan_intensity(outlier_percent)",
    details = "the argument `outlier_percent` has been superseded by `agc_window`"
  )
  orbi_flag_outliers(
    ...,
    agc_window = c(outlier_percent, 100 - outlier_percent)
  ) |>
    orbi_filter_flagged_data()
}


#' Flag outlier scans
#'
#' This function flags outliers using one of the different methods provided by the parameters (to use multiple, please call this function several times sequentially).
#' Note that this function evaluates outliers within each "uidx", "filename", and "injection" (for those of the columns that exist),
#' and additionally within each "block" and "segment" if `by_block = TRUE`.
#' in addition to any groupings already defined before calling this function using dplyr's `group_by()`. It restores the original groupings in the returned datasert.
#'
#' @inheritParams orbi_flag_satellite_peaks
#' @param agc_fold_cutoff flags scans with a fold cutoff based on the average number of ions in the Orbitrap analyzer. For example, `agc_fold_cutoff = 2` flags scans that have more than 2 times, or less than 1/2 times the average. TIC multiplied by injection time serves as an estimate for the number of ions in the Orbitrap.
#' @param agc_window flags scans with a critically low or high number of ions in the Orbitrap analyzer. Provide a vector with 2 numbers `c(x,y)` flagging the lowest x percent and highest y percent. TIC multiplied by injection time serves as an estimate for the number of ions in the Orbitrap.
#' @param agc_absolute_cutoff flags scans with a number of ions in the Orbitrap analyzer outside of an absolute range. Provide a vector with 2 numbers `c(x,y)` flagging data below x and above y of the TIC multiplied by injection time (which serves as an estimate for the number of ions in the Orbitrap).
#' @param by_block if the `dataset` has block and segment definitions, should the outlier flag be evaluated within each block+segment or globally? default is within each block+segment, switch to globally by turning `by_block = FALSE`
#'
#' @examples
#' fpath <- system.file("extdata", "testfile_flow.isox", package = "isoorbi")
#' df <-
#'   orbi_read_isox(file = fpath) |>
#'   orbi_simplify_isox() |>
#'   orbi_flag_outliers(agc_window = c(1,99))
#'
#' @return same object as provided in `dataset` with new columns `is_outlier` and `outlier_type` (if they don't already exist) that flags outliers identified by the method and provides the type of outlier (e.g. "2 fold agc cutoff"), respectively.
#' @export
orbi_flag_outliers <- function(
  dataset,
  agc_fold_cutoff = NA_real_,
  agc_window = c(),
  agc_absolute_cutoff = c(),
  by_block = TRUE
) {
  # safety checks
  check_dataset_arg(dataset)
  stopifnot(
    "if provided, `agc_fold_cutoff` needs to be a single number" = (length(
      agc_fold_cutoff
    ) ==
      1L &&
      is.na(agc_fold_cutoff)) ||
      is_scalar_double(agc_fold_cutoff),
    "if provided, `agc_window` needs to be a vector of two numbers (low and high filter) between 0 and 100" = length(
      agc_window
    ) ==
      0L ||
      (is.numeric(agc_window) &&
        length(agc_window) == 2L &&
        agc_window[1] < agc_window[2]),
    "if provided, `agc_absolute_cutoff` needs to be a vector of two numbers (low and high cutoffs)" = length(
      agc_absolute_cutoff
    ) ==
      0L ||
      (is.numeric(agc_absolute_cutoff) &&
        length(agc_absolute_cutoff) == 2L &&
        agc_absolute_cutoff[1] < agc_absolute_cutoff[2])
  )

  # keep track for later
  is_agg <- is(dataset, "orbi_aggregated_data")
  scans <- if (is_agg) dataset$scans else dataset

  # check columns
  check_tibble(
    scans,
    c("uidx|filename", "scan.no", "tic", "it.ms"),
    regexps = TRUE,
    .arg = "dataset"
  )

  # check filter to apply
  method <- c(
    agc_fold_cutoff = !is.na(agc_fold_cutoff),
    agc_window = length(agc_window) == 2L,
    agc_absolute_cutoff = length(agc_absolute_cutoff) == 2L
  )
  if (sum(method) > 1L) {
    sprintf(
      "can only use one method at a time, please call this function sequentially for each of these parameters: '%s'",
      paste(names(method)[method], collapse = "', '")
    ) |>
      abort()
  } else if (sum(method) == 0L) {
    sprintf(
      "need to define at least one of these parameters for identifying outliers: '%s'",
      paste(names(method), collapse = "', '")
    ) |>
      abort()
  }
  method <- names(method)[method]

  # method message
  method_msg <- ""
  method_type <- ""
  if (method == "agc_window") {
    method_msg <- format_inline(
      "scans whose number of ions {.field tic} * {.field it.ms} in the Orbitrap analyzer fall into the lowest (<{agc_window[1]}%) or highest (>{agc_window[2]}%) quantiles"
    )
    method_type <- sprintf(
      "AGC window (%s%% to %s%%) cutoff",
      agc_window[1],
      agc_window[2]
    )
  } else if (method == "agc_fold_cutoff") {
    method_msg <- format_inline(
      "scans below 1/{agc_fold_cutoff} and above {agc_fold_cutoff} times the average number of ions {.field tic} * {.field it.ms} in the Orbitrap analyzer"
    )
    method_type <- sprintf("%s fold AGC cutoff", agc_fold_cutoff)
  } else if (method == "agc_absolute_cutoff") {
    method_msg <- sprintf(
      "scans below %s and above %s the number of ions (`tic` * `it.ms`) in the Orbitrap analyzer",
      agc_absolute_cutoff[1],
      agc_absolute_cutoff[2]
    )
    method_type <- sprintf(
      "AGC cutoff (%s to %s)",
      agc_absolute_cutoff[1],
      agc_absolute_cutoff[2]
    )
  }

  # info
  start <- start_info("is running")

  # simplify scans down to single scans (already the case if we're in agg_data but this works universally)
  # note that the orignal scans' groupings are not changed so don't need to be re-recreated later on
  single_scans <- scans |>
    group_if_exists(
      c(
        "uidx",
        "filename",
        "injection",
        if (by_block) "block",
        if (by_block) "segment"
      ),
      add = TRUE
    )
  single_scans <- single_scans |>
    dplyr::select(dplyr::all_of(c(
      dplyr::group_vars(single_scans),
      "scan.no",
      "tic",
      "it.ms"
    ))) |>
    # filtering out missing data here (don't need to considert it again later)
    dplyr::filter(!is.na(.data$tic), !is.na(.data$it.ms)) |>
    dplyr::distinct()
  n_scans <- nrow(single_scans)

  # calculation
  out <- try_catch_cnds(
    if (method == "agc_window") {
      single_scans |>
        dplyr::mutate(
          TICxIT = .data$tic * .data$it.ms,
          is_new_outlier = .data$TICxIT <
            stats::quantile(.data$TICxIT, agc_window[1] / 100) |
            .data$TICxIT > stats::quantile(.data$TICxIT, agc_window[2] / 100)
        ) |>
        dplyr::select(-"TICxIT")
    } else if (method == "agc_fold_cutoff") {
      single_scans |>
        dplyr::mutate(
          TICxIT = .data$tic * .data$it.ms,
          TICxIT_mean = mean(.data$TICxIT),
          is_new_outlier = .data$TICxIT <
            1 / agc_fold_cutoff * .data$TICxIT_mean |
            .data$TICxIT > agc_fold_cutoff * .data$TICxIT_mean
        ) |>
        dplyr::select(-"TICxIT", -"TICxIT_mean")
    } else if (method == "agc_absolute_cutoff") {
      single_scans |>
        dplyr::mutate(
          TICxIT = .data$tic * .data$it.ms,
          is_new_outlier = .data$TICxIT < agc_absolute_cutoff[1] |
            .data$TICxIT > agc_absolute_cutoff[2]
        ) |>
        dplyr::select(-"TICxIT")
    }
  )

  # stop if error
  abort_cnds(
    out$conditions,
    message = "something went wrong flagging outliers:"
  )
  single_scans <- out$result

  # outlier and type update
  if (!"is_outlier" %in% names(scans)) {
    scans$is_outlier <- FALSE
    scans$outlier_type <- ""
  }
  scans <- scans |>
    # merge single scans back in
    dplyr::left_join(
      single_scans |>
        dplyr::select(dplyr::any_of(c(
          dplyr::group_vars(single_scans),
          "scan.no",
          "is_new_outlier"
        ))),
      by = c(dplyr::group_vars(single_scans), "scan.no")
    ) |>
    dplyr::mutate(
      is_outlier = ifelse(
        !is.na(.data$is_new_outlier) & .data$is_new_outlier,
        TRUE,
        .data$is_outlier
      ),
      outlier_type = ifelse(
        .data$is_new_outlier,
        method_type,
        .data$outlier_type
      )
    ) |>
    dplyr::select(-"is_new_outlier")

  # info
  n_scans_flagged <- single_scans |>
    dplyr::filter(.data$is_new_outlier) |>
    nrow()
  n_groups <- dplyr::n_groups(single_scans)
  if (n_scans_flagged > 0) {
    finish_info(
      "flagged {n_scans_flagged}/{n_scans} scans ({signif(n_scans_flagged / n_scans * 100, 2)}%) as {cli::col_yellow('outliers')} ",
      "based on {.field {method_type}}, i.e. based on {.emph {method_msg}}",
      if (n_groups > 1) {
        ", in {n_groups} data groups (based on {.field {dplyr::group_vars(single_scans)}}) "
      },
      " {cli::symbol$arrow_right} use {.strong orbi_plot_raw_data(y = tic * it.ms)} to visualize them",
      start = start
    )
  } else {
    finish_info(
      "confirmed that none of the {n_scans} scans are {cli::col_yellow('outliers')} ",
      "based on {.field {method_type}}, i.e. based on {.emph {method_msg}}",
      if (n_groups > 1) {
        ", in {n_groups} data groups (based on {.field {dplyr::group_vars(single_scans)}}) "
      },
      start = start
    )
  }

  if (is_agg) {
    # got aggregated data to begin with --> return aggregated data
    dataset$scans <- scans
    return(dataset)
  } else {
    # got a plain peaks tibble
    return(scans)
  }
}

# filter flagged data (fast internal function)
filter_flagged_data <- function(
  dataset,
  filter_satellite_peaks = TRUE,
  filter_weak_isotopocules = TRUE,
  filter_outliers = TRUE
) {
  if (filter_satellite_peaks && "is_satellite_peak" %in% names(dataset)) {
    dataset <- dataset |> filter(!.data$is_satellite_peak)
  }
  if (filter_weak_isotopocules && "is_weak_isotopocule" %in% names(dataset)) {
    dataset <- dataset |> filter(!.data$is_weak_isotopocule)
  }
  if (filter_outliers && "is_outlier" %in% names(dataset)) {
    dataset <- dataset |> filter(!.data$is_outlier)
  }
  return(dataset |> droplevels())
}

#' @title Filter out flagged data
#' @description This function filters out data that have been previously flagged using functions `orbi_flag_satellite_peaks()`, `orbi_flag_weak_isotopocules()`, and/or `orbi_flag_outliers()`. Note that this function is no longer necessary to call explicitly as `orbi_analyze_shot_noise()` and `orbi_summarize_results()` automatically exclude flagged data.
#' @param dataset a tibble with previously flagged data from `orbi_flag_satellite_peaks()`, `orbi_flag_weak_isotopocules()`, and/or `orbi_flag_outliers()`
#' @return a dataset with the flagged data filtered out
#' @export
orbi_filter_flagged_data <- function(dataset) {
  # safety checks
  stopifnot(
    "need a `dataset` data frame" = !missing(dataset) && is.data.frame(dataset)
  )

  # deprecation
  lifecycle::deprecate_soft(
    "1.3.0",
    "orbi_filter_flagged_data()",
    details = "filtering flagged data is no longer necessary as orbi_summarize_results() and other functions take flagged data into consideration and treat it appropriately"
  )

  # original n
  nall <- nrow(dataset)
  nsat_peaks <- 0L
  nweak_isos <- 0L
  noutliers <- 0L

  # remove flagged
  start <- start_info()
  details <- c()
  if ("is_satellite_peak" %in% names(dataset)) {
    dataset <- dataset |> filter(!.data$is_satellite_peak)
    nsat_peaks <- nall - nrow(dataset)
    details <- c(details, sprintf("%d satellite peaks", nsat_peaks))
  }
  if ("is_weak_isotopocule" %in% names(dataset)) {
    dataset <- dataset |> filter(!.data$is_weak_isotopocule)
    nweak_isos <- nall - nsat_peaks - nrow(dataset)
    details <- c(details, sprintf("%d weak isotopocule peaks", nweak_isos))
  }
  if ("is_outlier" %in% names(dataset)) {
    dataset <- dataset |> filter(!.data$is_outlier)
    noutliers <- nall - nsat_peaks - nweak_isos - nrow(dataset)
    details <- c(details, sprintf("%d outlier peaks", noutliers))
  }

  # info
  sprintf(
    "removed %d flagged records (%.1f%%)%s",
    nsat_peaks + nweak_isos + noutliers,
    100 * (nsat_peaks + nweak_isos + noutliers) / nall,
    if (length(details) > 0) {
      sprintf(" - %s", paste(details, collapse = ", "))
    } else {
      ""
    }
  ) |>
    finish_info(start = start)

  # return
  return(dataset |> droplevels())
}


#' @title Define the denominator for ratio calculation
#' @description `orbi_define_basepeak()` sets one isotopocule in the data frame as the base peak (ratio denominator) and calculates the instantaneous isotope ratios against it.
#' @inheritParams orbi_flag_satellite_peaks
#' @param basepeak_def The isotopocule that gets defined as base peak, i.e. the denominator to calculate ratios
#'
#' @examples
#' fpath <- system.file("extdata", "testfile_flow.isox", package = "isoorbi")
#' df <- orbi_read_isox(file = fpath) |>
#'   orbi_simplify_isox() |>
#'   orbi_define_basepeak(basepeak_def = "M0")
#'
#' @returns same object as provided in `dataset` without the rows of the basepeak isotopocule and instead three new columns called `basepeak`, `basepeak_ions`, and `ratio` holding the basepeak information and the isotope ratios vs. the base peak
#' @export
orbi_define_basepeak <- function(dataset, basepeak_def) {
  # safety checks
  check_dataset_arg(dataset)
  is_agg <- is(dataset, "orbi_aggregated_data")
  peaks <- if (is_agg) {
    dataset$peaks |>
      dplyr::left_join(
        dataset$file_info |> dplyr::select("uidx", "filename"),
        by = "uidx"
      )
  } else {
    dataset
  }

  ## columns
  check_tibble(
    peaks,
    c("uidx|filename", "scan.no"),
    regexps = TRUE,
    .arg = "dataset"
  )
  if (!"isotopocule" %in% names(peaks)) {
    # more specialized error message
    cli_abort(
      "no {.field isotopocule} column, make sure to run {.strong orbi_identify_isotopocules()} first"
    )
  }
  if (!"ions.incremental" %in% names(peaks)) {
    # more specialized error message
    cli_abort(
      "no {.field ions.incremental} column, make sure to run {.strong orbi_calculate_ions()} first"
    )
  }

  ## basepeak_def
  check_arg(
    basepeak_def,
    !missing(
      basepeak_def
    ) &&
      rlang::is_scalar_character(basepeak_def),
    "must be a single text value identifying the isotopocule to use as the basepeak"
  )
  ## basepeak_def in isotopocules
  peaks <- peaks |> factorize_dataset(cols = "isotopocule")
  check_arg(
    basepeak_def,
    basepeak_def %in% levels(peaks$isotopocule),
    format_inline(
      "is not an isotopocule in the dataset, available: {.field {levels(peaks$isotopocule)}}"
    ),
    include_type = FALSE
  )

  # info message
  start <- start_info()

  # identify `basepeak` for each scan
  potential_bys <- c("uidx", "filename", "compound", "fragment", "scan.no")
  out <-
    tryCatch(
      {
        df.out <-
          peaks |>
          dplyr::mutate(basepeak = !!basepeak_def) |>
          dplyr::mutate(
            .by = dplyr::any_of(potential_bys),
            # flag the basepeak
            ..is_bp = .data$isotopocule == .data$basepeak &
              if ("is_satellite_peak" %in% names(peaks)) {
                !.data$is_satellite_peak
              } else {
                TRUE
              },
            ..n_too_few = sum(.data$..is_bp) == 0L,
            ..n_too_many = sum(.data$..is_bp) > 1L,
          )

        # check if there are issues
        if (any(df.out$..n_too_many)) {
          # there are some scans with too many baepeaks!
          if (!"is_satellite_peak" %in% names(peaks)) {
            cli_abort(
              "the {.field {basepeak_def}} isotopocule exists multiple times in some scans, make sure to run {.strong orbi_flag_satellite_peaks()} first",
              call = expr(NULL)
            )
          } else {
            cli_abort(
              "the {.field {basepeak_def}} isotopocule exists multiple times in some scans despite satellite peaks already flagged - this should not be possible, please file a bug report and provide a dataset that reproduces this issue",
              call = expr(NULL)
            )
          }
        } else if (any(df.out$..n_too_few)) {
          # base peak is missing in some groupings
          df.info <- df.out |>
            dplyr::select(dplyr::any_of(potential_bys), "..n_too_few") |>
            dplyr::distinct()
          groupings <- setdiff(names(df.info), c("uidx", "..n_too_few"))
          cli_bullets(
            c(
              "!" = "{cli::col_yellow('Warning')}: {format_number(sum(df.info$..n_too_few))}/{format_number(nrow(df.info))} data groups ({.field {paste(groupings, collapse = ' + ')}}) cannot be used because the {.field {basepeak_def}} isotopocule is missing",
              "i" = "To investigate, use {.strong orbi_flag_weak_isotopocules()} and {.strong orbi_plot_isotopocule_coverage()} or {.strong orbi_get_isotopocule_coverage()} {.emph before} defining the base peak"
            )
          )
          df.out <- df.out |> dplyr::filter(!.data$..n_too_few)
        }

        # return
        df.out |>
          dplyr::mutate(
            .by = dplyr::any_of(potential_bys),
            basepeak_ions = .data$ions.incremental[.data$..is_bp]
          ) |>
          dplyr::select(-"..is_bp", -"..n_too_few", -"..n_too_many")
      }
    ) |>
    try_catch_cnds()

  # abort if issues
  abort_cnds(
    out$conditions,
    message = "something went wrong identifying the base peak:"
  )

  # remove basepeak from isotopocule list
  out <-
    out$result |>
    dplyr::filter(.data$isotopocule != basepeak_def) |>
    droplevels() |>
    try_catch_cnds()

  # abort if issues
  abort_cnds(
    out$conditions,
    message = "something went wrong removing the base peak isotopocule:"
  )

  # calculate ratios (only if no error yet)
  out <-
    out$result |>
    dplyr::mutate(
      ratio = .data$ions.incremental / .data$basepeak_ions,
      .after = "ions.incremental"
    ) |>
    try_catch_cnds()

  # abort if issues
  abort_cnds(
    out$conditions,
    message = "something went wrong calculating ratios:"
  )

  # result
  df.out <- out$result

  # info message
  finish_info(
    "set {.field {basepeak_def}} as the ratio denominator ",
    if (is.data.frame(df.out)) {
      "and calculated {format_number(nrow(df.out))} {.field ratio} values "
    },
    if (is.data.frame(df.out) && "isotopocule" %in% names(df.out)) {
      "for {length(levels(df.out$isotopocule))} isotopocules ({.field {levels(df.out$isotopocule)}})"
    },
    start = start,
    conditions = out$conditions,
    # don't proceed if there are issues
    abort_if_errors = TRUE
  )

  # return
  if (is_agg) {
    # got aggregated data to begin with --> return aggregated data
    dataset$peaks <- df.out |> dplyr::select(-dplyr::any_of("filename"))
    return(dataset)
  } else {
    # got a plain peaks tibble
    return(df.out)
  }
}
