#include <Rcpp.h>
using namespace Rcpp;

namespace {

double rstable(double alpha, double gamma) {
    const double BIG = std::numeric_limits<double>::max() / 100.0;
    double result;
    double u = R::runif(0.0, 1.0);
    double w = R::rexp(1.0);
    if (alpha == 2.0) {
        // Gaussian -- Box-Muller 
        result = 2.0 * gamma * sqrt(w) * cos(M_2PI * u);
    } else if (alpha == 1.0) {
        // Cauchy
        result = gamma * tan(M_PI * (u - 0.5));
    } else {
        // General case --  Chambers-Mallows-Stuck
        double theta = M_PI * (u - 0.5);
        double alpha_theta = alpha * theta;
        double cos_theta = cos(theta);
        double numerator = sin(alpha_theta);
        double denominator = pow(cos_theta, 1.0 / alpha);
        double factor =
            pow(cos(theta - alpha_theta) / w, (1.0 - alpha) / alpha);
        double x = (numerator / denominator) * factor;
        result = gamma * x;
    }
    // Check the result and if not finite substitute +/- BIG
    if (std::isinf(result)) {
        result = (result > 0) ? BIG : -BIG;
    } else if (std::isnan(result)) {
        result = (u > 0.5) ? BIG : -BIG;
    }
    return result;
}
}  // namespace

//' @rdname toad-model
//' @name toad-model
//' @param theta a vector of length 3 containing the model parameters
//' @param data a (number of days)x(number of toads) numerical matrix 
//'        containing the locations of a set of toads. The simulated
//'        dataset will replicates the size, the NA pattern and
//'        the initial positions (first row) of this argument. 
//' @return
//' `toadSim` return a numerical matrix of the same size of `data`
//' @export
// [[Rcpp::export]]
NumericMatrix toadSim(NumericVector theta, NumericMatrix data) {
    int i, j, k, nd = data.nrow(), nt = data.ncol();
    NumericMatrix ans = clone(data);
    NumericVector ot(nd);
    double alpha = theta[0], gamma = theta[1], pr = theta[2];
    double *sim = ans.begin();
    for (i = 0; i < nt; i++, sim += nd) {
        for (k = nd - 1; (k >= 0) && R_IsNA(sim[k]); k--);
        for (j = 1, ot[0] = sim[0]; j <= k; j++) {
            if (R::unif_rand() < pr) {
                ot[j] = ot[std::floor(R::runif(0, j))];
            } else {
                ot[j] = ot[j-1]+rstable(alpha, gamma);
            }
            if (!R_IsNA(sim[j])) sim[j] = ot[j];
        }
    }
    return ans;
}
