/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define H5G_FRIEND 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5Gpkg.h"      
#include "H5Iprivate.h"  
#include "H5Oprivate.h"  
#include "H5VLprivate.h" 

#include "H5VLnative_private.h" 

void *
H5VL__native_group_create(void *obj, const H5VL_loc_params_t *loc_params, const char *name, hid_t lcpl_id,
                          hid_t gcpl_id, hid_t H5_ATTR_UNUSED gapl_id, hid_t H5_ATTR_UNUSED dxpl_id,
                          void H5_ATTR_UNUSED **req)
{
    H5G_loc_t loc;        
    H5G_t    *grp = NULL; 
    void     *ret_value;

    FUNC_ENTER_PACKAGE

    
    if (H5G_loc_real(obj, loc_params->obj_type, &loc) < 0)
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "not a file or file object");

    
    if (name == NULL) {
        H5G_obj_create_t gcrt_info; 

        
        gcrt_info.gcpl_id    = gcpl_id;
        gcrt_info.cache_type = H5G_NOTHING_CACHED;
        memset(&gcrt_info.cache, 0, sizeof(gcrt_info.cache));

        
        if (NULL == (grp = H5G__create(loc.oloc->file, &gcrt_info)))
            HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, NULL, "unable to create group");
    } 
    
    else {
        
        if (NULL == (grp = H5G__create_named(&loc, name, lcpl_id, gcpl_id)))
            HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, NULL, "unable to create group");
    } 

    ret_value = (void *)grp;

done:
    if (name == NULL) {
        
        if (grp) {
            H5O_loc_t *oloc; 

            
            if (NULL == (oloc = H5G_oloc(grp)))
                HDONE_ERROR(H5E_SYM, H5E_CANTGET, NULL, "unable to get object location of group");

            
            if (H5O_dec_rc_by_loc(oloc) < 0)
                HDONE_ERROR(H5E_SYM, H5E_CANTDEC, NULL,
                            "unable to decrement refcount on newly created object");
        } 
    }     

    FUNC_LEAVE_NOAPI(ret_value)
} 

void *
H5VL__native_group_open(void *obj, const H5VL_loc_params_t *loc_params, const char *name,
                        hid_t H5_ATTR_UNUSED gapl_id, hid_t H5_ATTR_UNUSED dxpl_id, void H5_ATTR_UNUSED **req)
{
    H5G_loc_t loc;        
    H5G_t    *grp = NULL; 
    void     *ret_value;

    FUNC_ENTER_PACKAGE

    
    if (H5G_loc_real(obj, loc_params->obj_type, &loc) < 0)
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "not a file or file object");

    
    if ((grp = H5G__open_name(&loc, name)) == NULL)
        HGOTO_ERROR(H5E_SYM, H5E_CANTOPENOBJ, NULL, "unable to open group");

    ret_value = (void *)grp;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5VL__native_group_get(void *obj, H5VL_group_get_args_t *args, hid_t H5_ATTR_UNUSED dxpl_id,
                       void H5_ATTR_UNUSED **req)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    switch (args->op_type) {
        
        case H5VL_GROUP_GET_GCPL: {
            if ((args->args.get_gcpl.gcpl_id = H5G_get_create_plist((H5G_t *)obj)) < 0)
                HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "can't get creation property list for group");

            break;
        }

        
        case H5VL_GROUP_GET_INFO: {
            H5VL_group_get_info_args_t *get_info_args = &args->args.get_info;
            H5G_loc_t                   loc;

            if (H5G_loc_real(obj, get_info_args->loc_params.obj_type, &loc) < 0)
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a file or file object");

            if (get_info_args->loc_params.type == H5VL_OBJECT_BY_SELF) {
                

                
                if (H5G__obj_info(loc.oloc, get_info_args->ginfo) < 0)
                    HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "can't retrieve group info");
            } 
            else if (get_info_args->loc_params.type == H5VL_OBJECT_BY_NAME) {
                

                
                if (H5G__get_info_by_name(&loc, get_info_args->loc_params.loc_data.loc_by_name.name,
                                          get_info_args->ginfo) < 0)
                    HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "can't retrieve group info");
            } 
            else if (get_info_args->loc_params.type == H5VL_OBJECT_BY_IDX) {
                

                
                if (H5G__get_info_by_idx(&loc, get_info_args->loc_params.loc_data.loc_by_idx.name,
                                         get_info_args->loc_params.loc_data.loc_by_idx.idx_type,
                                         get_info_args->loc_params.loc_data.loc_by_idx.order,
                                         get_info_args->loc_params.loc_data.loc_by_idx.n,
                                         get_info_args->ginfo) < 0)
                    HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "can't retrieve group info");
            } 
            else
                HGOTO_ERROR(H5E_VOL, H5E_UNSUPPORTED, FAIL, "unknown get info parameters");
            break;
        }

        default:
            HGOTO_ERROR(H5E_VOL, H5E_CANTGET, FAIL, "can't get this type of information from group");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5VL__native_group_specific(void *obj, H5VL_group_specific_args_t *args, hid_t H5_ATTR_UNUSED dxpl_id,
                            void H5_ATTR_UNUSED **req)
{
    H5G_t *grp       = (H5G_t *)obj;
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    switch (args->op_type) {
        
        case H5VL_GROUP_MOUNT: {
            H5G_loc_t loc;

            if (H5G_loc_real(grp, H5I_GROUP, &loc) < 0)
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a group object");

            if (H5F_mount(&loc, args->args.mount.name, args->args.mount.child_file,
                          args->args.mount.fmpl_id) < 0)
                HGOTO_ERROR(H5E_FILE, H5E_MOUNT, FAIL, "unable to mount file");

            break;
        }

        
        case H5VL_GROUP_UNMOUNT: {
            H5G_loc_t loc;

            if (H5G_loc_real(grp, H5I_GROUP, &loc) < 0)
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a group object");

            if (H5F_unmount(&loc, args->args.unmount.name) < 0)
                HGOTO_ERROR(H5E_FILE, H5E_UNMOUNT, FAIL, "unable to unmount file");

            break;
        }

        
        case H5VL_GROUP_FLUSH: {
            
            if (H5F_HAS_FEATURE(grp->oloc.file, H5FD_FEAT_HAS_MPI))
                HGOTO_ERROR(H5E_SYM, H5E_UNSUPPORTED, FAIL, "H5Oflush isn't supported for parallel");

            if (H5O_flush_common(&grp->oloc, args->args.flush.grp_id) < 0)
                HGOTO_ERROR(H5E_SYM, H5E_CANTFLUSH, FAIL, "unable to flush group");

            break;
        }

        
        case H5VL_GROUP_REFRESH: {
            if ((H5O_refresh_metadata(&grp->oloc, args->args.refresh.grp_id)) < 0)
                HGOTO_ERROR(H5E_SYM, H5E_CANTLOAD, FAIL, "unable to refresh group");

            break;
        }

        default:
            HGOTO_ERROR(H5E_VOL, H5E_UNSUPPORTED, FAIL, "invalid specific operation");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5VL__native_group_optional(void H5_ATTR_UNUSED *obj, H5VL_optional_args_t *args,
                            hid_t H5_ATTR_UNUSED dxpl_id, void H5_ATTR_UNUSED **req)
{
#ifndef H5_NO_DEPRECATED_SYMBOLS
    H5VL_native_group_optional_args_t *opt_args = args->args; 
#endif                                                        
    herr_t ret_value = SUCCEED;                               

    FUNC_ENTER_PACKAGE

    switch (args->op_type) {
#ifndef H5_NO_DEPRECATED_SYMBOLS
        
        case H5VL_NATIVE_GROUP_ITERATE_OLD: {
            H5VL_native_group_iterate_old_t *iter_args = &opt_args->iterate_old;
            H5G_link_iterate_t               lnk_op; 
            H5G_loc_t                        grp_loc;

            
            if (H5G_loc_real(obj, iter_args->loc_params.obj_type, &grp_loc) < 0)
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a file or file object");

            
            lnk_op.op_type        = H5G_LINK_OP_OLD;
            lnk_op.op_func.op_old = iter_args->op;

            
            if ((ret_value = H5G_iterate(&grp_loc, iter_args->loc_params.loc_data.loc_by_name.name,
                                         H5_INDEX_NAME, H5_ITER_INC, iter_args->idx, iter_args->last_obj,
                                         &lnk_op, iter_args->op_data)) < 0)
                HERROR(H5E_SYM, H5E_BADITER, "error iterating over group's links");

            break;
        }

        
        case H5VL_NATIVE_GROUP_GET_OBJINFO: {
            H5VL_native_group_get_objinfo_t *goi_args = &opt_args->get_objinfo;
            H5G_loc_t                        grp_loc;

            
            if (H5G_loc_real(obj, goi_args->loc_params.obj_type, &grp_loc) < 0)
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a file or file object");

            
            if (H5G__get_objinfo(&grp_loc, goi_args->loc_params.loc_data.loc_by_name.name,
                                 goi_args->follow_link, goi_args->statbuf) < 0)
                HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "cannot stat object");

            break;
        }
#endif 

        default:
            HGOTO_ERROR(H5E_VOL, H5E_UNSUPPORTED, FAIL, "invalid optional operation");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5VL__native_group_close(void *grp, hid_t H5_ATTR_UNUSED dxpl_id, void H5_ATTR_UNUSED **req)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    if (H5G_close((H5G_t *)grp) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CLOSEERROR, FAIL, "can't close group");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
