/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Tmodule.h" 

#include "H5private.h"  
#include "H5Eprivate.h" 
#include "H5Iprivate.h" 
#include "H5Tpkg.h"     

herr_t
H5Tget_fields(hid_t type_id, size_t *spos , size_t *epos , size_t *esize ,
              size_t *mpos , size_t *msize )
{
    H5T_t *dt;                  
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (dt = (H5T_t *)H5I_object_verify(type_id, H5I_DATATYPE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype");
    while (dt->shared->parent)
        dt = dt->shared->parent; 
    if (H5T_FLOAT != dt->shared->type)
        HGOTO_ERROR(H5E_DATATYPE, H5E_BADTYPE, FAIL, "operation not defined for datatype class");

    
    if (spos)
        *spos = dt->shared->u.atomic.u.f.sign;
    if (epos)
        *epos = dt->shared->u.atomic.u.f.epos;
    if (esize)
        *esize = dt->shared->u.atomic.u.f.esize;
    if (mpos)
        *mpos = dt->shared->u.atomic.u.f.mpos;
    if (msize)
        *msize = dt->shared->u.atomic.u.f.msize;

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Tset_fields(hid_t type_id, size_t spos, size_t epos, size_t esize, size_t mpos, size_t msize)
{
    H5T_t *dt;                  
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (dt = (H5T_t *)H5I_object_verify(type_id, H5I_DATATYPE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype");
    if (H5T_STATE_TRANSIENT != dt->shared->state)
        HGOTO_ERROR(H5E_ARGS, H5E_CANTSET, FAIL, "datatype is read-only");
    while (dt->shared->parent)
        dt = dt->shared->parent; 
    if (H5T_FLOAT != dt->shared->type)
        HGOTO_ERROR(H5E_DATATYPE, H5E_BADTYPE, FAIL, "operation not defined for datatype class");
    if (epos + esize - dt->shared->u.atomic.offset > dt->shared->u.atomic.prec)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "exponent bit field size/location is invalid");
    if (mpos + msize - dt->shared->u.atomic.offset > dt->shared->u.atomic.prec)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "mantissa bit field size/location is invalid");
    if (spos - dt->shared->u.atomic.offset >= dt->shared->u.atomic.prec)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "sign location is not valid");

    
    if (spos >= epos && spos < epos + esize)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "sign bit appears within exponent field");
    if (spos >= mpos && spos < mpos + msize)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "sign bit appears within mantissa field");
    if ((mpos < epos && mpos + msize > epos) || (epos < mpos && epos + esize > mpos))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "exponent and mantissa fields overlap");

    
    dt->shared->u.atomic.u.f.sign  = spos;
    dt->shared->u.atomic.u.f.epos  = epos;
    dt->shared->u.atomic.u.f.mpos  = mpos;
    dt->shared->u.atomic.u.f.esize = esize;
    dt->shared->u.atomic.u.f.msize = msize;

done:
    FUNC_LEAVE_API(ret_value)
} 

size_t
H5Tget_ebias(hid_t type_id)
{
    H5T_t *dt;        
    size_t ret_value; 

    FUNC_ENTER_API(0)

    
    if (NULL == (dt = (H5T_t *)H5I_object_verify(type_id, H5I_DATATYPE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, 0, "not a datatype");
    while (dt->shared->parent)
        dt = dt->shared->parent; 
    if (H5T_FLOAT != dt->shared->type)
        HGOTO_ERROR(H5E_DATATYPE, H5E_BADTYPE, 0, "operation not defined for datatype class");

    
    H5_CHECKED_ASSIGN(ret_value, size_t, dt->shared->u.atomic.u.f.ebias, uint64_t);

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Tset_ebias(hid_t type_id, size_t ebias)
{
    H5T_t *dt;                  
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (dt = (H5T_t *)H5I_object_verify(type_id, H5I_DATATYPE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype");
    if (H5T_STATE_TRANSIENT != dt->shared->state)
        HGOTO_ERROR(H5E_ARGS, H5E_CANTSET, FAIL, "datatype is read-only");
    while (dt->shared->parent)
        dt = dt->shared->parent; 
    if (H5T_FLOAT != dt->shared->type)
        HGOTO_ERROR(H5E_DATATYPE, H5E_BADTYPE, FAIL, "operation not defined for datatype class");

    
    dt->shared->u.atomic.u.f.ebias = ebias;

done:
    FUNC_LEAVE_API(ret_value)
} 

H5T_norm_t
H5Tget_norm(hid_t type_id)
{
    H5T_t     *dt;        
    H5T_norm_t ret_value; 

    FUNC_ENTER_API(H5T_NORM_ERROR)

    
    if (NULL == (dt = (H5T_t *)H5I_object_verify(type_id, H5I_DATATYPE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5T_NORM_ERROR, "not a datatype");
    while (dt->shared->parent)
        dt = dt->shared->parent; 
    if (H5T_FLOAT != dt->shared->type)
        HGOTO_ERROR(H5E_DATATYPE, H5E_BADTYPE, H5T_NORM_ERROR, "operation not defined for datatype class");

    
    ret_value = dt->shared->u.atomic.u.f.norm;

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Tset_norm(hid_t type_id, H5T_norm_t norm)
{
    H5T_t *dt;                  
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (dt = (H5T_t *)H5I_object_verify(type_id, H5I_DATATYPE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype");
    if (H5T_STATE_TRANSIENT != dt->shared->state)
        HGOTO_ERROR(H5E_ARGS, H5E_CANTSET, FAIL, "datatype is read-only");
    if (norm < H5T_NORM_IMPLIED || norm > H5T_NORM_NONE)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "illegal normalization");
    while (dt->shared->parent)
        dt = dt->shared->parent; 
    if (H5T_FLOAT != dt->shared->type)
        HGOTO_ERROR(H5E_DATATYPE, H5E_BADTYPE, FAIL, "operation not defined for datatype class");

    
    dt->shared->u.atomic.u.f.norm = norm;

done:
    FUNC_LEAVE_API(ret_value)
} 

H5T_pad_t
H5Tget_inpad(hid_t type_id)
{
    H5T_t    *dt;        
    H5T_pad_t ret_value; 

    FUNC_ENTER_API(H5T_PAD_ERROR)

    
    if (NULL == (dt = (H5T_t *)H5I_object_verify(type_id, H5I_DATATYPE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5T_PAD_ERROR, "not a datatype");
    while (dt->shared->parent)
        dt = dt->shared->parent; 
    if (H5T_FLOAT != dt->shared->type)
        HGOTO_ERROR(H5E_DATATYPE, H5E_BADTYPE, H5T_PAD_ERROR, "operation not defined for datatype class");

    
    ret_value = dt->shared->u.atomic.u.f.pad;

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Tset_inpad(hid_t type_id, H5T_pad_t pad)
{
    H5T_t *dt;                  
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (dt = (H5T_t *)H5I_object_verify(type_id, H5I_DATATYPE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype");
    if (H5T_STATE_TRANSIENT != dt->shared->state)
        HGOTO_ERROR(H5E_ARGS, H5E_CANTSET, FAIL, "datatype is read-only");
    if (pad < H5T_PAD_ZERO || pad >= H5T_NPAD)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "illegal internal pad type");
    while (dt->shared->parent)
        dt = dt->shared->parent; 
    if (H5T_FLOAT != dt->shared->type)
        HGOTO_ERROR(H5E_DATATYPE, H5E_BADTYPE, FAIL, "operation not defined for datatype class");

    
    dt->shared->u.atomic.u.f.pad = pad;

done:
    FUNC_LEAVE_API(ret_value)
} 
