% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/benchmarking.R
\name{benchmarking}
\alias{benchmarking}
\title{Restore temporal constraints}
\usage{
benchmarking(
  series_df,
  benchmarks_df,
  rho,
  lambda,
  biasOption,
  bias = NA,
  tolV = 0.001,
  tolP = NA,
  warnNegResult = TRUE,
  tolN = -0.001,
  var = "value",
  with = NULL,
  by = NULL,
  verbose = FALSE,

  # New in G-Series 3.0
  constant = 0,
  negInput_option = 0,
  allCols = FALSE,
  quiet = FALSE
)
}
\arguments{
\item{series_df}{(mandatory)

Data frame (object of class "data.frame") that contains the indicator time series data to be
benchmarked. In addition to the series data variable(s), specified with argument \code{var}, the data
frame must also contain two numeric variables, \code{year} and \code{period}, identifying the periods of
the indicator time series.}

\item{benchmarks_df}{(mandatory)

Data frame (object of class "data.frame") that contains the benchmarks. In addition to the benchmarks
data variable(s), specified with argument \code{with}, the data frame must also contain four numeric
variables, \code{startYear}, \code{startPeriod}, \code{endYear} and \code{endPeriod}, identifying the indicator time
series periods covered by each benchmark.}

\item{rho}{(mandatory)

Real number in the \eqn{[0,1]} interval that specifies the value of the autoregressive parameter
\eqn{\rho}. See section \strong{Details} for more information on the effect of parameter \eqn{\rho}.}

\item{lambda}{(mandatory)

Real number, with suggested values in the \eqn{[-3,3]} interval, that specifies the value of the
adjustment model parameter \eqn{\lambda}. Typical values are \code{lambda = 0.0} for an additive model
and \code{lambda = 1.0} for a proportional model.}

\item{biasOption}{(mandatory)

Specification of the bias estimation option:
\itemize{
\item \code{1}: Do not estimate the bias. The bias used to correct the indicator series will be the value
specified with argument \code{bias}.
\item \code{2}: Estimate the bias, display the result, but do not use it. The bias used to correct the indicator
series will be the value specified with argument \code{bias}.
\item \code{3}: Estimate the bias, display the result and use the estimated bias to correct the indicator series.
Any value specified with argument \code{bias} will be ignored.
}

Argument \code{biasOption} is ignored when \code{rho = 1.0}. See section \strong{Details} for more information on the bias.}

\item{bias}{(optional)

Real number, or \code{NA}, specifying the value of the user-defined bias to be used for the correction of the
indicator series prior to benchmarking. The bias is added to the indicator series with an additive model
(argument  \code{lambda = 0.0}) while it is multiplied otherwise (argument \code{lambda != 0.0}). No bias
correction is applied when \code{bias = NA}, which is equivalent to specifying \code{bias = 0.0} when \code{lambda = 0.0}
and \code{bias = 1.0} otherwise. Argument \code{bias} is ignored when \code{biasOption = 3} or \code{rho = 1.0}. See section
\strong{Details} for more information on the bias.

\strong{Default value} is \code{bias = NA} (no user-defined bias).}

\item{tolV, tolP}{(optional)

Nonnegative real number, or \code{NA}, specifying the tolerance, in absolute value or percentage, to be used
for the validation of the output binding benchmarks (alterability coefficient of \eqn{0.0}). This validation
compares the input binding benchmark values with the equivalent values calculated from the benchmarked series
(output) data. Arguments \code{tolV} and \code{tolP} cannot be both specified (one must be specified while the other
must be \code{NA}).

\strong{Example:} to set a tolerance of 10 \emph{units}, specify \verb{tolV = 10, tolP = NA}; to set a tolerance of 1\%,
specify \verb{tolV = NA, tolP = 0.01}.

\strong{Default values} are \code{tolV = 0.001} and \code{tolP = NA}.}

\item{warnNegResult}{(optional)

Logical argument specifying whether a warning message is generated when a negative value created by the
function in the benchmarked (output) series is smaller than the threshold specified by argument \code{tolN}.

\strong{Default value} is \code{warnNegResult = TRUE}.}

\item{tolN}{(optional)

Negative real number specifying the threshold for the identification of negative values. A value is
considered negative when it is smaller than this threshold.

\strong{Default value} is \code{tolN = -0.001}.}

\item{var}{(optional)

String vector (minimum length of 1) specifying the variable name(s) in the indicator series data frame
(argument \code{series_df}) containing the values and (optionally) the user-defined alterability coefficients
of the series to be benchmarked. These variables must be numeric.

The syntax is \code{var = c("series1 </ alt_ser1>", "series2 </ alt_ser2>", ...)}. Default alterability
coefficients of \eqn{1.0} are used when a user-defined alterability coefficients variable is not specified
alongside an indicator series variable. See section \strong{Details} for more information on alterability coefficients.

\strong{Example:} \code{var = "value / alter"} would benchmark indicator series data frame variable \code{value} with the
alterability coefficients contained in variable \code{alter} while \code{var = c("value / alter", "value2")} would
additionally benchmark variable \code{value2} with default alterability coefficients of \eqn{1.0}.

\strong{Default value} is \code{var = "value"} (benchmark variable \code{value} using default alterability coefficients of
\eqn{1.0}).}

\item{with}{(optional)

String vector (same length as argument \code{var}), or \code{NULL}, specifying the variable name(s) in the benchmarks data
frame (argument \code{benchmarks_df}) containing the values and (optionally) the user-defined alterability coefficients
of the benchmarks. These variables must be numeric. Specifying \code{with = NULL} results in using benchmark variable(s)
with the same names(s) as those specified with argument \code{var} without user-defined benchmark alterability coefficients
(i.e., default alterability coefficients of \eqn{0.0} corresponding to binding benchmarks).

The syntax is \code{with = NULL} or \code{with = c("bmk1 </ alt_bmk1>", "bmk2 </ alt_bmk2>", ...)}. Default alterability
coefficients of \eqn{0.0} (binding benchmarks) are used when a user-defined alterability coefficients variable
is not specified alongside a benchmark variable. See section \strong{Details} for more information on alterability
coefficients.

\strong{Example:} \code{with = "val_bmk"} would use benchmarks data frame variable \code{val_bmk} with default benchmark
alterability coefficients of \eqn{0.0} to benchmark the indicator series while
\code{with = c("val_bmk", "val_bmk2 / alt_bmk2")} would additionally benchmark a second indicator series using
benchmark variable \code{val_bmk2} with the benchmark alterability coefficients contained in variable \code{alt_bmk2}.

\strong{Default value} is \code{with = NULL} (same benchmark variable(s) as argument \code{var} using default benchmark
alterability coefficients of \eqn{0.0}).}

\item{by}{(optional)

String vector (minimum length of 1), or \code{NULL}, specifying the variable name(s) in the input data frames
(arguments \code{series_df} and \code{benchmarks_df}) to be used to form groups (for \emph{BY-group} processing) and allow
the benchmarking of multiple series in a single function call. BY-group variables can be numeric or
character (factors or not), must be present in both input data frames and will appear in all three output
data frames (see section \strong{Value}). BY-group processing is not  implemented when \code{by = NULL}. See
"Benchmarking Multiple Series" in section \strong{Details} for more information.

\strong{Default value} is \code{by = NULL} (no BY-group processing).}

\item{verbose}{(optional)

Logical argument specifying whether information on intermediate steps with execution time (real time,
not CPU time) should be displayed. Note that specifying argument \code{quiet = TRUE} would \emph{nullify} argument
\code{verbose}.

\strong{Default value} is \code{verbose = FALSE}.}

\item{constant}{(optional)

Real number that specifies a value to be temporarily added to both the indicator series and the
benchmarks before solving proportional benchmarking problems (\code{lambda != 0.0}). The temporary
constant is removed from the final output benchmarked series. E.g., specifying a (small) constant
would allow proportional benchmarking with \code{rho = 1} (e.g., proportional Denton benchmarking) on
indicator series that include values of 0. Otherwise, proportional benchmarking with values of 0 in
the indicator series is only possible when \code{rho < 1}. Specifying a constant with additive benchmarking
(\code{lambda = 0.0}) has no impact on the resulting benchmarked data. The data variables in the \code{graphTable}
output data frame include the constant, corresponding to the benchmarking problem that was actually solved.

\strong{Default value} is \code{constant = 0} (no temporary additive constant).}

\item{negInput_option}{(optional)

Handling of negative values in the input data for proportional benchmarking (\code{lambda != 0.0}):
\itemize{
\item \code{0}: Do not allow negative values with proportional benchmarking. An error message is displayed in the
presence of negative values in the input indicator series or benchmarks and missing (\code{NA}) values are
returned for the benchmarked series. This corresponds to the G-Series 2.0 behaviour.
\item \code{1}: Allow negative values with proportional benchmarking but display a warning message.
\item \code{2}: Allow negative values with proportional benchmarking without displaying any message.
}

\strong{Default value} is \code{negInput_option = 0} (do not allow negative values with proportional benchmarking).}

\item{allCols}{(optional)

Logical argument specifying whether all variables in the indicator series data frame (argument \code{series_df}),
other than \code{year} and \code{period}, determine the set of series to benchmark. Values specified with arguments \code{var}
and \code{with} are ignored when \code{allCols = TRUE}, which automatically implies default alterability coefficients,
and variables with the same names as the indicator series must exist in the benchmarks data frame (argument
\code{benchmarks_df}).

\strong{Default value} is \code{allCols = FALSE}.}

\item{quiet}{(optional)

Logical argument specifying whether or not to display only essential information such as warning messages,
error messages and variable (series) or BY-group information when multiple series are benchmarked in a single call
to the function. We advise against \emph{wrapping} your \code{\link[=benchmarking]{benchmarking()}} call with \code{\link[=suppressMessages]{suppressMessages()}} to further suppress
the display of variable (series) or BY-group information when processing multiple series as this would make
troubleshooting difficult in case of issues with individual series. Note that specifying \code{quiet = TRUE} would
also \emph{nullify} argument \code{verbose}.

\strong{Default value} is \code{quiet = FALSE}.}
}
\value{
The function returns is a list of three data frames:
\itemize{
\item \code{series}: data frame containing the benchmarked data (primary function output). BY variables specified
with argument \code{by} would be included in the data frame but not alterability coefficient variables
specified with argument \code{var}.
\item \code{benchmarks}: copy of the input benchmarks data frame (excluding invalid benchmarks when applicable).
BY variables specified with argument \code{by} would be included in the data frame but not alterability
coefficient variables specified with argument \code{with}.
\item \code{graphTable}: data frame containing supplementary data useful for producing analytical tables and graphs
(see function \code{\link[=plot_graphTable]{plot_graphTable()}}). It contains the following variables in addition to the BY variables
specified with argument \code{by}:
\itemize{
\item \code{varSeries}: Name of the indicator series variable
\item \code{varBenchmarks}: Name of the benchmark variable
\item \code{altSeries}: Name of the user-defined indicator series alterability coefficients variable
\item \code{altSeriesValue}: Indicator series alterability coefficients
\item \code{altbenchmarks}: Name of the user-defined benchmark alterability coefficients variable
\item \code{altBenchmarksValue}: Benchmark alterability coefficients
\item \code{t}: Indicator series period identifier (1 to \eqn{T})
\item \code{m}: Benchmark coverage periods identifier (1 to \eqn{M})
\item \code{year}: Data point calendar year
\item \code{period}: Data point period (cycle) value (1 to \code{periodicity})
\item \code{constant}: Temporary additive constant (argument \code{constant})
\item \code{rho}: Autoregressive parameter \eqn{\rho} (argument \code{rho})
\item \code{lambda}: Adjustment model parameter \eqn{\lambda} (argument \code{lambda})
\item \code{bias}: Bias adjustment (default, user-defined or estimated bias according to arguments \code{biasOption} and \code{bias})
\item \code{periodicity}: The maximum number of periods in a year (e.g. 4 for a quarterly indicator series)
\item \code{date}: Character string combining the values of variables \code{year} and \code{period}
\item \code{subAnnual}: Indicator series values
\item \code{benchmarked}: Benchmarked series values
\item \code{avgBenchmark}: Benchmark values divided by the number of coverage periods
\item \code{avgSubAnnual}: Indicator series values (variable \code{subAnnual}) averaged over the benchmark coverage period
\item \code{subAnnualCorrected}: Bias corrected indicator series values
\item \code{benchmarkedSubAnnualRatio}: Difference (\eqn{\lambda = 0}) or ratio (\eqn{\lambda \ne 0}{lambda != 0}) of the values
of variables \code{benchmarked} and \code{subAnnual}
\item \code{avgBenchmarkSubAnnualRatio}: Difference (\eqn{\lambda = 0}) or ratio (\eqn{\lambda \ne 0}{lambda != 0}) of the values
of variables \code{avgBenchmark} and \code{avgSubAnnual}
\item \code{growthRateSubAnnual}: Period to period difference (\eqn{\lambda = 0}) or relative difference (\eqn{\lambda \ne 0}{
  lambda != 0}) of the indicator series values (variable \code{subAnnual})
\item \code{growthRateBenchmarked}: Period to period difference (\eqn{\lambda = 0}) or relative difference (\eqn{\lambda \ne 0}{
  lambda != 0}) of the benchmarked series values (variable \code{benchmarked})
}
}

Notes:
\itemize{
\item The output \code{benchmarks} data frame always contains the original benchmarks provided in the input
benchmarks data frame. Modified nonbinding benchmarks, when applicable, can be recovered (calculated)
from the output \code{series} data frame.
\item The function returns a \code{NULL} object if an error occurs before data processing could start. Otherwise,
if execution gets far enough so that data processing could start, then an incomplete object would be
returned in case of errors (e.g., output \code{series} data frame with \code{NA} values for the benchmarked data).
\item The function returns "data.frame" objects that can be explicitly coerced to other types of objects with
the appropriate \verb{as*()} function (e.g., \code{tibble::as_tibble()} would coerce any of them to a tibble).
}
}
\description{
\if{html,text}{(\emph{version française: 
\url{https://StatCan.github.io/gensol-gseries/fr/reference/benchmarking.html}})}

\emph{Replication of the G-Series 2.0 SAS\eqn{^\circledR}{®} BENCHMARKING procedure (PROC BENCHMARKING).
See the G-Series 2.0 documentation for details (Statistics Canada 2016).}

This function ensures coherence between time series data of the same target variable measured at different
frequencies (e.g., sub-annually and annually). Benchmarking consists of imposing the level of the benchmark
series (e.g., annual data) while minimizing the revisions of the observed movement in the indicator series
(e.g.,  sub-annual data) as much as possible. The function also allows nonbinding benchmarking where the
benchmark series can also be revised.

The function may also be used for benchmarking-related topics such as \emph{temporal distribution}
(the reciprocal action of benchmarking: disaggregation of the benchmark series into more frequent observations),
\emph{calendarization} (a special case of temporal distribution) and \emph{linking} (the connection of different time series
segments into a single consistent time series).

Several series can be benchmarked in a single function call.
}
\details{
When \eqn{\rho < 1}, this function returns the generalized least squared solution of a special case of the general
regression-based benchmarking model proposed by Dagum and Cholette (2006). The model, in matrix form, is:
\deqn{\displaystyle
\begin{bmatrix} s^\dagger \\ a \end{bmatrix} = 
\begin{bmatrix} I \\ J \end{bmatrix} \theta + 
\begin{bmatrix} e \\ \varepsilon \end{bmatrix}
}{[s^dag; a] = [I; J] theta + [e; epsilion]}
where
\itemize{
\item \eqn{a} is the vector of length \eqn{M} of the benchmarks.
\item \eqn{s^\dagger = \left\{
    \begin{array}{cl}
      s + b & \text{if } \lambda = 0 \\
      s \cdot b  & \text{otherwise}
    \end{array} \right.
  }{s^dag = s + b if lambda = 0, s^dag = s * b otherwise} is the vector of length \eqn{T} of the bias corrected
indicator series values, with \eqn{s} denoting the initial (input) indicator series.
\item \eqn{b} is the bias, which is specified with argument \code{bias} when argument \code{bias_option != 3} or, when \code{bias_option = 3},
is estimated as \eqn{\hat{b} = \left\{
    \begin{array}{cl}
      \frac{{1_M}^\mathrm{T} (a - Js)}{{1_M}^\mathrm{T} J 1_T} & \text{if } \lambda = 0 \\
      \frac{{1_M}^\mathrm{T} a}{{1_M}^\mathrm{T} Js} & \text{otherwise}
    \end{array} \right.
  }{b^hat = ({1_M}'(a - J s)) / ({1_M}' J 1_T) if \lambda = 0, b^hat = ({1_M}' a) / ({1_M}' J s) otherwise}, where
\eqn{1_X = (1, ..., 1)^\mathrm{T}}{1_X = (1, ..., 1)'} is a vector of \eqn{1} of length \eqn{X}.
\item \eqn{J} is the \eqn{M \times T}{M x T} matrix of temporal aggregation constraints with elements \if{latex}{\cr}
\eqn{j_{m, t} = \left\{
    \begin{array}{cl}
      1 & \text{if benchmark } m \text{ covers period } t \\
      0 & \text{otherwise}
    \end{array} \right.
  }{j_{m,t} = 1 if benchmark m covers period t, j_{m,t} = 0 otherwise}.
\item \eqn{\theta} is the vector of the final (benchmarked) series values.
\item \eqn{e \sim \left( 0, V_e \right)}{e ~ (0, V_e)} is the vector of the measurement errors of \eqn{s^\dagger}{s^dag}
with covariance matrix \eqn{V_e = C \Omega_e C}.
\item \eqn{C = \mathrm{diag} \left( \sqrt{c_{s^\dagger}} \left| s^\dagger \right|^\lambda \right)}{C = diag(\sqrt{c_{s^dag}} 
|s^dag|^lambda)} where \eqn{c_{s^\dagger}}{c_{s^dag}} is the vector of the alterability coefficients of \eqn{s^\dagger}{s^dag},
assuming \eqn{0^0 = 1}.
\item \eqn{\Omega_e} is a \eqn{T \times T}{T x T} matrix with elements \eqn{\omega_{e_{i,j}} = \rho^{|i-j|}} representing the
autocorrelation of an AR(1) process, again assuming \eqn{0^0 = 1}.
\item \eqn{\varepsilon \sim (0, V_\varepsilon)}{epsilon ~ (0, V_epsilon)} is the vector of the measurement errors of the benchmarks
\eqn{a} with covariance matrix \eqn{V_\varepsilon = \mathrm{diag} \left( c_a a \right)}{V_epsilion = diag(c_a a)} where \eqn{c_a}
is the vector of the alterability coefficients of the benchmarks \eqn{a}.
}

The generalized least squared solution is:
\deqn{\displaystyle 
\hat{\theta} = s^\dagger + V_e J^{\mathrm{T}} \left( J V_e J^{\mathrm{T}} + V_\varepsilon \right)^+ \left( a - J s^\dagger \right)
}{theta^hat = s^dag + V_e J' (J V_e J' + V_epsilion)^{+} (a - J s^dag)}
where \eqn{A^{+}} designates the Moore-Penrose inverse of matrix \eqn{A}.

When \eqn{\rho = 1}, the function returns the solution of the (modified) Denton method:
\deqn{\displaystyle 
\hat{\theta} = s + W \left( a - J s \right)
}{theta^hat = s + W (a - J s)}
where
\itemize{
\item \eqn{W} is the upper-right corner matrix from the following matrix product
\deqn{
    \left[\begin{array}{cc}
      D^{+} \Delta^{\mathrm{T}} \Delta D^{+} & J^{\mathrm{T}} \\
      J & 0
    \end{array} \right]^{+}
    \left[\begin{array}{cc}
      D^{+} \Delta^{\mathrm{T}} \Delta D^{+} & 0 \\
      J & I_M
    \end{array} \right] = 
    \left[\begin{array}{cc}
      I_T & W \\
      0 & W_\nu
    \end{array} \right]
  }{[D^{+} Delta' Delta D^{+}, J'; J, 0]^{+} [D^{+} Delta' Delta D^{+}, 0; J, I_M] = [I_T, W; 0, W_nu]}
\item \eqn{D = \mathrm{diag} \left( \left| s \right|^\lambda \right)}{D = diag(|s|^\lambda)}, assuming \eqn{0^0 = 1}. Note that
\eqn{D} corresponds to \eqn{C} with \eqn{c_{s^\dagger} = 1.0}{c_{s^dag} = 1.0} and without bias correction (arguments
\code{bias_option = 1} and \code{bias = NA}).
\item \eqn{\Delta}{Delta} is a \eqn{T-1 \times T}{T-1 x T} matrix with elements \eqn{\delta_{i,j} = \left\{
    \begin{array}{cl}
      -1 & \text{if } i=j \\
      1 & \text{if } j=i+1 \\
      0 & \text{otherwise}
    \end{array} \right.
  }{delta_{i,j} = 1 if i = j, delta_{i,j} = 1 if j = i + 1, delta_{i,j} = 0 otherwise}.
\item \eqn{W_\nu} is a \eqn{M \times M}{M x M} matrix associated with the Lagrange multipliers of the corresponding
minimization problem, expressed as:
\deqn{\displaystyle 
\begin{aligned}
& \underset{\theta}{\text{minimize}} 
& & \sum_{t \ge 2} \left[ \frac{\left( s_t - \theta_t \right)}{\left| s_t\right|^\lambda}
      - \frac{\left( s_{t-1} - \theta_{t-1} \right)}{\left| s_{t-1}\right|^\lambda} \right]^2 \\
& \text{subject to} 
& & a = J \theta
\end{aligned}
}{min(theta) sum_{t>1}{[(s_t - theta_t) / |s_t|^lambda - (s_{t-1} - theta_{t-1}) / |s_{t-1}|^lambda]^2}, subject to a = J theta}
}

See Quenneville et al. (2006) and Dagum and Cholette (2006) for details.
\subsection{Autoregressive Parameter \eqn{\rho} and \emph{bias}}{

Parameter \eqn{\rho} (argument \code{rho}) is associated to the change between the (input) indicator and the (output)
benchmarked series for two consecutive periods and is often called the \emph{movement preservation parameter}. The larger
the value of \eqn{\rho}, the more the indicator series period to period movements are preserved in the
benchmarked series. With \eqn{\rho = 0}, period to period movement preservation is not enforced and the
resulting benchmarking adjustments are not smooth, as in the case of prorating (\eqn{\rho = 0} and
\eqn{\lambda = 0.5}) where the adjustments take the shape of a \emph{step function}. At the other end of the
spectrum is \eqn{\rho = 1}, referred to as \emph{Denton benchmarking}, where period to period movement preservation
is maximized, which results in the smoothest possible set of benchmarking adjustments available with the function.

The \emph{bias} represents the expected discrepancies between the benchmarks and the indicator series. It can be
used to pre-adjust the indicator series in order to reduce, on average, the discrepancies between the two
sources of data. Bias correction, which is specified with arguments \code{biasOption} and \code{bias}, can be particularly
useful for periods not covered by benchmarks when \eqn{\rho < 1}. In this context, parameter \eqn{\rho} dictates the
speed at which the projected benchmarking adjustments converge to the bias (or converge to \emph{no adjustment} without bias
correction) for periods not covered by a benchmark. The smaller the value of \eqn{\rho}, the faster the convergence to
the bias, with immediate convergence when \eqn{\rho = 0} and no convergence at all (the adjustment of the last period
covered by a benchmark is repeated) when \eqn{\rho = 1} (Denton benchmarking). Arguments \code{biasOption} and \code{bias} are
actually ignored when \eqn{\rho = 1} since correcting for the bias has no impact on Denton benchmarking solutions.
The suggested value for \eqn{\rho} is \eqn{0.9} for monthly indicators and \eqn{0.9^3 = 0.729} for quarterly indicators,
representing a reasonable compromise between maximizing movement preservation and reducing revisions as new benchmarks
become available in the future (benchmarking \emph{timeliness issue}). In practice, note that Denton benchmarking could be
\emph{approximated} with the regression-based model by using a \eqn{\rho} value that is smaller than, but very close to,
\eqn{1.0} (e.g., \eqn{\rho = 0.999}). See Dagum and Cholette (2006) for a complete discussion on this topic.
}

\subsection{Alterability Coefficients}{

Alterability coefficients \eqn{c_{s^\dagger}}{c_{s^dag}} and \eqn{c_a} conceptually represent the measurement errors
associated with the (bias corrected) indicator time series values \eqn{s^\dagger}{s^dag} and benchmarks \eqn{a}
respectively. They are nonnegative real numbers which, in practice, specify the extent to which an initial value can be
modified in relation to other values. Alterability coefficients of \eqn{0.0} define fixed (binding) values while
alterability coefficients greater than \eqn{0.0} define free (nonbinding) values. Increasing the alterability coefficient
of an intial value results in more changes for that value in the benchmarking solution and, conversely, less changes when
decreasing the alterability coefficient. The default alterability coefficients are \eqn{0.0} for the benchmarks (binding
benchmarks) and \eqn{1.0} for the indicator series values (nonbinding indicator series). Important notes:
\itemize{
\item With a value of \eqn{\rho = 1} (argument \code{rho = 1}, associated to Denton Benchmarking), only the default
alterability coefficients (\eqn{0.0} for a benchmark and \eqn{1.0} for a an indicator series value) are valid. The
specification of user-defined alterability coefficients variables is therefore not allowed. If such variables are
specified (see arguments \code{var} and \code{with}), the function ignores them and displays a warning message in the console.
\item Alterability coefficients \eqn{c_{s^\dagger}}{c_{s^dag}} come into play after the indicator series has been corrected for
the bias, when applicable (\eqn{c_{s^\dagger}}{c_{s^dag}} is associated to \eqn{s^\dagger}{s^dag}, not \eqn{s}). This means
that specifying an alterability coefficient of \eqn{0.0} for a given indicator series value \strong{will not} result in an unchanged
value after benchmarking \strong{with bias correction} (see arguments \code{biasOption} and \code{bias}).
}

Nonbinding benchmarks, when applicable, can be recovered (calculated) from the benchmarked series (see output data frame
\code{series} in section \strong{Value}). The output \code{benchmarks} data frame always contains the original benchmarks provided
in the input benchmarks data frame (argument \code{benchmarks_df}).
}

\subsection{Benchmarking Multiple Series}{

Multiple series can be benchmarked in a single \code{\link[=benchmarking]{benchmarking()}} call, by specifying \code{allCols = TRUE}, by
(manually) specifying multiple variables with argument \code{var} (and argument \code{with}) or with BY-group processing
(argument \code{by != NULL}). An important distinction is that all indicator series specified with \code{allCols = TRUE}
or with argument \code{var} (and benchmarks with argument \code{with}) are expected to be of the same length, i.e., same
set of periods and same set (number) of benchmarks. Benchmarking series of different lengths (different sets of
periods) or with different sets (number) of benchmarks must be done with BY-group processing on stacked indicator
series and benchmarks input data frames (see utility functions \code{\link[=stack_tsDF]{stack_tsDF()}} and \code{\link[=stack_bmkDF]{stack_bmkDF()}}). Arguments
\code{by} and \code{var} can be combined in order to implement BY-group processing for multiple series as illustrated by
\emph{Example 2} in the \strong{Examples} section. While multiple variables with argument \code{var} (or \code{allCols = TRUE})
without BY-group processing (argument \code{by = NULL}) is slightly more efficient (faster), a BY-group approach with
a single series variable is usually recommended as it is more general (works in all contexts). The latter is
illustrated by \emph{Example 3} in the \strong{Examples} section. The BY variables specified with argument \code{by} appear in
all three output data frames.
}

\subsection{Arguments \code{constant} and \code{negInput_option}}{

These arguments extend the usage of proportional benchmarking to a larger set of problems. Their default
values correspond to the G-Series 2.0 behaviour (SAS\eqn{^\circledR}{®} PROC BENCHMARKING) for which equivalent
options are not defined. Although proportional benchmarking may not necessarily be the most appropriate approach
(additive benchmarking may be more appropriate) when the values of the indicator series approach 0 (unstable
period-to-period ratios) or "cross the 0 line" and can therefore go from positive to negative and vice versa
(confusing, difficult to interpret period-to-period ratios), these cases are not invalid mathematically
speaking (i.e., the associated proportional benchmarking problem can be solved). It is strongly recommended,
however, to carefully analyze and validate the resulting benchmarked data in these situations and make sure they
correspond to reasonable, interpretable solutions.
}

\subsection{Treatment of Missing (\code{NA}) Values}{
\itemize{
\item If a missing value appears in one of the variables of the benchmarks input data frame (other than the
BY variables), the observations with the missing values are dropped, a warning message is displayed and the
function executes.
\item If a missing value appears in the \code{year} and/or \code{period} variables of the indicator series input data frame
and BY variables are specified, the corresponding BY-group is skipped, a warning message is displayed and
the function moves on to the next BY-group. If no BY variables are specified, a warning message is displayed
and no processing is done.
\item If a missing value appears in one of the indicator series variables in the indicator series input data
frame and BY variables are specified, the corresponding BY-group is skipped, a warning message is displayed
and the function moves on to the next BY-group. If no BY variables are specified, the affected indicator
series is not processed, a warning message is displayed and the function moves on to the next indicator
series (when applicable).
}
}
}
\examples{
# Set the working directory (for the PDF files)
iniwd <- getwd() 
setwd(tempdir())


###########
# Example 1: Simple case with a single quarterly series to benchmark to annual values

# Quarterly indicator series
my_series1 <- ts_to_tsDF(ts(c(1.9, 2.4, 3.1, 2.2, 2.0, 2.6, 3.4, 2.4, 2.3),
                            start = c(2015, 1),
                            frequency = 4))
my_series1

# Annual benchmarks for quarterly data
my_benchmarks1 <- ts_to_bmkDF(ts(c(10.3, 10.2),
                                 start = 2015,
                                 frequency = 1),
                              ind_frequency = 4)
my_benchmarks1

# Benchmarking using...
#   - recommended `rho` value for quarterly series (`rho = 0.729`)
#   - proportional model (`lambda = 1`)
#   - bias-corrected indicator series with the estimated bias (`biasOption = 3`)
out_bench1 <- benchmarking(my_series1,
                           my_benchmarks1,
                           rho = 0.729,
                           lambda = 0,
                           biasOption = 3)

# Generate the benchmarking graphs
\donttest{plot_graphTable(out_bench1$graphTable, "Ex1_graphs.pdf")}


###########
# Example 2: Two quarterly series to benchmark to annual values,
#            with BY-groups and user-defined alterability coefficients

# Sales data (same sales for groups A and B; only alter coefs for van sales differ)
qtr_sales <- ts(matrix(c(# Car sales
                         1851, 2436, 3115, 2205, 1987, 2635, 3435, 2361, 2183, 2822,
                         3664, 2550, 2342, 3001, 3779, 2538, 2363, 3090, 3807, 2631,
                         2601, 3063, 3961, 2774, 2476, 3083, 3864, 2773, 2489, 3082,
                         # Van sales
                         1900, 2200, 3000, 2000, 1900, 2500, 3800, 2500, 2100, 3100,
                         3650, 2950, 3300, 4000, 3290, 2600, 2010, 3600, 3500, 2100,
                         2050, 3500, 4290, 2800, 2770, 3080, 3100, 2800, 3100, 2860),
                       ncol = 2),
                start = c(2011, 1),
                frequency = 4,
                names = c("car_sales", "van_sales"))

ann_sales <- ts(matrix(c(# Car sales
                         10324, 10200, 10582, 11097, 11582, 11092,
                         # Van sales
                         12000, 10400, 11550, 11400, 14500, 16000),
                       ncol = 2),
                start = 2011,
                frequency = 1,
                names = c("car_sales", "van_sales"))

# Quarterly indicator series (with default alter coefs for now)
my_series2 <- rbind(cbind(data.frame(group = rep("A", nrow(qtr_sales)),
                                     alt_van = rep(1, nrow(qtr_sales))),
                          ts_to_tsDF(qtr_sales)),
                    cbind(data.frame(group = rep("B", nrow(qtr_sales)),
                                     alt_van = rep(1, nrow(qtr_sales))),
                          ts_to_tsDF(qtr_sales)))

# Set binding van sales (alter coef = 0) for 2012 Q1 and Q2 in group A (rows 5 and 6)
my_series2$alt_van[c(5,6)] <- 0
head(my_series2, n = 10)
tail(my_series2)


# Annual benchmarks for quarterly data (without alter coefs)
my_benchmarks2 <- rbind(cbind(data.frame(group = rep("A", nrow(ann_sales))),
                              ts_to_bmkDF(ann_sales, ind_frequency = 4)),
                        cbind(data.frame(group = rep("B", nrow(ann_sales))),
                              ts_to_bmkDF(ann_sales, ind_frequency = 4)))
my_benchmarks2

# Benchmarking using...
#   - recommended `rho` value for quarterly series (`rho = 0.729`)
#   - proportional model (`lambda = 1`)
#   - without bias correction (`biasOption = 1` and `bias` not specified)
#   - `quiet = TRUE` to avoid generating the function header
out_bench2 <- benchmarking(my_series2,
                           my_benchmarks2,
                           rho = 0.729,
                           lambda = 1,
                           biasOption = 1,
                           var = c("car_sales", "van_sales / alt_van"),
                           with = c("car_sales", "van_sales"),
                           by = "group",
                           quiet = TRUE)

# Generate the benchmarking graphs
\donttest{plot_graphTable(out_bench2$graphTable, "Ex2_graphs.pdf")}

# Check the value of van sales for 2012 Q1 and Q2 in group A (fixed values)
all.equal(my_series2$van_sales[c(5,6)], out_bench2$series$van_sales[c(5,6)])


###########
# Example 3: same as example 2, but benchmarking all 4 series as BY-groups
#            (4 BY-groups of 1 series instead of 2 BY-groups of 2 series)

qtr_sales2 <- ts.union(A = qtr_sales, B = qtr_sales)
my_series3 <- stack_tsDF(ts_to_tsDF(qtr_sales2))
my_series3$alter <- 1
my_series3$alter[my_series3$series == "A.van_sales"
                & my_series3$year == 2012 & my_series3$period <= 2] <- 0
head(my_series3)
tail(my_series3)


ann_sales2 <- ts.union(A = ann_sales, B = ann_sales)
my_benchmarks3 <- stack_bmkDF(ts_to_bmkDF(ann_sales2, ind_frequency = 4))
head(my_benchmarks3)
tail(my_benchmarks3)

out_bench3 <- benchmarking(my_series3,
                           my_benchmarks3,
                           rho = 0.729,
                           lambda = 1,
                           biasOption = 1,
                           var = "value / alter",
                           with = "value",
                           by = "series",
                           quiet = TRUE)

# Generate the benchmarking graphs
\donttest{plot_graphTable(out_bench3$graphTable, "Ex3_graphs.pdf")}

# Convert data frame `out_bench3$series` to a "mts" object
qtr_sales2_bmked <- tsDF_to_ts(unstack_tsDF(out_bench3$series), frequency = 4)

# Print the first 10 observations
ts(qtr_sales2_bmked[1:10, ], start = start(qtr_sales2), deltat = deltat(qtr_sales2))

# Check the value of van sales for 2012 Q1 and Q2 in group A (fixed values)
all.equal(window(qtr_sales2[, "A.van_sales"], start = c(2012, 1), end = c(2012, 2)),
          window(qtr_sales2_bmked[, "A.van_sales"], start = c(2012, 1), end = c(2012, 2)))


# Reset the working directory to its initial location
setwd(iniwd)
}
\references{
Dagum, E. B. and P. Cholette (2006). \strong{Benchmarking, Temporal Distribution and Reconciliation
Methods of Time Series}. Springer-Verlag, New York, Lecture Notes in Statistics, Vol. 186

Fortier, S. and B. Quenneville (2007). "Theory and Application of Benchmarking in Business Surveys".
\strong{Proceedings of the Third International Conference on Establishment Surveys (ICES-III)}. Montréal, June 2007.

Latendresse, E., M. Djona and S. Fortier (2007). "Benchmarking Sub-Annual Series to Annual Totals –
From Concepts to SAS\eqn{^\circledR}{®} Procedure and Enterprise Guide\eqn{^\circledR}{®} Custom Task". \strong{Proceedings
of the SAS\eqn{^\circledR}{®} Global Forum 2007 Conference}. Cary, NC: SAS Institute Inc.

Quenneville, B., S. Fortier, Z.-G. Chen and E. Latendresse (2006). "Recent Developments in Benchmarking to
Annual Totals in X-12-ARIMA and at Statistics Canada". \strong{Proceedings of the Eurostat Conference on Seasonality,
Seasonal Adjustment and Their Implications for Short-Term Analysis and Forecasting}. Luxembourg, May 2006.

Quenneville, B., P. Cholette, S. Fortier and J. Bérubé (2010). "Benchmarking Sub-Annual Indicator
Series to Annual Control Totals (Forillon v1.04.001)". \strong{Internal document}. Statistics Canada, Ottawa, Canada.

Quenneville, B. and S. Fortier (2012). "Restoring Accounting Constraints in Time Series – Methods and
Software for a Statistical Agency". \strong{Economic Time Series: Modeling and Seasonality}. Chapman & Hall, New York.

Statistics Canada (2012). \strong{Theory and Application of Benchmarking (Course code 0436)}.
Statistics Canada, Ottawa, Canada.

Statistics Canada (2016). "The BENCHMARKING Procedure". \strong{G-Series 2.0 User Guide}.
Statistics Canada, Ottawa, Canada.
}
\seealso{
\code{\link[=stock_benchmarking]{stock_benchmarking()}} \code{\link[=plot_graphTable]{plot_graphTable()}} \link{bench_graphs} \code{\link[=plot_benchAdj]{plot_benchAdj()}} \code{\link[=gs.gInv_MP]{gs.gInv_MP()}} \link{aliases}
}
