% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plot.R
\name{plot,corridor,ANY-method}
\alias{plot,corridor,ANY-method}
\alias{plot,grain,ANY-method}
\alias{plot,mpg,ANY-method}
\title{Plot quick visualizations of \code{grainscape} objects}
\usage{
\S4method{plot}{corridor,ANY}(x, y, quick = NULL, print = TRUE, theme = TRUE, ...)

\S4method{plot}{grain,ANY}(x, y, quick = NULL, print = TRUE, theme = TRUE, ...)

\S4method{plot}{mpg,ANY}(x, y, quick = NULL, print = TRUE, theme = TRUE, ...)
}
\arguments{
\item{x}{A \code{grainscape} object (\code{corridor}, \code{grain}, or \code{mpg}).}

\item{y}{Ignored.}

\item{quick}{If \code{NULL} (the default) it will plot the most useful quick
visualization for the supplied object type. See below for a
description of the available quick plots, and the defaults.}

\item{print}{Render the \code{ggplot} on the default graphics device.
Default is \code{TRUE}.}

\item{theme}{Apply grainscape theme and scale aesthetics. Default is \code{TRUE}.}

\item{...}{Additional arguments (not used).}
}
\value{
\if{html}{\out{<div class="sourceCode">}}\preformatted{   Invisibly, a `ggplot2` object to which additional `ggplot`
           geoms and adjustments can be applied. Has the side effect of
           rendering the plot, unless `print = FALSE`.
}\if{html}{\out{</div>}}
}
\description{
Plot quick visualizations of \code{mpg}, \code{grain}, and \code{corridor} objects.

This function is intended to get a quick look at the state of a \code{grainscape}
object by rendering what are likely to be the most universally useful
visualizations of the spatial data within these objects.

Much more control is available using \code{\link[=ggGS]{ggGS()}} with \code{\link[ggplot2:ggplot]{ggplot2::ggplot()}}
enabling the layering of different different analytical products, and the
visualization of node and link attributes.

For high-resolution visualization and the greatest level of control use
\code{\link[=export]{export()}} to export spatial objects for cartographic representation
in a geographic information system (GIS).
}
\section{Types of visualization available with the \code{quick} parameter}{


\code{"mpgPerimPlot"} gives a a vector rendering of the minimum planar
graph with vector links connecting the perimeters of the patches. This
doesn't accurately represent the sinuosity of paths of the links between patches
but offers a good approximation that renders better at large extents.
Default for \code{mpg} objects. Not available for other objects.

\code{"mpgPlot"} gives a raster-only rendering of the minimum planar graph
where \code{patchId} are positive integers, and \code{linkId} are negative
integers showing the shortest paths between patches  Only available for
\code{mpg} objects.

\code{"network"} gives a vector rendering of the minimum planar graph or
the grains of connectivity network with nodes and links plotted at the
patch or polygon centroid locations. Available for \code{mpg} and \code{grain}
objects. Default for \code{grain} objects.

\code{"grainPlot"} gives a raster and vector rendering of the grains of
connectivity network with nodes and links plotted at polygon centroid locations,
superimposed over the boundaries of the Voronoi polygons.
Can be time consuming on large rasters due to the Voronoi boundary extraction.
Only available for \code{grain} objects.

\code{"corridorPlot"} renders the output of a \code{\link[=corridor]{corridor()}} analysis.
It is the only option available with \code{corridor} objects and the default.
}

\examples{
## Load raster landscape
tiny <- raster::raster(system.file("extdata/tiny.asc", package = "grainscape"))

## Create a resistance surface from a raster using an is-becomes reclassification
tinyCost <- raster::reclassify(tiny, rcl = cbind(c(1, 2, 3, 4), c(1, 5, 10, 12)))
## Produce a patch-based MPG where patches are resistance features=1
tinyPatchMPG <- MPG(cost = tinyCost, patch = tinyCost == 1)
## Extract a representative subset of 5 grains of connectivity
tinyPatchGOC <- GOC(tinyPatchMPG, nThresh = 5)
if (interactive()) {
  library(ggplot2)

  ## MPG and showing simplified links among the perimeters of patches
  plot(tinyPatchMPG)

  ## MPG showing links among the nodes of connected patches
  plot(tinyPatchMPG, quick = "network")

  ## MPG showing the shortest paths between patches actually used to
  ## to calculate link weight values
  plot(tinyPatchMPG, quick = "mpgPlot")

  ## A grain of connectivity network plot with Voronoi boundaries
  plot(grain(tinyPatchGOC, 3), quick = "grainPlot")

  ## Capture plot output for further processing with ggplot
  g <- plot(tinyPatchMPG, print = FALSE, theme = FALSE)
  g <- g + theme_minimal() + ggtitle("Minimum planar graph") +
    theme(plot.title = element_text(size = 20, hjust = 0.5)) +
    theme(legend.position = "none") +
    xlab("Easting") + ylab("Northing")
  g

  ## To change aesthetics it is best to build the plot from scratch
  ## using grainscape::ggGS(). See examples therein.
}
}
\seealso{
\code{\link[=ggGS]{ggGS()}},
\code{\link[=export]{export()}},
\linkS4class{corridor},
\linkS4class{grain},
\linkS4class{mpg}
}
\author{
Alex Chubaty and Paul Galpern
}
