% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/price_indexes.R
\name{price_indexes}
\alias{price_indexes}
\alias{arithmetic_index}
\alias{geometric_index}
\alias{harmonic_index}
\alias{laspeyres_index}
\alias{paasche_index}
\alias{jevons_index}
\alias{lowe_index}
\alias{young_index}
\alias{fisher_index}
\alias{hlp_index}
\alias{lm_index}
\alias{cswd_index}
\alias{cswdb_index}
\alias{bw_index}
\alias{stuvel_index}
\alias{arithmetic_agmean_index}
\alias{geometric_agmean_index}
\alias{lehr_index}
\alias{martini_index}
\title{Price indexes}
\usage{
arithmetic_index(type)

geometric_index(type)

harmonic_index(type)

laspeyres_index(p1, p0, q0, na.rm = FALSE)

paasche_index(p1, p0, q1, na.rm = FALSE)

jevons_index(p1, p0, na.rm = FALSE)

lowe_index(p1, p0, qb, na.rm = FALSE)

young_index(p1, p0, pb, qb, na.rm = FALSE)

fisher_index(p1, p0, q1, q0, na.rm = FALSE)

hlp_index(p1, p0, q1, q0, na.rm = FALSE)

lm_index(elasticity)

cswd_index(p1, p0, na.rm = FALSE)

cswdb_index(p1, p0, q1, q0, na.rm = FALSE)

bw_index(p1, p0, na.rm = FALSE)

stuvel_index(a, b)

arithmetic_agmean_index(elasticity)

geometric_agmean_index(elasticity)

lehr_index(p1, p0, q1, q0, na.rm = FALSE)

martini_index(a)
}
\arguments{
\item{type}{The name of the index. See details for the possible types of
indexes.}

\item{p1}{Current-period prices.}

\item{p0}{Base-period prices.}

\item{q0}{Base-period quantities.}

\item{na.rm}{Should missing values be removed? By default missing values for
prices or quantities return a missing value.}

\item{q1}{Current-period quantities.}

\item{qb}{Period-b quantities for the Lowe/Young index.}

\item{pb}{Period-b prices for the Lowe/Young index.}

\item{elasticity}{The elasticity of substitution for the Lloyd-Moulton and
AG mean indexes.}

\item{a, b}{Parameters for the generalized Stuvel index or Martini index.}
}
\value{
\code{arithmetic_index()}, \code{geometric_index()}, \code{harmonic_index()}, and
\code{stuvel_index()} each return a function to compute the relevant price
indexes; \code{lm_index()}, \code{arithmetic_agmean_index()}, and
\code{geometric_agmean_index()} each return a function to calculate the
relevant index for a given elasticity of substitution. The others return a
numeric value giving the change in price between the base period and current
period.
}
\description{
Calculate a variety of price indexes using information on prices and
quantities at two points in time.
}
\details{
The \code{arithmetic_index()}, \code{geometric_index()}, and
\code{harmonic_index()} functions return a function to calculate a given
type of arithmetic, geometric (logarithmic), and harmonic index. Together,
these functions produce functions to calculate the following indexes.
\itemize{
\item \strong{Arithmetic indexes}
\item Carli
\item Dutot
\item Laspeyres
\item Palgrave
\item Unnamed index (arithmetic mean of Laspeyres and Palgrave)
\item Drobisch (or Sidgwick, arithmetic mean of Laspeyres and Paasche)
\item Walsh-I (arithmetic Walsh)
\item Marshall-Edgeworth
\item Geary-Khamis
\item Lowe
\item Young
\item Hybrid-CSWD
\item \strong{Geometric indexes}
\item Jevons
\item Geometric Laspeyres (or Jöhr)
\item Geometric Paasche
\item Geometric Young
\item Törnqvist (or Törnqvist-Theil)
\item Montgomery-Vartia / Vartia-I
\item Sato-Vartia / Vartia-II
\item Walsh-II (geometric Walsh)
\item Theil
\item Rao
\item \strong{Harmonic indexes}
\item Coggeshall (equally weighted harmonic index)
\item Paasche
\item Harmonic Laspeyres
\item Harmonic Young
}

Along with the \code{lm_index()} function to calculate the Lloyd-Moulton
index, these are just convenient wrappers for
\code{\link[=generalized_mean]{generalized_mean()}} and \code{\link[=index_weights]{index_weights()}}.

The Laspeyres, Paasche, Jevons, Lowe, and Young indexes are among the most
common price indexes, and so they get their own functions. The
\code{laspeyres_index()}, \code{lowe_index()}, and \code{young_index()}
functions correspond to setting the appropriate \code{type} in
\code{arithmetic_index()}; \code{paasche_index()} and \code{jevons_index()}
instead come from the \code{harmonic_index()} and \code{geometric_index()}
functions.

In addition to these indexes, there are also functions for calculating a
variety of indexes based on nested generalized means. The Fisher index is the
geometric mean of the arithmetic Laspeyres and Paasche indexes; the Harmonic
Laspeyres Paasche (or Harmonic Paasche Laspeyres) index is the harmonic
analog of the Fisher index (8054 on Fisher's list). The
Carruthers-Sellwood-Ward-Dalen and Carruthers-Sellwood-Ward-Dalen-Balk
indexes are sample analogs of the Fisher
index; the Balk-Walsh index is the sample analog of the Walsh index. The AG
mean index is the arithmetic or geometric mean of the geometric and
arithmetic Laspeyres indexes, weighted by the elasticity of substitution.
The \code{stuvel_index()} function returns a function to calculate a Stuvel
index of the given parameters. The Lehr index is an alternative to the
Geary-Khamis index, and is the implicit price index for Fisher's index 4153.
The Martini index is a Lowe index where the quantities are the weighted
geometric average of current and base period quantities.
}
\note{
There are different ways to deal with missing values in a price index,
and care should be taken when relying on these functions to remove missing
values. Setting \code{na.rm = TRUE} removes price relatives with missing
information, either because of a missing price or a missing weight, while
using all available non-missing information to make the weights.

Certain properties of an index-number formula may not work as expected when
removing missing values if there is ambiguity about how to remove missing
values from the weights (as in, e.g., a Törnqvist or Sato-Vartia index). The
\code{\link[=balanced]{balanced()}} operator may be helpful, as it balances the removal of missing
values across prices and quantities prior to making the weights.
}
\examples{
p1 <- price6[[2]]
p2 <- price6[[3]]
q1 <- quantity6[[2]]
q2 <- quantity6[[3]]

# Most indexes can be calculated by combining the appropriate weights
# with the correct type of mean.

laspeyres_index(p2, p1, q1)
arithmetic_mean(p2 / p1, index_weights("Laspeyres")(p1, q1))

geometric_index("Laspeyres")(p2, p1, q1)
geometric_mean(p2 / p1, index_weights("Laspeyres")(p1, q1))

# NAs get special treatment.

p_na <- replace(p1, 6, NA)

laspeyres_index(p2, p_na, q1, na.rm = TRUE) # drops the last price relative

sum(p2 * q1, na.rm = TRUE) /
  sum(p_na * q1, na.rm = TRUE) # drops the last period-0 price

# von Bortkiewicz decomposition

paasche_index(p2, p1, q2) / laspeyres_index(p2, p1, q1) - 1

wl <- scale_weights(index_weights("Laspeyres")(p1, q1))
pl <- laspeyres_index(p2, p1, q1)
ql <- quantity_index(laspeyres_index)(q2, q1, p1)

sum(wl * (p2 / p1 / pl - 1) * (q2 / q1 / ql - 1))

# Similar decomposition for geometric Laspeyres/Paasche.

wp <- scale_weights(index_weights("Paasche")(p2, q2))
gl <- geometric_index("Laspeyres")(p2, p1, q1)
gp <- geometric_index("Paasche")(p2, p1, q2)

log(gp / gl)

sum(scale_weights(wl) * (wp / wl - 1) * log(p2 / p1 / gl))

}
\references{
Balk, B. M. (2008). \emph{Price and Quantity Index Numbers}.
Cambridge University Press.

Fisher, I. (1922). \emph{The Making of Index Numbers}. Houghton Mifflin
Company.

IMF, ILO, Eurostat, UNECE, OECD, and World Bank. (2020).
\emph{Consumer Price Index Manual: Concepts and Methods}.
International Monetary Fund.

von der Lippe, P. (2007). \emph{Index Theory and Price Statistics}. Peter Lang.

Selvanathan, E. A. and Rao, D. S. P. (1994).
\emph{Index Numbers: A Stochastic Approach}. MacMillan.
}
\seealso{
\code{\link[=generalized_mean]{generalized_mean()}} for the generalized mean that powers
most of these functions.

\code{\link[=contributions]{contributions()}} for calculating percent-change contributions.

\code{\link[=quantity_index]{quantity_index()}} to remap the arguments in these functions for a
quantity index.

\code{\link[=price6]{price6()}} for an example of how to use these functions with more
than two time periods.

The \pkg{piar} package has more functionality working with price indexes for
multiple groups of products over many time periods.

Other price index functions: 
\code{\link{geks}()},
\code{\link{index_weights}()},
\code{\link{splice_index}()}
}
\concept{price index functions}
