% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bpmc.R
\name{rmc}
\alias{rmc}
\title{Random Number Generation for the McDonald (Mc)/Beta Power Distribution}
\usage{
rmc(n, gamma = 1, delta = 0, lambda = 1)
}
\arguments{
\item{n}{Number of observations. If \code{length(n) > 1}, the length is
taken to be the number required. Must be a non-negative integer.}

\item{gamma}{Shape parameter \code{gamma} > 0. Can be a scalar or a vector.
Default: 1.0.}

\item{delta}{Shape parameter \code{delta} >= 0. Can be a scalar or a vector.
Default: 0.0.}

\item{lambda}{Shape parameter \code{lambda} > 0. Can be a scalar or a vector.
Default: 1.0.}
}
\value{
A vector of length \code{n} containing random deviates from the Mc
distribution, with values in (0, 1). The length of the result is determined
by \code{n} and the recycling rule applied to the parameters (\code{gamma},
\code{delta}, \code{lambda}). Returns \code{NaN} if parameters
are invalid (e.g., \code{gamma <= 0}, \code{delta < 0}, \code{lambda <= 0}).
}
\description{
Generates random deviates from the McDonald (Mc) distribution (also known as
Beta Power) with parameters \code{gamma} (\eqn{\gamma}), \code{delta}
(\eqn{\delta}), and \code{lambda} (\eqn{\lambda}). This distribution is a
special case of the Generalized Kumaraswamy (GKw) distribution where
\eqn{\alpha = 1} and \eqn{\beta = 1}.
}
\details{
The generation method uses the relationship between the GKw distribution and the
Beta distribution. The general procedure for GKw (\code{\link{rgkw}}) is:
If \eqn{W \sim \mathrm{Beta}(\gamma, \delta+1)}, then
\eqn{X = \{1 - [1 - W^{1/\lambda}]^{1/\beta}\}^{1/\alpha}} follows the
GKw(\eqn{\alpha, \beta, \gamma, \delta, \lambda}) distribution.

For the Mc distribution, \eqn{\alpha=1} and \eqn{\beta=1}. Therefore, the
algorithm simplifies significantly:
\enumerate{
\item Generate \eqn{U \sim \mathrm{Beta}(\gamma, \delta+1)} using
\code{\link[stats]{rbeta}}.
\item Compute the Mc variate \eqn{X = U^{1/\lambda}}.
}
This procedure is implemented efficiently, handling parameter recycling as needed.
}
\examples{
\donttest{
set.seed(2028) # for reproducibility

# Generate 1000 random values from a specific Mc distribution
gamma_par <- 2.0
delta_par <- 1.5
lambda_par <- 1.0 # Equivalent to Beta(gamma, delta+1)

x_sample_mc <- rmc(1000,
  gamma = gamma_par, delta = delta_par,
  lambda = lambda_par
)
summary(x_sample_mc)

# Histogram of generated values compared to theoretical density
hist(x_sample_mc,
  breaks = 30, freq = FALSE, # freq=FALSE for density
  main = "Histogram of Mc Sample (Beta Case)", xlab = "x"
)
curve(dmc(x, gamma = gamma_par, delta = delta_par, lambda = lambda_par),
  add = TRUE, col = "red", lwd = 2, n = 201
)
curve(stats::dbeta(x, gamma_par, delta_par + 1), add = TRUE, col = "blue", lty = 2)
legend("topright",
  legend = c("Theoretical Mc PDF", "Theoretical Beta PDF"),
  col = c("red", "blue"), lwd = c(2, 1), lty = c(1, 2), bty = "n"
)

# Comparing empirical and theoretical quantiles (Q-Q plot)
lambda_par_qq <- 0.7 # Use lambda != 1 for non-Beta case
x_sample_mc_qq <- rmc(1000,
  gamma = gamma_par, delta = delta_par,
  lambda = lambda_par_qq
)
prob_points <- seq(0.01, 0.99, by = 0.01)
theo_quantiles <- qmc(prob_points,
  gamma = gamma_par, delta = delta_par,
  lambda = lambda_par_qq
)
emp_quantiles <- quantile(x_sample_mc_qq, prob_points, type = 7)

plot(theo_quantiles, emp_quantiles,
  pch = 16, cex = 0.8,
  main = "Q-Q Plot for Mc Distribution",
  xlab = "Theoretical Quantiles", ylab = "Empirical Quantiles (n=1000)"
)
abline(a = 0, b = 1, col = "blue", lty = 2)

# Compare summary stats with rgkw(..., alpha=1, beta=1, ...)
# Note: individual values will differ due to randomness
x_sample_gkw <- rgkw(1000,
  alpha = 1.0, beta = 1.0, gamma = gamma_par,
  delta = delta_par, lambda = lambda_par_qq
)
print("Summary stats for rmc sample:")
print(summary(x_sample_mc_qq))
print("Summary stats for rgkw(alpha=1, beta=1) sample:")
print(summary(x_sample_gkw)) # Should be similar
}

}
\references{
McDonald, J. B. (1984). Some generalized functions for the size distribution
of income. \emph{Econometrica}, 52(3), 647-663.

Cordeiro, G. M., & de Castro, M. (2011). A new family of generalized
distributions. \emph{Journal of Statistical Computation and Simulation},

Kumaraswamy, P. (1980). A generalized probability density function for
double-bounded random processes. \emph{Journal of Hydrology}, \emph{46}(1-2), 79-88.

Devroye, L. (1986). \emph{Non-Uniform Random Variate Generation}. Springer-Verlag.
(General methods for random variate generation).
}
\seealso{
\code{\link{rgkw}} (parent distribution random generation),
\code{\link{dmc}}, \code{\link{pmc}}, \code{\link{qmc}} (other Mc functions),
\code{\link[stats]{rbeta}}
}
\author{
Lopes, J. E.
}
\keyword{distribution}
\keyword{mcdonald}
\keyword{random}
