% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/beta.R
\name{rbeta_}
\alias{rbeta_}
\title{Random Generation for the Beta Distribution (gamma, delta+1 Parameterization)}
\usage{
rbeta_(n, gamma = 1, delta = 0)
}
\arguments{
\item{n}{Number of observations. If \code{length(n) > 1}, the length is
taken to be the number required. Must be a non-negative integer.}

\item{gamma}{First shape parameter (\code{shape1}), \eqn{\gamma > 0}. Can be a
scalar or a vector. Default: 1.0.}

\item{delta}{Second shape parameter is \code{delta + 1} (\code{shape2}), requires
\eqn{\delta \ge 0} so that \code{shape2 >= 1}. Can be a scalar or a vector.
Default: 0.0 (leading to \code{shape2 = 1}, i.e., Uniform).}
}
\value{
A numeric vector of length \code{n} containing random deviates from the
Beta(\eqn{\gamma, \delta+1}) distribution, with values in (0, 1). The length
of the result is determined by \code{n} and the recycling rule applied to
the parameters (\code{gamma}, \code{delta}). Returns \code{NaN} if parameters
are invalid (e.g., \code{gamma <= 0}, \code{delta < 0}).
}
\description{
Generates random deviates from the standard Beta distribution, using a
parameterization common in generalized distribution families. The distribution
is parameterized by \code{gamma} (\eqn{\gamma}) and \code{delta} (\eqn{\delta}),
corresponding to the standard Beta distribution with shape parameters
\code{shape1 = gamma} and \code{shape2 = delta + 1}. This is a special case
of the Generalized Kumaraswamy (GKw) distribution where \eqn{\alpha = 1},
\eqn{\beta = 1}, and \eqn{\lambda = 1}.
}
\details{
This function generates samples from a Beta distribution with parameters
\code{shape1 = gamma} and \code{shape2 = delta + 1}. It is equivalent to
calling \code{stats::rbeta(n, shape1 = gamma, shape2 = delta + 1)}.

This distribution arises as a special case of the five-parameter
Generalized Kumaraswamy (GKw) distribution (\code{\link{rgkw}}) obtained
by setting \eqn{\alpha = 1}, \eqn{\beta = 1}, and \eqn{\lambda = 1}.
It is therefore also equivalent to the McDonald (Mc)/Beta Power distribution
(\code{\link{rmc}}) with \eqn{\lambda = 1}.

The function likely calls R's underlying \code{rbeta} function but ensures
consistent parameter recycling and handling within the C++ environment,
matching the style of other functions in the related families.
}
\examples{
\donttest{
set.seed(2030) # for reproducibility

# Generate 1000 samples using rbeta_
gamma_par <- 2.0 # Corresponds to shape1
delta_par <- 3.0 # Corresponds to shape2 - 1
shape1 <- gamma_par
shape2 <- delta_par + 1

x_sample <- rbeta_(1000, gamma = gamma_par, delta = delta_par)
summary(x_sample)

# Compare with stats::rbeta
x_sample_stats <- stats::rbeta(1000, shape1 = shape1, shape2 = shape2)
# Visually compare histograms or QQ-plots
hist(x_sample, main = "rbeta_ Sample", freq = FALSE, breaks = 30)
curve(dbeta_(x, gamma_par, delta_par), add = TRUE, col = "red", lwd = 2)
hist(x_sample_stats, main = "stats::rbeta Sample", freq = FALSE, breaks = 30)
curve(stats::dbeta(x, shape1, shape2), add = TRUE, col = "blue", lwd = 2)
# Compare summary stats (should be similar due to randomness)
print(summary(x_sample))
print(summary(x_sample_stats))

# Compare summary stats with rgkw(alpha=1, beta=1, lambda=1)
x_sample_gkw <- rgkw(1000,
  alpha = 1.0, beta = 1.0, gamma = gamma_par,
  delta = delta_par, lambda = 1.0
)
print("Summary stats for rgkw(a=1,b=1,l=1) sample:")
print(summary(x_sample_gkw))

# Compare summary stats with rmc(lambda=1)
x_sample_mc <- rmc(1000, gamma = gamma_par, delta = delta_par, lambda = 1.0)
print("Summary stats for rmc(l=1) sample:")
print(summary(x_sample_mc))
}

}
\references{
Johnson, N. L., Kotz, S., & Balakrishnan, N. (1995). \emph{Continuous Univariate
Distributions, Volume 2} (2nd ed.). Wiley.

Cordeiro, G. M., & de Castro, M. (2011). A new family of generalized
distributions. \emph{Journal of Statistical Computation and Simulation},

Devroye, L. (1986). \emph{Non-Uniform Random Variate Generation}. Springer-Verlag.
}
\seealso{
\code{\link[stats]{rbeta}} (standard R implementation),
\code{\link{rgkw}} (parent distribution random generation),
\code{\link{rmc}} (McDonald/Beta Power random generation),
\code{dbeta_}, \code{pbeta_}, \code{qbeta_} (other functions for this parameterization, if they exist).
}
\author{
Lopes, J. E.
}
\keyword{beta}
\keyword{distribution}
\keyword{random}
