% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gg_kdmap.R
\name{gg_kdmap}
\alias{gg_kdmap}
\title{Kinetic Rate Map (Association/Dissociation plot)}
\usage{
gg_kdmap(
  data,
  id = "id",
  ka = "ka",
  kd = "kd",
  labels = NULL,
  size = 4,
  shape = 21,
  fill = "grey",
  color = "black",
  ref_id = NULL,
  ref_shape = 21,
  ref_color = "black",
  ref_fill = "white",
  ref_size = size,
  rep_lines = TRUE,
  iso_color = "black",
  iso_alpha = 1,
  iso_width = 0.3,
  iso_type = "dashed",
  iso_n = 8,
  text_padding = 1,
  title = NULL,
  show_anno = FALSE
)
}
\arguments{
\item{data}{A data frame containing kinetic data. Must include columns
specified by the \code{id}, \code{ka}, and \code{kd} parameters.}

\item{id}{Character string specifying the column name in \code{data} that
contains identifiers for each measurement. Used to group replicates and
identify reference points. Default is "id".}

\item{ka}{Character string specifying the column name in \code{data} that
contains association rate constants. Values must be in M^-1s^-1 units.
Default is "ka".}

\item{kd}{Character string specifying the column name in \code{data} that
contains dissociation rate constants. Values must be in s^-1 units.
Default is "kd".}

\item{labels}{Character string specifying a column name in \code{data}
containing text labels to display next to points. If NULL (default), no
labels are shown.}

\item{size}{Numeric value for point size, or character string specifying
a column name in \code{data} to map to point size. Default is 3.}

\item{shape}{Numeric value (0-25) for point shape, or character string
specifying a column name in \code{data} to map to point shape. Default
is 21 (filled circle).}

\item{fill}{Character string specifying a color for point fill, or a column
name in \code{data} to map to fill aesthetic. Default is "grey".}

\item{color}{Character string specifying a color for point border, or a
column name in \code{data} to map to color aesthetic. Default is "black".}

\item{ref_id}{Character string specifying the ID of a reference point to
highlight with different aesthetics. Must match a value in the column
specified by \code{id}. Default is NULL (no reference point).}

\item{ref_shape}{Numeric value (0-25) specifying the shape for the reference
point. Default is 21.}

\item{ref_color}{Character string specifying the border color for the
reference point. Default is "black".}

\item{ref_fill}{Character string specifying the fill color for the reference
point. Default is "white".}

\item{ref_size}{Numeric value specifying the size for the reference point.
Default is the value of the \code{size} parameter.}

\item{rep_lines}{Logical indicating whether to connect replicate points
(points with the same ID) with lines. Default is TRUE.}

\item{iso_color}{Character string specifying the color of iso-KD lines.
Default is "black".}

\item{iso_alpha}{Numeric value (0-1) specifying the transparency of iso-KD
lines. Default is 1.}

\item{iso_width}{Numeric value specifying the line width of iso-KD lines.
Default is 0.3.}

\item{iso_type}{Character string specifying the line type of iso-KD lines.
Must be one of "solid", "dashed", "dotted", "dotdash", "longdash", or
"twodash". Default is "dashed".}

\item{iso_n}{Numeric value specifying the number of iso-KD lines to draw,
spaced evenly in log10(KD) across the current plot range. Default is 8.}

\item{text_padding}{Numeric value specifying the padding around text labels
when \code{labels} is used. Passed to \code{ggrepel::geom_text_repel}.
Default is 1.}

\item{title}{Character string for the plot title. If NULL (default), no
title is displayed.}

\item{show_anno}{Logical indicating whether to show corner annotations
indicating fast on-rate (top left) and fast off-rate (bottom right)
regions. Default is FALSE.}
}
\value{
A ggplot2 object showing the kinetic rate map. The plot displays:
\itemize{
\item X-axis: dissociation rate (kd) in s^-1 on log scale
\item Y-axis: association rate (ka) in M^-1s^-1 on log scale
\item Points representing individual kinetic measurements
\item Diagonal iso-affinity lines representing constant KD values
\item Secondary axes (top and right) showing KD values in appropriate units (pM, nM, µM, or mM)
\item Optional connecting lines between replicates (same ID)
\item Optional highlighted reference point
\item Optional text labels for points
\item Optional corner annotations indicating rate directions
}
KD values are automatically calculated from ka and kd rates and displayed
in the most appropriate units based on the data range.
}
\description{
Generates a log-log plot of association rate (ka) vs dissociation rate (kd)
with iso-affinity (KD) lines. Useful for visualizing kinetic binding data
from surface plasmon resonance (SPR), biolayer interferometry (BLI), or
other biophysical assays.
}
\details{
The function creates a log-log plot with:
\itemize{
\item X-axis: dissociation rate (kd, in s^-1)
\item Y-axis: association rate (ka, in M^-1s^-1)
\item Diagonal lines representing iso-affinity contours (constant KD values)
}

\strong{Required units:}
\itemize{
\item ka: M^-1s^-1 (association rate constant)
\item kd: s^-1 (dissociation rate constant)
}

The function calculates equilibrium dissociation constant as KD = kd/ka and
automatically converts to the most appropriate units (pM, nM, µM, or mM) for
display on secondary axes.

Iso-KD lines are generated so that there are always \code{iso_n} lines spanning
the visible plot window. They are equally spaced in log10(KD) across the
KD range implied by the current x/y limits. The top (x) and right (y)
secondary axes are labeled with the KD values corresponding to these lines.
}
\examples{
# Basic example: 5 variants with single measurements
kinetic_data <- data.frame(
  id = c("WT", "Mut1", "Mut2", "Mut3", "Mut4"),
  ka = c(1.2e5, 2.5e5, 2e5, 8.0e4, 1.8e5),
  kd = c(1.5e-3, 2.0e-3, 1.5e-3, 1.2e-3, 1.8e-3)
)

gg_kdmap(data = kinetic_data, show_anno = TRUE)

# With replicates: lines connect points with same ID
kinetic_rep <- data.frame(
  id = c("WT", "WT", "WT", "Mut1", "Mut1", "Mut2", "Mut3", "Mut4"),
  ka = c(1.2e5, 1.5e5, 1.1e5, 2.5e5, 2.4e5, 2e5, 8.0e4, 1.8e5),
  kd = c(1.5e-3, 1.6e-3, 1.4e-3, 2.0e-3, 1.9e-3, 1.5e-3, 1.2e-3, 1.8e-3)
)

gg_kdmap(data = kinetic_rep, show_anno = TRUE, fill = "id")

# Add labels and highlight reference
gg_kdmap(
  data = kinetic_rep,
  labels = "id",
  ref_id = "WT",
  ref_fill = "white",
  ref_color = "red",
  fill = "id"
)

# Customize iso-KD lines
gg_kdmap(
  data = kinetic_rep,
  iso_n = 12,
  iso_color = "#7192ad",
  iso_type = "solid"
)

# Turn off replicate lines
gg_kdmap(data = kinetic_rep, rep_lines = FALSE)

# Custom column names
custom_data <- data.frame(
  sample = c("WT", "Mut1", "Mut2"),
  kon = c(1.2e5, 2.5e5, 5.0e4),
  koff = c(1.5e-3, 2.0e-3, 5.0e-4)
)

gg_kdmap(data = custom_data, id = "sample", ka = "kon", kd = "koff")
}
\seealso{
\code{\link[ggplot2]{geom_point}} for point customization,
\code{\link[ggrepel]{geom_text_repel}} for label placement.
}
