#' Process PhyloSuite RSCU Data for Visualization
#'
#' Processes Relative Synonymous Codon Usage (RSCU) data from PhyloSuite CSV files,
#' preparing it for visualization with the `ggmRSCU` package.
#'
#' @param folder_path Path to folder containing CSV files. Files must contain:
#'   - 'RSCU' column
#'   - Species-reflecting file names
#'   - Required columns: AA, Codon, Fill, RSCU
#'
#' @return Visualization-ready data frame with:
#' \describe{
#'   \item{AA}{Amino acid code}
#'   \item{Codon}{RNA triplet}
#'   \item{Fill}{Position index (1-6)}
#'   \item{Species}{Organism identifier}
#'   \item{RSCU}{Normalized usage values}
#' }
#'
#' @details Performs:
#' \itemize{
#'   \item File validation for required structure
#'   \item Species name extraction
#'   \item Data reshaping to long format
#'   \item NA removal
#' }
#'
#' @examples
#' # Using example data
#' example_dir <- system.file("extdata", "phy", package = "ggmRSCU")
#' result <- read_phy(example_dir)
#' head(result)
#'
#' @export
read_phy <- function(folder_path) {
  requireNamespace("dplyr", quietly = TRUE)
  requireNamespace("tidyr", quietly = TRUE)
  requireNamespace("purrr", quietly = TRUE)

  files <- list.files(
    path = folder_path,
    pattern = "\\.csv$",
    full.names = TRUE
  )

  if (length(files) == 0) {
    stop("No CSV files found in the specified folder.")
  }

  df_list <- purrr::map(files, function(file_path) {
    df <- utils::read.csv(file_path)
    species <- gsub("\\.csv$", "", basename(file_path))

    required_cols <- c("AA", "Codon", "Fill", "RSCU")
    missing_cols <- setdiff(required_cols, colnames(df))
    if (length(missing_cols) > 0) {
      stop("Missing required columns in ", file_path, ": ",
           paste(missing_cols, collapse = ", "))
    }

    df %>%
      dplyr::select(AA, Codon, Fill, RSCU) %>%
      dplyr::mutate(Species = species)
  })

  df_combined <- dplyr::bind_rows(df_list)

  df_combined %>%
    tidyr::drop_na(RSCU) %>%
    dplyr::select(AA, Codon, Fill, Species, RSCU)
}
