% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fuzzy_regression.R
\name{fuzzy_lm}
\alias{fuzzy_lm}
\title{Fuzzy Linear Regression}
\usage{
fuzzy_lm(X_fuzzy, Y_fuzzy, p, X_crisp = NULL)
}
\arguments{
\item{X_fuzzy}{A list of fuzzified predictor values.}

\item{Y_fuzzy}{A list of fuzzified response values.}

\item{p}{An integer specifying the number of predictors.}

\item{X_crisp}{Optional. The original crisp predictor matrix or data frame. Used to retrieve variable names. Default is \code{NULL}.}
}
\value{
A list object of class \code{fuzzy_lm} containing:
\item{Coefficients}{A data frame with estimated coefficients, standard errors, t-values, p-values, and significance stars.}
\item{Residuals}{The residuals from the fitted model.}
\item{Predictions}{The predicted fuzzified response values.}
\item{RSS}{The residual sum of squares.}
\item{R_squared}{The coefficient of determination (R-squared).}
\item{Sigma_squared}{The estimated variance of the residuals.}
\item{Degrees_of_Freedom}{The degrees of freedom for the model.}
}
\description{
Fits a fuzzy linear regression model given fuzzified predictors and response variables.
}
\examples{
# Simulate complex data for fuzzy linear regression
set.seed(123)

# Generate a dataset with 100 observations and 4 predictors
n <- 100
X_crisp <- data.frame(
  Age = round(runif(n, 20, 70)),            # Random ages between 20 and 70
  Income = round(runif(n, 20000, 120000)),  # Random incomes between 20k and 120k
  Education = round(runif(n, 10, 20)),      # Random years of education between 10 and 20
  Experience = round(runif(n, 1, 40))       # Random years of work experience between 1 and 40
)

# Define true coefficients
beta <- c(5.0, 1.2, -0.5, 0.8, 0.05)  # Intercept and coefficients for the predictors

# Generate the crisp response variable with noise
Y_crisp <- round(beta[1] + as.matrix(X_crisp) \%*\% beta[-1] + rnorm(n, mean = 0, sd = 50))

# Fuzzify the predictor and response variables
X_fuzzy <- fuzzify_crisp_matrix(as.matrix(X_crisp), spread = 10)  # Larger spread for predictors
Y_fuzzy <- fuzzify_crisp_vector(Y_crisp, spread = 20)             # Larger spread for responses

# Fit the fuzzy linear model
object <- fuzzy_lm(X_fuzzy, Y_fuzzy, p = 4, X_crisp = X_crisp)

# Print the coefficients
print("Fuzzy Linear Model Coefficients:")
print(object$Coefficients)

# Example residuals and predictions
print("Example Residuals:")
print(head(object$Residuals, 6))

print("Example Predictions:")
print(head(object$Predictions, 6))

}
