.rqrtau.fast <- function(y, x, w = NULL, tau, zeta, m, b0) {
  #' rqrtau.fast
  #'
  #' Algorithm 1: algorithm with preprocessing for Rotated Quantile Regression
  #' (RQR) with initial values of the beta coefficients for a single quantile 
  #' tau
  #'
  #' @param y = Dependent variable (N x 1)
  #' @param x = Regressors matrix (N x K)
  #' @param w = Sample weights (N x 1)
  #' @param zeta =  Conservative estimate of standard error of residuals (N x 1)
  #' @param tau =  Quantile indexes rotated at individual level (N x 1)
  #' @param m =  Parameter to select interval of observations in top and bottom
  #' groups
  #' @param b0 = Initial values of the beta coefficients (K x 1)
  #'
  #' @return b = Estimated beta coefficients (K x 1)

  # Get dimensions
  N <- NROW(x)
  K <- NCOL(x)
  
  # Set parameters
  M <- m * sqrt(N * (K+1))
  maxit <- 100
  
  # Weight observations
  xw <- x * w
  yw <- y * w
  
  # Calculate residuals
  r <- yw - xw %*% b0
  
  taumean <- mean(tau)
  
  for (i1 in seq_len(maxit)) {
    # Classify observations into three groups based on residuals
    ub <- stats::quantile(r / zeta, probs = min(taumean + M / (2 * N), (N - 1) / N))
    lb <- stats::quantile(r / zeta, probs = max(taumean - M / (2 * N), 1 / N))
    
    J <- (r > lb * zeta & r < ub * zeta)
    JL <- (r < lb * zeta)
    JH <- (r > ub * zeta)
    
    # Preliminary regression
    xx <- rbind(
      xw[J, , drop = FALSE],
      colSums((tau[JL] - 1) * xw[JL, , drop = FALSE] / taumean),
      colSums(tau[JH] * xw[JH, , drop = FALSE] / taumean))
    yy <- c(
      yw[J],
      sum((tau[JL] - 1) * yw[JL] / taumean),
      sum(tau[JH] * yw[JH] / taumean))
    b <- quantreg::rq.fit.fnb(
      x = xx,
      y = yy,
      tau = taumean,
      rhs = t(xx) %*% (1 - c(tau[J], taumean, taumean)),
      beta = 0.99995, eps = 1e-06) 
    
    # Calculate new residuals
    rr <- yw - xw %*% b$coefficients
    
    # Check for mispredicted signs
    sum_mispred <- sum(sign(r[JL]) != sign(rr[JL])) + sum(sign(r[JH]) != sign(rr[JH]))
    
    if (sum_mispred == 0) {
      # All signs predicted correctly, solution found
      break
    } else if (sum_mispred >= M / 10) {
      # Many mispredictions, double M and continue
      M <- 2 * M
      r <- rr
    } else {
      # Few mispredictions, refine the groups
      while (sum_mispred > 0) {
        J[JL] <- (sign(r[JL]) != sign(rr[JL]))
        J[JH] <- (sign(r[JH]) != sign(rr[JH]))
        JL <- (JL - J) > 0
        JH <- (JH - J) > 0
        
        # New regression with adjusted groups
        if (sum(JH) == 0) {
          xx <- rbind(
            xw[J, , drop = FALSE],
            colSums((tau[JL] - 1) * xw[JL, , drop = FALSE] / taumean))
          yy <- c(
            yw[J],
            sum((tau[JL] - 1) * yw[JL] / taumean))
          b <- quantreg::rq.fit.fnb(
            x = xx,
            y = yy,
            tau = taumean,
            rhs = t(xx) %*% (1 - c(tau[J], taumean)),
            beta = 0.99995, eps = 1e-06) 
        } else if (sum(JL) == 0) {
          xx <- rbind(
            xw[J, , drop = FALSE],
            colSums(tau[JH] * xw[JH, , drop = FALSE] / taumean))
          yy <- c(
            yw[J],
            sum(tau[JH] * yw[JH] / taumean))
          b <- quantreg::rq.fit.fnb(
            x = xx,
            y = yy,
            tau = taumean,
            rhs = t(xx) %*% (1 - c(tau[J], taumean)),
            beta = 0.99995, eps = 1e-06) 
        } else {
          xx <- rbind(
            xw[J, , drop = FALSE],
            colSums((tau[JL] - 1) * xw[JL, , drop = FALSE] / taumean),
            colSums(tau[JH] * xw[JH, , drop = FALSE] / taumean))
          yy <- c(
            yw[J],
            sum((tau[JL] - 1) * y[JL] / taumean),
            sum(tau[JH] * y[JH] / taumean))
          b <- quantreg::rq.fit.fnb(
            x = xx,
            y = yy,
            tau = taumean,
            rhs = t(xx) %*% (1 - c(tau[J], taumean, taumean)),
            beta = 0.99995, eps = 1e-06)
        }
        
        # Update residuals and mispredictions
        rr <- yw - xw %*% b$coefficients
        sum_mispred <- sum(sign(r[JL]) != sign(rr[JL])) + sum(sign(r[JH]) != sign(rr[JH]))
        r <- rr
      }
      break
    }
  }
  
  return(b$coefficients)
}