.evidence_regression <- function(formula, data, family, coef, prior_sd, level) {
	
	if (!inherits(family, "family"))
		stop("family must be a valid family object.")
	
	fam <- family$family
	if (!fam %in% c("gaussian", "binomial"))
		stop("Supported families are gaussian and binomial.")
	
	fit <- if (fam == "gaussian") {
		stats::lm(formula, data = data)
	} else {
		stats::glm(formula, data = data, family = family)
	}
	
	cf <- stats::coef(summary(fit))
	if (!(coef %in% rownames(cf)))
		stop("Specified coefficient not found.")
	
	est <- unname(cf[coef, 1])
	se  <- unname(cf[coef, 2])
	
  z <- stats::qnorm((1 + level) / 2)
	interval <- c(est - z * se, est + z * se)
	
	tau2 <- if (is.null(prior_sd)) se^2 else prior_sd^2
	
	log_bf <- 0.5 * (
		log(se^2 / (se^2 + tau2)) +
			est^2 * tau2 / (se^2 * (se^2 + tau2))
	)
	
	log10_er <- log_bf / log(10)
	
	.evidence_result(
		estimate = est,
		interval = interval,
		interval_level = level,
		log10_er = log10_er,
		null = paste0(coef, " = 0"),
		alternative = paste0(coef, " != 0"),
		model = paste(fam, "regression"),
		call = match.call()
	)
}
