% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ddbs_quadkey.R
\name{ddbs_quadkey}
\alias{ddbs_quadkey}
\title{Convert geometries to QuadKey tiles}
\usage{
ddbs_quadkey(
  x,
  level = 10,
  output = "polygon",
  field = NULL,
  fun = "mean",
  background = NA,
  conn = NULL,
  name = NULL,
  crs = NULL,
  crs_column = "crs_duckspatial",
  overwrite = FALSE,
  quiet = FALSE
)
}
\arguments{
\item{x}{An \code{sf} spatial object. Alternatively, it can be a string with the
name of a table with geometry column within the DuckDB database \code{conn}.
Data is returned from this object.}

\item{level}{An integer specifying the zoom level for QuadKey generation (1-23).
Higher values provide finer spatial resolution. Default is 10.}

\item{output}{Character string specifying output format. One of:
\itemize{
\item \code{"polygon"} - Returns QuadKey tile boundaries as polygons (default)
\item \code{"raster"} - Returns QuadKey values as a raster grid
\item \code{"tilexy"} - Returns tile XY coordinates
}}

\item{field}{Character string specifying the field name for raster output.
Only used when \code{output = "raster"}}

\item{fun}{summarizing function for when there are multiple geometries in one cell (e.g. "mean",
"min", "max", "sum"). Only used when \code{output = "raster"}}

\item{background}{numeric. Default value in raster cells without values. Only used when
\code{output = "raster"}}

\item{conn}{A connection object to a DuckDB database. If \code{NULL}, the function
runs on a temporary DuckDB database.}

\item{name}{A character string of length one specifying the name of the table,
or a character string of length two specifying the schema and table
names. If \code{NULL} (the default), the function returns the result as an
\code{sf} object}

\item{crs}{The coordinates reference system of the data. Specify if the data
doesn't have a \code{crs_column}, and you know the CRS.}

\item{crs_column}{a character string of length one specifying the column
storing the CRS (created automatically by \code{\link{ddbs_write_vector}}).
Set to \code{NULL} if absent.}

\item{overwrite}{Boolean. whether to overwrite the existing table if it exists. Defaults
to \code{FALSE}. This argument is ignored when \code{name} is \code{NULL}.}

\item{quiet}{A logical value. If \code{TRUE}, suppresses any informational messages.
Defaults to \code{FALSE}.}
}
\value{
An sf object or TRUE (invisibly) for table creation
}
\description{
Converts POINT geometries to QuadKey tile representations at a specified zoom level.
QuadKeys are a hierarchical spatial indexing system used by mapping services like Bing Maps.
}
\details{
QuadKeys divide the world into a hierarchical grid of tiles, where each tile
is subdivided into four smaller tiles at the next zoom level. This function
wraps DuckDB's ST_QuadKey spatial function to generate these tiles from input geometries.
}
\examples{
\dontrun{
## load packages
library(duckspatial)
library(sf)
library(terra)

# create a duckdb database in memory (with spatial extension)
conn <- ddbs_create_conn(dbdir = "memory")

## create random points in Argentina
argentina_sf <- st_read(system.file("spatial/argentina.geojson", package = "duckspatial"))
rand_sf <- st_sample(argentina_sf, 100) |> st_as_sf()
rand_sf["var"] <- runif(100)

## store in duckdb
ddbs_write_vector(conn, rand_sf, "rand_sf")

## generate QuadKey polygons at zoom level 8
qkey_sf <- ddbs_quadkey(conn = conn, "rand_sf", level = 8, output = "polygon")

## generate QuadKey raster with custom field name
qkey_rast <- ddbs_quadkey(conn = conn, "rand_sf", level = 6, output = "raster", field = "var")

## generate Quadkey XY tiles
qkey_tiles_tbl <- ddbs_quadkey(conn = conn, "rand_sf", level = 10, output = "tilexy")
}
}
