% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/glm_nb.r
\name{glm_nb}
\alias{glm_nb}
\title{GLM for NB ratio of means}
\usage{
glm_nb(data, equal_dispersion = FALSE, test = "wald", ci_level = NULL, ...)
}
\arguments{
\item{data}{(list)\cr
A list whose first element is the vector of negative binomial values
from group 1 and the second element is the vector of negative binomial
values from group 2.
\link[base]{NA}s are silently excluded. The default output from
\code{\link[=sim_nb]{sim_nb()}}.}

\item{equal_dispersion}{(Scalar logical: \code{FALSE})\cr
If \code{TRUE}, the model is fit assuming both groups have the same
population dispersion parameter. If \code{FALSE} (default), the model is
fit assuming different dispersions.}

\item{test}{(String: \code{"wald"}; \verb{"c("wald", "lrt")})\cr
The statistical method used for the test results. \code{test = "wald"}
performs a Wald test and optionally the Wald confidence intervals.
\code{test = "lrt"} performs a likelihood ratio test and optionally
the profile likelihood confidence intervals. Wald confidence intervals
are always returned for the limits of the mean of sample 2 and, when
\code{equal_dispersion=FALSE}, for the limits of the dispersion of sample 2.}

\item{ci_level}{(Scalar numeric: \code{NULL}; \verb{(0, 1)})\cr
If \code{NULL}, confidence intervals are set as \code{NA}. If in \verb{(0, 1)},
confidence intervals are calculated at the specified level.
Profile likelihood intervals are computationally intensive, so
intervals from \code{test = "lrt"} may be slow.}

\item{...}{Optional arguments passed to \code{\link[glmmTMB:glmmTMB]{glmmTMB::glmmTMB()}}.}
}
\value{
A list with the following elements:
\tabular{llll}{
Slot \tab Subslot \tab Name \tab Description \cr
1 \tab \tab \code{chisq} \tab \eqn{\chi^2} test statistic for the ratio of means. \cr
2 \tab \tab \code{df}    \tab Degrees of freedom. \cr
3 \tab \tab \code{p}     \tab p-value. \cr

4 \tab   \tab \code{ratio}    \tab Estimated ratio of means (group 2 / group 1). \cr
4 \tab 1 \tab \code{estimate} \tab Point estimate. \cr
4 \tab 2 \tab \code{lower}    \tab Confidence interval lower bound. \cr
4 \tab 3 \tab \code{upper}    \tab Confidence interval upper bound. \cr

5 \tab   \tab \code{mean1}    \tab Estimated mean of group 1. \cr
5 \tab 1 \tab \code{estimate} \tab Point estimate. \cr
5 \tab 2 \tab \code{lower}    \tab Confidence interval lower bound. \cr
5 \tab 3 \tab \code{upper}    \tab Confidence interval upper bound. \cr

6 \tab   \tab \code{mean2}    \tab Estimated mean of group 2. \cr
6 \tab 1 \tab \code{estimate} \tab Point estimate. \cr
6 \tab 2 \tab \code{lower}    \tab Wald confidence interval lower bound. \cr
6 \tab 3 \tab \code{upper}    \tab Wald confidence interval upper bound. \cr

7 \tab   \tab \code{dispersion1} \tab Estimated dispersion of group 1. \cr
7 \tab 1 \tab \code{estimate}    \tab Point estimate. \cr
7 \tab 2 \tab \code{lower}       \tab Confidence interval lower bound. \cr
7 \tab 3 \tab \code{upper}       \tab Confidence interval upper bound. \cr

8 \tab   \tab \code{dispersion2} \tab Estimated dispersion of group 2. \cr
8 \tab 1 \tab \code{estimate}    \tab Point estimate. \cr
8 \tab 2 \tab \code{lower}       \tab Wald confidence interval lower bound. \cr
8 \tab 3 \tab \code{upper}       \tab Wald confidence interval upper bound. \cr

9 \tab \tab \code{n1} \tab Sample size of group 1. \cr
10 \tab \tab \code{n2} \tab Sample size of group 2. \cr

11 \tab \tab \code{method}      \tab Method used for the results. \cr
12 \tab \tab \code{test}        \tab Type of hypothesis test. \cr
13 \tab \tab \code{alternative} \tab The alternative hypothesis. \cr
14 \tab \tab \code{equal_dispersion} \tab Whether or not equal dispersions were assumed. \cr
15 \tab \tab \code{ci_level}    \tab Confidence level of the intervals. \cr
16 \tab \tab \code{hessian}     \tab Information about the Hessian matrix. \cr
17 \tab \tab \code{convergence} \tab Information about convergence.
}
}
\description{
Generalized linear model for two independent negative binomial outcomes.
}
\details{
Uses \code{\link[glmmTMB:glmmTMB]{glmmTMB::glmmTMB()}} in the form

\if{html}{\out{<div class="sourceCode r">}}\preformatted{glmmTMB(
  formula = value ~ condition,
  data = data,
  dispformula = ~ condition,
  family = nbinom2
)
}\if{html}{\out{</div>}}

to model independent negative binomial outcomes
\eqn{X_1 \sim \text{NB}(\mu, \theta_1)} and \eqn{X_2 \sim \text{NB}(r\mu, \theta_2)}
where \eqn{\mu} is the mean of group 1, \eqn{r} is the ratio of the means of
group 2 with respect to group 1, \eqn{\theta_1} is the dispersion parameter
of group 1, and \eqn{\theta_2} is the dispersion parameter of group 2.

The hypotheses for the LRT and Wald test of \eqn{r} are

\deqn{
\begin{aligned}
H_{null} &: log(r) = 0 \\
H_{alt} &: log(r) \neq 0
\end{aligned}
}

where \eqn{r = \frac{\bar{X}_2}{\bar{X}_1}} is the population ratio of
arithmetic means for group 2 with respect to group 1 and
\eqn{log(r_{null}) = 0} assumes the population means are identical.
}
\examples{
#----------------------------------------------------------------------------
# glm_nb() examples
#----------------------------------------------------------------------------
library(depower)

set.seed(1234)
d <- sim_nb(
  n1 = 60,
  n2 = 40,
  mean1 = 10,
  ratio = 1.5,
  dispersion1 = 2,
  dispersion2 = 8
)

lrt <- glm_nb(d, equal_dispersion = FALSE, test = "lrt", ci_level = 0.95)
lrt

wald <- glm_nb(d, equal_dispersion = FALSE, test = "wald", ci_level = 0.95)
wald

#----------------------------------------------------------------------------
# Compare results to manual calculation of chi-square statistic
#----------------------------------------------------------------------------
# Use the same data, but as a data frame instead of list
set.seed(1234)
df <- sim_nb(
  n1 = 60,
  n2 = 40,
  mean1 = 10,
  ratio = 1.5,
  dispersion1 = 2,
  dispersion2 = 8,
  return_type = "data.frame"
)

mod_alt <- glmmTMB::glmmTMB(
  formula = value ~ condition,
  data = df,
  dispformula = ~ condition,
  family = glmmTMB::nbinom2
)
mod_null <- glmmTMB::glmmTMB(
  formula = value ~ 1,
  data = df,
  dispformula = ~ condition,
  family = glmmTMB::nbinom2
)

lrt_chisq <- as.numeric(-2 * (logLik(mod_null) - logLik(mod_alt)))
lrt_chisq
wald_chisq <- summary(mod_alt)$coefficients$cond["condition2", "z value"]^2
wald_chisq

anova(mod_null, mod_alt)

#----------------------------------------------------------------------------
# Compare results to wald_test_nb()
#----------------------------------------------------------------------------
wald2 <- wald_test_nb(d, equal_dispersion = FALSE, ci_level = 0.95)
all.equal(wald$chisq, wald2$chisq, tolerance = 0.01)

}
\references{
\insertRef{hilbe_2011}{depower}

\insertRef{hilbe_2014}{depower}
}
\seealso{
\code{\link[=wald_test_nb]{wald_test_nb()}}, \code{\link[=lrt_nb]{lrt_nb()}}
}
