% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/SemiParametricCopulawithoutCovariates.R
\name{SurvDC}
\alias{SurvDC}
\title{Semiparametric Estimation of the Survival Function under Dependent Censoring}
\usage{
SurvDC(
  yobs,
  delta,
  tm = NULL,
  copfam = "frank",
  margins = list(survfam = NULL, censfam = "lnorm"),
  cure = FALSE,
  Var = list(do = TRUE, nboot = 200, level = 0.05),
  control = list(maxit = 300, eps = 1e-06, trace = TRUE, ktau.inits = NULL)
)
}
\arguments{
\item{yobs}{a numeric vector that indicated the observed survival times.}

\item{delta}{a numeric vector that stores the right-censoring indicators.}

\item{tm}{a numeric vector that contains interested non-negative time points at which the survival probabilities will be evluated.
Note that if we omit the definition of this argument (the default value becomes \code{NULL}), our function will automatically output survival probabilities at all oberserved time points, that is, \code{yobs}.}

\item{copfam}{a character string that specifies the copula family.
Currently, it supports Archimedean copula families, including \code{"frank"} (the default value), \code{"clayton"}, \code{"gumbel"}, and \code{"joe"}.
The degenerated independent censoring case can be considered as well by setting \code{"indep"}.
(other options will be added in the near future!)}

\item{margins}{a list used to define the distribution structures of both the survival and censoring margins.
Specifically, it contains the following elements:
\describe{
  \item{\code{survfam}}{ a character string that defines the assumed distribution for the survival time random variable,
    including \code{"lnorm"} for log-normal distribution, \code{"weibull"} for weibull distribution (other options will be added in the near future).}
  \item{\code{censfam}}{ a character string that defines the assumed distribution for the censoring time random variable, and the details are the same as those shown in \code{survfam}.}
  \item{\code{survtrunc}}{ a positive numeric value thats denotes the value of truncation for the assumed distribution, that is, \code{survfam}.}
  \item{\code{censtrunc}}{ a positive numeric value thats denotes the value of truncation for the assumed distribution, that is, \code{censfam}.}
}
Note if one of the marginal distributions should be modeled nonparametrically, one can let the corresponding argument to be \code{NULL} directly.
For example if a semiparametric framework that defines the survival margin to be nonparametric and the censoring margin to be parametric, say log-normal, is desired,
we can let \code{survfam = NULL} and \code{censfam = "lnorm"}, which is indeed the default value.
Furthermore, if no truncation is imposed in \code{survfam} (or \code{censfam}), one can directly omit the specification of \code{survtrunc} (or \code{censtrunc}), which is the default specification.
We also remark here that when a cured fraction is included (\code{cure = TRUE}), if \code{survfam} is not \code{NULL} and \code{survtrunc = NULL}, we will automatically let \code{survtrunc} to be \code{max(yobs)}.
If we wants to model the data with a non-truncated survival distribution when there is a cured fraction, we can set \code{survtrunc = Inf}.}

\item{cure}{a logical value that indicates whether the existence of a cured fraction should be considered.}

\item{Var}{a list that controls the execution of the bootstrap for variance estimation,
and it contains two elements:
\code{do} is a logical value with default \code{FALSE} to tell the function whether the boostrap-based variances should be calculated;
\code{nboot} is a numeric integer that specifies the number of bootstrap samples.}

\item{control}{indicates more detailed control of the underlying model fitting procedures.
It is a list of the following three arguments:
\describe{
  \item{\code{maxit}}{ a positive integer that denotes the maximum iteration number in optimization. The default value is \code{300}.}
  \item{\code{eps}}{ a positive small numeric value that denotes the tolerance for convergence. The default value is \code{1e-6}.}
  \item{\code{trace}}{ a logical value that judges whereh the tracing information on the progress of the model fitting should be produced. The default value if \code{TRUE}.}
  \item{\code{ktau.inits}}{ a numeric vector that contains initial values of the Kendall's tau.
    The default value is \code{NULL}, meaning that a grids of initial values will be automatically generated within our function.}
}}
}
\value{
A list of fitted results is returned.
  Within this outputted list, the following elements can be found:
  \describe{
    \item{\code{probs}}{survival probabilities of the survial margin at \code{tm}.}
    \item{\code{ktau}}{Kendall's tau.}
    \item{\code{parapar}}{estimation of all parameters (except Kendall's tau) contained in the parametric part.}
    \item{\code{GoF}}{goodness-of-test results.}
    \item{\code{curerate}}{cure rate. If \code{cure = FALSE}, it is \code{NULL}.}
  }
}
\description{
Provide semiparametric approaches that can be used to model right-censored survival data under dependent censoring (without covariates).
  The copula-based approach is adopted and there is no need to explicitly specify the association parameter.
  One of the margins can be modeled nonparametrically. As a byproduct, both marginal distributions of survival and
  censoring times can be considered as fully parametric. The existence of a cured fraction concerning survival time
  can also be taken into consideration.

 @references Czado and Van Keilegom (2023). Dependent censoring based on parametric copulas. Biometrika, 110(3), 721-738.
 @references Delhelle and Van Keilegom (2024). Copula based dependent censoring in cure models. TEST (to appear).
 @references Ding and Van Keilegom (2024). Semiparametric estimation of the survival function under dependent censoring (in preparation).
}
\details{
This unified function provide approaches that can be used to model right-censored survival data under dependent censoring (without covariates).
  Various specifications of marginal distributions can be considered by choosing different combinations of the provided arguments.
  Generally speaking, the following two scenarios are what we mainly focused on:
  \describe{
    \item{\code{nonparametric survival margin and parametric censoring margin (without cure)}}{
      \code{survfam = NULL}, \code{censfam} is not \code{NULL} and \code{cure = FALSE}.}
    \item{\code{nonparametric survival margin and parametric censoring margin (with cure)}}{
      \code{survfam = NULL}, \code{censfam} is not \code{NULL} and \code{cure = TRUE}.}
  }
  As byproducts, several other scenarios (the distribution of the underlying survival time is not nonparametric
  but fully parametric) can also be considered by this R function:
  \describe{
    \item{\code{parametric survival and censoring margins (without cure)}}{
      both \code{survfam} and \code{censfam} are not \code{NULL} and \code{cure = FALSE}.}
    \item{\code{parametric survival and censoring margins (with cure)}}{
      both \code{survfam} and \code{censfam} are not \code{NULL} and \code{cure = TRUE}.}
    \item{\code{parametric survival margin and nonparametric censoring margin (without cure)}}{
      \code{survfam} is not \code{NULL}, \code{censfam = NULL} and \code{cure = FALSE}.}
  }
  Furthermore, one might expect that a scenario with "parametric survival margin and nonparametric censoring margin
  (with cure)" can also be included. Indeed, it can be done based on: \code{survfam} is not \code{NULL}, \code{censfam = NULL}
  and \code{cure = TRUE}. However, from a theoretical perspective of view, whether this type of modeling is reasonable or not
  still needs further investigations.

  We emphasize that the first scenario (in byproducts) has also be considered in another function of this package.
  Specifically, the scenario of "parametric survival margin and nonparametric censoring margin (without cure)" can be
  fitted based on \code{ParamCop()}. However, the default joint modeling of survival and censoring times are based on
  their joint survival function in line with the semiparametric case (instead of modeling joint distribution function
  directly as in  Czado and Van Keilegom (2023) <doi:10.1093/biomet/asac067>), but the idea of estimation methodology
  are exactly the same.
}
\examples{

\donttest{
#----------------------------------------------------------#
# Basic preparations before running subsequent examples ####
#----------------------------------------------------------#

# library necessary packages

#------------------------------------------------------------------------#
# simulated data from Frank copula log-Normal margins (without cure)
#------------------------------------------------------------------------#

# generate the simulated data

# - the sample size of the generated data
n <- 1000

# information on the used copula
copfam.true <- "frank"
ktau.true <- 0.5
coppar.true <- 5.74

# parameters of the underlying log-normal marginal distributions
survpar.true <- c(2.20,1.00)
censpar.true <- c(2.20,0.25)

# - true underlying survival and censoring times
set.seed(1)
u.TC <- copula::rCopula(
  n        = n,
  copula   = copula::archmCopula(
    family = copfam.true,
    param  = coppar.true,
    dim    = 2
  )
)
yobs.T <- qlnorm(1-u.TC[,1],survpar.true[1],survpar.true[2])
yobs.C <- qlnorm(1-u.TC[,2],censpar.true[1],censpar.true[2])

# observations
yobs  <- pmin(yobs.T,yobs.C)
delta <- as.numeric(yobs.T<=yobs.C)
cat("censoring rate is", mean(1-delta))

# model the data under different scenarios

# scenario 1: nonparametric survival margin and parametric censoring margin
set.seed(1)
sol.scenario1 <- SurvDC(
  yobs    = yobs,
  delta   = delta,
  tm      = quantile(yobs, c(0.25,0.50,0.75)),
  copfam  = copfam.true,
  margins = list(survfam = NULL, censfam = "lnorm"),
  Var     = list(do = FALSE, nboot = 50)
)
sol.scenario1$probs
sol.scenario1$ktau
sol.scenario1$parapar

# scenario 2: parametric survival and censoring margins
set.seed(1)
sol.scenario2 <- SurvDC(
  yobs    = yobs,
  delta   = delta,
  tm      = quantile(yobs, c(0.25,0.50,0.75)),
  copfam  = copfam.true,
  margins = list(survfam = "lnorm", censfam = "lnorm"),
  Var     = list(do = FALSE, nboot = 50)
)
sol.scenario2$probs
sol.scenario2$ktau
sol.scenario2$parapar

# scenario 3: parametric survival margin and nonparametric censoring margin
set.seed(1)
sol.scenario3 <- SurvDC(
  yobs    = yobs,
  delta   = delta,
  tm      = quantile(yobs, c(0.25,0.50,0.75)),
  copfam  = copfam.true,
  margins = list(survfam = "lnorm", censfam = NULL),
  Var     = list(do = FALSE, nboot = 50)
)
sol.scenario3$probs
sol.scenario3$ktau
sol.scenario3$parapar

#------------------------------------------------------------------------
# simulated data from Frank copula log-Normal margins (with cure)
#------------------------------------------------------------------------

# generate the simulated data

#  true underlying cure rate
curerate.true <- 0.2

# true underlying survival and censoring times
set.seed(1)
u.TC <- copula::rCopula(
  n        = n,
  copula   = copula::archmCopula(
    family = copfam.true,
    param  = coppar.true,
    dim    = 2
  )
)
yobs.T <- sapply(u.TC[,1],function(uT){
  if(uT<=curerate.true){ val <- Inf }else{
    val <- EnvStats::qlnormTrunc((1-uT)/(1-curerate.true),survpar.true[1],survpar.true[2],0,15)
  }
  return(val)
})
yobs.C <- qlnorm(1-u.TC[,2],censpar.true[1],censpar.true[2])
cat("cure rate is",mean(yobs.T==Inf))

#  observations
yobs  <- pmin(yobs.T,yobs.C)
delta <- as.numeric(yobs.T<=yobs.C)
cat("censoring rate is",mean(1-delta))

# model the data under different scenarios (with cure)

# scenario 4: parametric survival and censoring margins
set.seed(1)
sol.scenario4 <- SurvDC(
  yobs    = yobs,
  delta   = delta,
  tm      = quantile(yobs, c(0.25,0.50,0.75)),
  copfam  = copfam.true,
  margins = list(survfam = "lnorm", censfam = "lnorm"),
  Var     = list(do = FALSE, nboot = 50),
  cure    = TRUE
)
sol.scenario4$probs
sol.scenario4$ktau
sol.scenario4$parapar
sol.scenario4$curerate

# scenario 5: nonparametric survival margin and parametric censoring margin
set.seed(1)
sol.scenario5 <- SurvDC(
  yobs    = yobs,
  delta   = delta,
  tm      = quantile(yobs, c(0.25,0.50,0.75)),
  copfam  = copfam.true,
  margins = list(survfam = NULL, censfam = "lnorm"),
  Var     = list(do = FALSE, nboot = 50),
  cure    = TRUE
)
sol.scenario5$probs
sol.scenario5$ktau
sol.scenario5$parapar
sol.scenario5$curerate
}

}
