#' Format DRI survey
#'
#' `format_dri_survey` helps transform raw survey data into
#' useful objects for further manipulation with `deliberr`
#'
#' @param survey_info survey information needed to format DRI survey
#'
#' @returns A list of survey info, including `name`, `considerations`
#' data, `policies` data, `scale_max` or the upper bound of
#' Likert-scale survey questions, and `q_method` which flags whether the
#' survey uses Q methodology
#'
#' @seealso [deliberr::surveys] for raw survey data formatting
#'
#' @importFrom rlang .data
#' @export
#'
#' @examples
#' dri_survey <- format_dri_survey(surveys[surveys$name == "acp", ])
#'
#' dri_survey$name
#' dri_survey$considerations
#'
format_dri_survey <- function(survey_info = list(
  type = NA_character_,
  order = NA_integer_,
  statement = NA_character_,
  name = NA_character_,
  scale_max = NA_integer_,
  q_method = NA
)) {

  name <- survey_info$name[1]
  scale_max <- survey_info$scale_max[1]
  q_method <- survey_info$q_method[1]

  c_df <- survey_info %>%
    filter(.data$name == !!name, .data$type == "C") %>%
    select(.data$order, .data$statement) %>%
    arrange(.data$order)

  p_df <- survey_info %>%
    filter(.data$name == !!name, .data$type == "P") %>%
    select(.data$order, .data$statement) %>%
    arrange(.data$order)

  list(
    name = name,
    considerations = c_df,
    policies = p_df,
    scale_max = scale_max,
    q_method = q_method
  )

}
