## 
# ------------------------------------------------------------------------
# 
# "jackVarRegen.smallEnsemble(x,func,small,...)" --
# 
# Regenerative jackknife estimator of the variance of function
# 'func' for homogeneous Markov chains.
# 
# ------------------------------------------------------------------------
##
#' @aliases jackVarRegen.smallEnsemble
#' @title Jackknife Variance Estimation for General Harris Markov Chains
#' @description Estimates the jackknife variance of a function applied to
#'  general Harris Markov chains using a regenerative approach and a 
#'  \code{smallEnsemble} object.
#' @param x A vector or matrix representing the data from a general Harris Markov 
#' chain.
#' @param func The function to apply to each sample.
#' @param small An object of class \code{\link{smallEnsemble}}. It can be
#' created optimally using \code{\link{findBestEpsilon}}.
#' @param ... Optional additional arguments for the \code{func} function.
#' @details 	The function uses a regenerative approach to estimate the 
#' jackknife variance for functions applied to general Harris Markov chains. 
#' It relies on a \code{smallEnsemble}  object to define the regenerative structure 
#' of the data. It segments the chain using an estimated Nummelin splitting trick 
#' to create almost independent blocks. The function \code{func}, having output size equal to \emph{p}, 
#'  is applied to the data with each approximate regenerative block removed 
#'  in turn to finally compute an empirical
#' variance of the obtained values.
#' @return Returns a scalar or a covariance matrix, depending on whether the function \code{func} 
#' is univariate or multivariate. For a function returning a vector of length 
#' \emph{p}, the output will be a covariance matrix of size \emph{p x p}.
#' 
#' @references Bertail, P. and Dudek, A. (2025). \emph{Bootstrap for 
#' Dependent Data, with an R package} (by Bernard Desgraupes and Karolina Marek) - submitted.
#' 
#' Quenouille, M.H. (1949). Approximate tests of correlation in time-series. 
#' \emph{J. Roy. Statist. Soc., Ser. B}, \bold{11}, 68-84.
#' 
#' @seealso \code{\link{jackVar}}, 
#' \code{\link{jackFunc}},
#' \code{\link{regenboot}},
#' \code{\link{jackFuncRegen}}, 
#' \code{\link{jackFuncBlock}},
#' \code{\link{jackVarRegen}}.
#' @keywords Jackknife "Variance estimation" "Regenerative processes" "Harris Markov chains"
#' @export
#' @examples
#' \donttest{
#' B=10
#' bb=0*(1:B)
#' cc=0*(1:B)
#' dd=0*(1:B)
#' for (i in 1:B) {
#'   ts=arima.sim(list(ar=0.4),200)
#'    vv=function(ts){as.numeric(var(ts))}
#'    bb[i]=mean(ts)
#'    cc[i]=jackVarRegen.smallEnsemble(ts,mean, small= findBestEpsilon(ts))}
#' var(bb)  
#' mean(cc) 
#' # Monte Carlo simulations
#' mean(dd) 
#' }
jackVarRegen.smallEnsemble <- function(x,func,small,...) {
  if (class(small)[1] != "smallEnsemble") {
    stop("expecting an object of class 'smallEnsemble'.")
  }
  # Ensure func is a function
  #func <- match.fun(func)

  Tn <- func(x,...)
  if (!is.vector(Tn)) {
    stop("Function 'func' must return a vector")
  }
  n <- length(x)
  p <- length(Tn)
  Tnames <- names(Tn)	
  
  s <- small$s
  eps <- small$epsilon
  delta <- small$delta
  p_XiXip1 <- small$trans	
  
  # Find indices of pairs (X_i,X_{i+1}) belonging to the small ensemble
  isInSmall <- (x[1:(n-1)]>=s-eps)*(x[1:(n-1)]<=s+eps)*(x[2:n]>=s-eps)*(x[2:n]<=s+eps)
  
  # Compute the Bernoulli parameter if (X_i,X_{i+1}) \in S
  prob_regen <- delta*isInSmall/p_XiXip1
  
  # Simulate the Bernoulli drawing
  regen <- c((prob_regen>runif(n-1)),0)
  
  # Build the partition in blocks (ignoring first and last blocks)
  blocknums <- cumsum(c(0,regen[1:(n-1)]))
  if (regen[n-1]==1) {
    nb <- max(blocknums)
  } else {
    nb <- max(blocknums)-1
  }
  nbTn <- nb*Tn
  if (nb>0){
    data <- cbind(1:n,blocknums)
    # S: start index of block - L: block length
    S <- numeric(nb)
    L <- numeric(nb)
    for (i in 1:nb) {
      aux <- subset(data,data[,2]==i,1)
      S[i] <- aux[1]
      L[i] <- nrow(aux)
    }
    
    J <- matrix(nrow=nb,ncol=p)
    for (i in 1:nb) {
      start <- S[i]
      len <- L[i]
      # Indices of the block to remove
      bidx <- start:(start+len-1)
      Tni <- func(x[-bidx],...)
      J[i,] <- (nbTn - (nb-1)*Tni)
    }
    muJ <- colMeans(J)
    V <- matrix(0,nrow=p,ncol=p)
    for (i in 1:nb) {
      L <- J[i,]-muJ
      V <- V + L%*%t(L)
    }
    V <- V/(nb*(nb-1))
    if (!is.null(Tnames)) {
      rownames(V) <- Tnames
      colnames(V) <- Tnames
    }
    if (p == 1) {V <- as.vector(V)}
    return(V)}
  else{
    message("There is no regenerative blocks, the variance is 0")
    return(0)}
}